/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2011 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

#include <windows.h>

struct _nacore_fs_dir
  {
	HANDLE			handle;
	char			first;
	WIN32_FIND_DATAW	fdata;
  };

_NACORE_DEF const char * const nacore_fs_dir_sep = "\\";

_NACORE_DEF nacore_fs_dir
nacore_fs_dir_open(const char *name)
{
	nacore_fs_dir ret;
	char *namew, *tmp;
	int errsv;

	ret = malloc(sizeof(struct _nacore_fs_dir));
	if (ret == NULL)
	  {
		errsv = ENOMEM;
		goto alloc_err;
	  }

	nacore_asprintf(&tmp, "%s\\*", name);
	if (tmp == NULL)
	  {
		errno = ENOMEM;
		free(ret);
		return NULL;
	  }

	namew = nacore_string_utf8_to_utf16le(tmp);
	free(tmp);
	if (namew == NULL)
	  {
		errsv = ENOMEM;
		goto namew_err;
	  }

	ret->handle = FindFirstFileW((LPCWSTR)namew, &ret->fdata);
	free(namew);
	if (ret->handle == INVALID_HANDLE_VALUE)
	  {
		/* FIXME: examine error code properly */
		errsv = (GetLastError() == ERROR_FILE_NOT_FOUND)
			? ENOENT : NACORE_EUNKNOWN;
		goto ff_err;
	  }

	ret->first = 1;

	return ret;

ff_err:
namew_err:
	free(ret);
alloc_err:
	errno = errsv;
	return NULL;
}

_NACORE_DEF nacore_fs_dir_entry
nacore_fs_dir_get_next_entry(nacore_fs_dir dir)
{
	char *ret;

	ret = NULL;
	do
	  {
		if (ret != NULL)
			free(ret);

		if (!dir->first)
		  {
			if (FindNextFileW(dir->handle, &dir->fdata) == 0)
			  {
				if (GetLastError() != ERROR_NO_MORE_FILES)
					errno = NACORE_EUNKNOWN;
				return NULL;
			  }
		  }

		dir->first = 0;

		ret = nacore_string_utf16le_to_utf8(
				(const char *)dir->fdata.cFileName);
		if (ret == NULL)
		  {
			errno = ENOMEM;
			return NULL;
		  }
	  }
	while ((strcmp(ret, ".") == 0) || (strcmp(ret, "..") == 0));

	return (nacore_fs_dir_entry)ret;
}

_NACORE_DEF int
nacore_fs_dir_close(nacore_fs_dir dir)
{
	if (FindClose(dir->handle) == 0)
		return NACORE_EUNKNOWN;

	free(dir);

	return 0;
}

_NACORE_DEF const char *
nacore_fs_dir_entry_get_name(nacore_fs_dir_entry entry)
{
	return (const char *)entry;
}

_NACORE_DEF void
nacore_fs_dir_entry_free(nacore_fs_dir_entry entry)
{
	free(entry);
}
