/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2011 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Message reporting
 
   Callback-based message reporting to the user.

   This API helps in avoiding the use of strerror()-like functions and allowing
   for more meaningful and contextualized information.

   The basic idea is that, on one hand, one or more message boxes are created
   and callbacks are associated to them in order to process (e.g., display) the
   messages, while on the other hand messages are created using the functions
   offered by this API, which are also in charge of delivering them.

   There are two kinds of messages: status messages and text messages. The
   former are used to notify the beginning and the end of a lengthy and/or
   complex operation, as well as its end result. The latter consist of just
   simple text with a severity attribute.

   Messages are implicitly organized into a hierarchy where each of them belongs
   to a certain context, which is either a message box or a status message.

   Message reporting conventions:

   * The caller function may create a status message regarding the operations of
     a called function, not the other way round (e.g., main() creates a status
     message saying "Reading files x", then it calls read_file()).
   * The result of a status message shall be set by the caller function and
     never by the called function.
   * When displaying, the last received text message with the highet severity is
     the main message to be displayed.
   * The text of a message is meant to be plain text, not HTML or similar.
   * Messages referring to a certain line/column should be formatted like
     "line:column: message".
   * The first character in the text of a message should be in upper case if
     alphabetical, except for identifiers, etc.
   * One message, one sentence.
   * Do not end the text of a message with a period.
 */

#ifndef _NASPRO_CORE_MSG_H
#define _NASPRO_CORE_MSG_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Enum: nacore_msg_severity

   Severity of a text message.

   nacore_msg_severity_info	- Information message, nothing to worry about.
   nacore_msg_severity_warn	- Warning message, something weird happened,
   				  could be a problem.
   nacore_msg_severity_err	- Error message, impossible to get past this
				  point with the operation.
 */
typedef enum
  {
	nacore_msg_severity_info,
	nacore_msg_severity_warn,
	nacore_msg_severity_err
  } nacore_msg_severity;

/*
   Enum: nacore_msg_result

   Result of an operation described by a status message.

   nacore_msg_result_ok		- Everything fine.
   nacore_msg_result_warn	- Warning.
   nacore_msg_result_err	- Error.
 */
typedef enum
  {
	nacore_msg_result_ok,
	nacore_msg_result_warn,
	nacore_msg_result_err
  } nacore_msg_result;

/*
   Type: nacore_msg_context

   Message context (message box or status message).
 */
typedef struct _nacore_msg_context * nacore_msg_context;

/*
   Type: nacore_op_with_msg_cb

   Like <nacore_op_cb> but also passes message reporting-related data.

   Paramters:

     value		- Pointer to the value to operate on.
     msg_context	- Message context or NULL.
     msg_opaque		- Extra opaque data to be passed to message reporting
			  callback or NULL.
     opaque		- Extra opaque data pointer or NULL.
 */
typedef void
(*nacore_op_with_msg_cb)(void *value, nacore_msg_context msg_context,
			 void *msg_opaque, void *opaque);

/*
   Type: nacore_msg_status_begin_cb

   Status message begin callback.

   After the execution of the callback, text cannot be assumed to be still
   valid.

   Parameters:

   status_msg	- Message context representing the status message.
   text		- Message text.
   opaque	- Extra opaque data associated to the message or NULL.
 */
typedef void
(*nacore_msg_status_begin_cb)(nacore_msg_context status_msg, const char *text,
			      void *opaque);

/*
   Type: nacore_msg_status_end_cb
   
   Status message end callback.

   Parameters:

     status_msg	- Message context represeting the status message.
     result	- Result of the operation described by status_msg.
     opaque	- Extra opaque data associated to the message or NULL.
 */
typedef void
(*nacore_msg_status_end_cb)(nacore_msg_context status_msg,
			    nacore_msg_result result, void *opaque);

/*
   Type: nacore_msg_text_cb

   Text message callback.

   After the execution of the callback, text cannot be assumed to be still
   valid.

   Parameters:

     context	- Message context the text message belongs to.
     severity	- Severity of the text message.
     text	- Message text.
     opaque	- Extra opaque data associated to the message or NULL.
 */
typedef void
(*nacore_msg_text_cb)(nacore_msg_context context, nacore_msg_severity severity,
		      const char *text, void *opaque);

/*
   Function: nacore_msg_box_new()

   Creates a new message box.

   Parameters:

     opaque	- Extra opaque data to be associated to the message box or NULL.

   Returns:

     Message context representing the message box or NULL if there was not
     enough memory.
 */
_NACORE_DEF nacore_msg_context
nacore_msg_box_new(void *opaque);

/*
   Function: nacore_msg_box_free()

   Destroies a message box.

   It is not thread-safe to concurrently use this function while operating on
   the message box and/or the messages directly or indirectly belonging to it.

   Furthermore it should only be used when all the status messages belonging to
   it have been already destroied.

   Parameters:

     msg_box	- Message context representing the message box.
 */
_NACORE_DEF void
nacore_msg_box_free(nacore_msg_context msg_box);

/*
   Function: nacore_msg_box_set_callbacks()

   Associates callbacks to a message box.

   It is not thread-safe to concurrently use this function while operating on
   the message box and/or the messages directly or indirectly belonging to it.

   The function should be called only when there are no messages in the message
   box (i.e., all status messages belonging to it have been ended).

   Parameters:

     msg_box		- Message context representing the message box.
     status_begin_cb	- Status message begin callback or NULL.
     status_end_cb	- Status message end callback or NULL.
     text_cb		- Text message callback or NULL.
 */
_NACORE_DEF void
nacore_msg_box_set_callbacks(nacore_msg_context msg_box,
			     nacore_msg_status_begin_cb status_begin_cb,
			     nacore_msg_status_end_cb status_end_cb,
			     nacore_msg_text_cb text_cb);

/*
   Function: nacore_msg_status_begin()

   Creates a new status message, possibly notifying it to the status message
   begin callback.

   In order to simplify the usage of the API, context can be NULL, in which case
   the function doesn't do anything but returning NULL.

   Furthermore, this function uses <nacore_vasprintf()> internally to create the
   text string, hence refer to the documentation of that function for details on
   how the text formatting is done.

   The text should read like "Doing something", with the first letter in capital
   case if possible.

   Parameters:

     context	- Message context the new status message will belong to or NULL.
     opaque	- Extra opaque data to be associated to the message or NULL.
     fmt	- printf()-like format string.
     ...	- printf()-like extra arguments.

   Returns:

     NULL if context is NULL or the message box has no status message begin
     callback associated to it, otherwise the message context representing the
     status message or NULL if there was not enough memory.
 */
_NACORE_DEF NACORE_FORMAT_PRINTF(3, 4) nacore_msg_context
nacore_msg_status_begin(nacore_msg_context context, void *opaque,
			const char *fmt, ...);

/*
   Function: nacore_msg_status_end()

   Possibly reports the result of the operation described by a status message to
   the status message end callback and destroies such status message.

   In order to simplify the usage of the API, status_msg can be NULL, in which
   case the function does nothing.

   A status message should be ended only when all its child status messages have
   been ended already.

   Parameters:

     status_msg	- Message context representing the status message or NULL.
     result	- Result of the operation described by status_msg.
 */
_NACORE_DEF void
nacore_msg_status_end(nacore_msg_context status_msg, nacore_msg_result result);

/*
   Function: nacore_msg_text()

   Possibly reports a new text message to the text message callback.

   In order to simplify the usage of the API, context can be NULL, in which case
   the function does nothing.

   Furthermore, this function uses <nacore_vasprintf()> internally to create the
   text string, hence refer to the documentation of that function for details on
   how the text formatting is done.

   The text message should read like "Bla bla bla", with the first letter in
   capital case if possible.

   Parameters:

     context	- Message context the new text message will belong to or NULL.
     severity	- Severity of the text message.
     opaque	- Extra opaque data to be associated to the message or NULL.
     fmt	- printf()-like format string.
     ...	- printf()-like extra arguments.
 */
_NACORE_DEF NACORE_FORMAT_PRINTF(4, 5) void
nacore_msg_text(nacore_msg_context context, nacore_msg_severity severity,
		void *opaque, const char *fmt, ...);

/*
   Function: nacore_msg_context_get_opaque()

   Gets the opaque data associated to a given message context.

   Parameters:

     context	- The message context.

   Returns:

     The opaque data associated to the message context or NULL.
 */
_NACORE_DEF void *
nacore_msg_context_get_opaque(nacore_msg_context context);

/*
   Function: nacore_msg_context_get_parent()

   Gets the parent message context of a given message context.

   Parameters:

     context	- The message context.

   Returns:

     The parent message context if the given message context is a status
     message, NULL if it is a message box.
 */
_NACORE_DEF nacore_msg_context
nacore_msg_context_get_parent(nacore_msg_context context);

NACORE_END_C_DECLS

#endif
