/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sys/types.h>
#include <dirent.h>
#include <pwd.h>
#include <unistd.h>

#include <NASPRO/core/lib.h>

#include "src/path.h"

static char *home;

void
_nacore_path_init()
{
	struct passwd *pwd;

	home = nacore_env_get_var("HOME");
	if (NACORE_STRING_IS_NULL_OR_EMPTY(home))
	  {
		pwd = getpwuid(getuid());
		if (pwd == NULL)
		  {
			home = NULL;
			return;
		  }
		home = pwd->pw_dir;
		if (NACORE_STRING_IS_NULL_OR_EMPTY(home))
		  {
			home = NULL;
			return;
		  }
	  }
}

void
_nacore_path_fini()
{
	if (home != NULL)
		nacore_env_free_var_value(home);
}

/* TODO: UTF-8 support. */

static void
scan_dir(const char *path,
	 void (*callback)(const char *file,const char *basename, void *data),
	 char (*filter)(const char *file), void *data)
{
	DIR *dir;
	struct dirent *entry;
	char *filename, *p;
	size_t len;
	size_t path_len;

	dir = opendir(path);
	if (dir == NULL)
		return;

	path_len = strlen(path);
	filename = NULL;
	len = 0;
	while ((entry = readdir(dir)) != NULL)
	  {
		if (!strcmp(entry->d_name, "."))
			continue;
		if (!strcmp(entry->d_name, ".."))
			continue;

		if ((path_len + strlen(entry->d_name) + 1) > len)
		  {
			len = path_len + strlen(entry->d_name) + 1;
			p = realloc(filename, len + 1);
			if (p == NULL)
				goto end;
			if (filename == NULL)
			  {
				strcpy(p, path);
				strcat(p, "/");
			  }
			filename = p;
		  }

		strcpy(filename + 1 + path_len, entry->d_name);
		  
		if (filter != NULL)
			if (!filter(filename))
				continue;

		callback(filename, entry->d_name, data);
	  }

end:
	if (filename != NULL)
		free(filename);
	closedir(dir);
}

void
nacore_path_for_each(const char *path,
	void (*callback)(const char *file, const char *basename, void *data),
	char (*filter)(const char *file), void *data)
{
	char *str, *cur, *next;

	if (path[0] == '\0')
		return;

	str = malloc(strlen(path) + 1);
	if (str == NULL)
		return;
	strcpy(str, path);

	for (cur = str, next = strchr(str, ':'); cur != NULL;
	     cur = next, next = (cur != NULL) ? strchr(cur, ':') : NULL)
	  {
		if (next != NULL)
		  {
			*next = '\0';
			next++;
		  }

		scan_dir((*cur == '\0') ? "." : cur, callback, filter, data);
	  }

	free(str);
}

#if 0
void
nacore_path_for_each_in_subdir(const char *path, const char *subdir,
	void (*callback)(const char *file, const char *basename, void *data),
	char (*filter)(const char *file), void *data)
{
	const char *cur, *next;
	char *new_path, *p;
	char zlen_prefix;
	size_t subdir_len, len;
	
	if (path[0] == '\0')
		return;

	subdir_len = strlen(subdir);

	/* Count needed bytes to allocate a string for nacore_path_for_each() */
	zlen_prefix = 0;
	len = 0;
	for (cur = path, next = strchr(path, ':'); cur != NULL;
	     cur = next, next = (cur != NULL) ? strchr(cur, ':') : NULL)
	  {
		if (((next == (cur + 1)) && (*cur == ':')) ||
		    ((next == NULL) && (strlen(cur) == 0)))
		  {
			if (!zlen_prefix)
			  {
				len += subdir_len + 3;
				zlen_prefix = 1;
			  }

			if (next != NULL)
				while (*next == ':')
					next++;

			continue;
		  }

		if (next != NULL)
		  {
			len += 1 + next - cur + 1 + subdir_len;
			next++;
		  }
		else
			len += 1 + strlen(cur) + 1 + subdir_len;
	  }
	/* First ':' is not wanted, but 1 more byte is needed for trailing '\0'
	 * in malloc() */

	new_path = malloc(len);
	if (new_path == NULL)
		return;

	/* Build the new path */
	new_path[0] = '\0';
	if (zlen_prefix)
	  {
		strcpy(new_path, "./");
		strcat(new_path, subdir);
	  }

	p = new_path + strlen(new_path);
	for (cur = path, next = strchr(path, ':'); cur != NULL;
	     cur = next, next = (cur != NULL) ? strchr(cur, ':') : NULL)
	  {
		if (((next == (cur + 1)) && (*cur == ':')) ||
		    ((next == NULL) && (strlen(cur) == 0)))
		  {
			if (next != NULL)
				while (*next == ':')
					next++;

			continue;
		  }

		if (new_path[0] != '\0')
		  {
			*p = ':';
			p++;
		  }

		if (next != NULL)
		  {
			strncpy(p, cur, next - cur);
			p += next - cur;
			*p = '/';
			p++;
			strcpy(p, subdir);
			p += subdir_len;
			while (*next == ':')
				next++;
		  }
		else
		  {
			strcpy(p, cur);
			p += strlen(p);
			*p = '/';
			p++;
			strcpy(p, subdir);
		  }
	  }

	nacore_path_for_each(new_path, callback, filter, data);

	free(new_path);
}
#endif

void
nacore_path_home_for_each(const char *dir,
	void (*callback)(const char *file, const char *basename, void *data),
	char (*filter)(const char *file), void *data)
{
	char *path;
	
	if (home == NULL)
		return;

	path = malloc(strlen(home) + strlen(dir) + 3);
	if (path == NULL)
		return;

	strcpy(path, home);
	strcat(path, "/.");
	strcat(path, dir);

	scan_dir(path, callback, filter, data);

	free(path);
}
