/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * LV2 bridging helper library
 *
 * Copyright (C) 2007-2011 NASPRO Bridge it development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Bridges

   Bridging LV2 plugin libraries.
 */

#ifndef _NASPRO_BRIDGE_IT_BRIDGE_H
#define _NASPRO_BRIDGE_IT_BRIDGE_H

#ifndef _NASPRO_BRIDGE_IT_LIB_H
# error Only <NASPRO/brit/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Type: nabrit_bridge

   Bridge.
 */
typedef struct _nabrit_bridge	* nabrit_bridge;

/*
   Function: nabrit_bridge_new()

   Creates a new bridge.

   Parameters:

     binary	- Filename relative to the LV2 bundle (i.e., object to
		  lv2:binary predicate in the dynamic manifest).

   Returns:

     Bridge or NULL if some error occurred, in which case errno is set to EAGAIN
     if the system lacked the necessary resources (other than memory), ENOMEM if
     there was not enough memory, EPERM if the caller does not have the
     priviledge to perform the operation or NACORE_EUNKNOWN if another kind of
     error happened.
 */
_NABRIT_DEF nabrit_bridge
nabrit_bridge_new(const char *binary);

/*
   Function: nabrit_bridge_free()

   Destroies a bridge and the plugin libraries it contains.

   It does not destroy the content of plugin libraries, hence use free_cb for
   that.

   Once this function is called, referring to bridge will cause undefined
   behavior.

   Parameters:

     bridge		- Bridge.
     free_cb		- Callback called for each plugin library about to be
			  destroied (value is of type nabrit_pluglib) or NULL.
     free_opaque	- Extra opaque data to be passed to free_cb or NULL.
     msg_context	- Message context or NULL.
     msg_opaque		- Extra data to be passed to message reporting callback
			  or NULL.
 */
_NABRIT_DEF void
nabrit_bridge_free(nabrit_bridge bridge, nacore_op_with_msg_cb free_cb,
		   void *free_opaque, nacore_msg_context msg_context,
		   void *msg_opaque);

/*
   Function: nabrit_bridge_get_descriptor()

   Gets an LV2 plugin descriptor by index.

   The association of index (0 to number of plugins-1) to plugins is
   determined by an internal ordering. As long as no plugins are added or
   removed between two calls, the same index will correspond to the same plugin.

   Parameters:

     bridge	- Bridge.
     index	- Index.

   Returns:

     LV2 plugin descriptor or NULL if index >= number of plugins.
 */
_NABRIT_DEF const LV2_Descriptor *
nabrit_bridge_get_descriptor(nabrit_bridge bridge, uint32_t index);

/*
   Function: nabrit_bridge_begin_op()

   Starts a section of code in which the bridge is guaranteed not to be modified
   by other threads.

   Each successful call has to be matched by a following call to
   <nabrit_bridge_end_op()>.

   Parameters:

     bridge	- Bridge.

   Returns:

     0 on success or EDEADLK if the current thread is already in such a section.
 */
_NABRIT_DEF int
nabrit_bridge_begin_op(nabrit_bridge bridge);

/*
   Function: nabrit_bridge_end_op()

   Ends a section of code in which the bridge is guaranteed not to be modified
   by other threads.

   Parameters:

     bridge	- Bridge.
 */
_NABRIT_DEF void
nabrit_bridge_end_op(nabrit_bridge bridge);

NACORE_END_C_DECLS

#endif
