#!/usr/bin/perl -w

# HTML.pm - The NagiosGrapher FE helper class.
# Copyright (C) 2004, NETWAYS GmbH, Gerd Mueller, Marius Hein
# $Id: HTML.pm 1506 2006-11-20 10:19:32Z gmueller $

#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

# $Id: HTML.pm 1506 2006-11-20 10:19:32Z gmueller $
package NagiosGrapher::HTML;

no warnings qw/redefine/;

use strict;
use NagiosGrapher;

use File::Basename;
use CGI qw/:standard/;

use vars qw (
  @ISA
  @EXPORT
  @EXPORT_OK
  %EXPORT_TAGS
  $VERSION

  %ACTIONS
  %DEFAULTS
  %SITETITLES
  %REPLACEMARKS

  @FORM_HIDDENFIELDS

  %PAGE_REFRESH
  %RRD_TYPEVALUES

  $GRAPH_SCRIPT
);

our $VERSION = '1.6.1-rc3';
our @ISA = qw (Exporter);
our @EXPORT = qw(%ACTIONS %DEFAULTS %SITETITLES);

# The actions used by the CGI
our %ACTIONS = (
  'graph' => 'graphs an single graph or multigraph',
  'hostgraph' => 'graphing all services on a host',
  'rrdinfo' => 'displays the content of the rrddb',
  'showtemplate' => 'displays the template matching the criterias',
  'calendar' => 'shows the calendar and the AT-STYLE TIME SPECs',
);

our %DEFAULTS = (
  'refresh'	=> 300,
  'height'	=> 250,
  'width'		=> 595,
  'type'		=> 'AVERAGE',
);

our %SITETITLES = (
  'graph' => 'graph analysis for __HOST__, __SERVICE__',
  'hostgraph' => 'all graphs for __HOST__',
  'calendar' => 'calendar',
  'rrdinfo' => 'rrd info for __HOST__, __SERVICE__',
  'showtemplate' => 'templateinfo for __SERVICE__',
);

our %REPLACEMARKS = (
  '__HOST__' => 'host',
  '__SERVICE__' => 'service',
);

our @FORM_HIDDENFIELDS = ('browser', 'service', 'host', 'old_service', 'old_host');

our %RRD_TYPEVALUES = (
  'AVERAGE' => 'Average values',
  'MAX' => 'Max values',
  'MIN' => 'Min values',
);

our %PAGE_REFRESH = (
  ""		=> "None",
  "30"	=> "30 seconds",
  "60"	=> "1 minute",
  "300"	=> "5 minutes",
  "600"	=> "10 minutes",
);


our $GRAPH_SCRIPT = './rrd2-system.cgi';

sub new {
  my $this = shift;
  my $class = ref($this) || $this;

  my $NagiosGrapher = NagiosGrapher::getInstance();
  my $CGI = new CGI;

  my $self = bless {
    '_NagiosGrapher' => $NagiosGrapher,
    '_CGI' => $CGI,
    '_Script' => basename($0),

    '_SERVICE' => $CGI->param('service'),
    '_HOST' => $CGI->param('host'),

    '_FE_BROWSER_AUTH' => undef,

  }, $class;

  $self->SetParamDefaults();

  return $self;
}

sub SetParamDefaults {
  my $self = shift;
  while (my ($key, $value) = each(%NagiosGrapher::HTML::DEFAULTS)) {
    if (!$self->Param($key)) {
      $self->Param($key, $value);
    }
  }

  return 1;
}

sub NG {
  my $self = shift;
  return $self->{_NagiosGrapher} || undef;
}

sub Q {
  my $self = shift;
  return $self->{_CGI} || undef;
}

sub Service {
  my $self = shift;
  my ($service) = @_;

  if ($service) {
    $self->{'_SERVICE'} = $service;
    $self->Param('service', $service);
  }

  return $self->{'_SERVICE'};
}

sub Host {
  my $self = shift;
  my ($host) = @_;

  if ($host) {
    $self->{'_HOST'} = $host;
    $self->Param('host', $host);
  }

  return $self->{'_HOST'};
}

sub Param {
  my $self = shift;
  my ($param, $new) = @_;

  my %params = $self->Q->Vars;

  if ($new) {
    $self->Q->param($param, $new);
  }

  if (exists $params{$param} && length($params{$param}) > 0) {
    return $params{$param};
  }

  return undef;
}

sub getPages {
  my $self = shift;
  my ($service) = @_;

    my (%out);
    my @conf = $self->NG->get_ngrapherdef($service || $self->{'_SERVICE'} || $self->Param('service'));

    foreach (@conf) {
        if (exists $_->{page}) {
            $out{$_->{page}} = $_->{page}
        }
    }

    unless (keys %out > 0) {
      return undef;
    }

    $out{"-ALL"} = '--ALL--';

    return %out;
}

sub getXMLHashWithMultigraph {
  my $self = shift;

  if ($self->NG) {
    my $xmlhash = $self->NG->get_ngrapherxml();

    my @nmgraphs = $self->NG->get_nmgrapherdef();
    if ($self->Host) {
      foreach (@nmgraphs) {
          unless ($xmlhash->{host}->{$_->{host_name}}) {
              $xmlhash->{host}->{$_->{host_name}}->{service}->{$_->{service_name}} = {};
          }
          else {
              unless ($xmlhash->{host}->{$_->{host_name}}->{service}->{$_->{service_name}}) {
                  $xmlhash->{host}->{$_->{host_name}}->{service}->{$_->{service_name}} = {};
              }
          }
      }
    }
    # Add the hostgraph servicename
    my $service = [sort(keys(%{$xmlhash->{host}->{$self->Host}->{service}}))]->[0];

    if ($self->Host && $xmlhash->{host}->{$self->Host}->{service}->{$service}->{rrd_file}) {
      unless ($xmlhash->{host}->{$self->Host}->{service}->{'* All Services'}) {
        $xmlhash->{host}->{$self->Host}->{service}->{'* All Services'} = {};
      }
    }

    return $xmlhash;
  }

  return undef;
}

sub replaceMarks {
  my $self = shift;
  my $text = shift;

  while (my($mark, $replace) = each(%NagiosGrapher::HTML::REPLACEMARKS)) {
    if ($text =~ m/$mark/g && $self->Param($replace)) {
      $text =~ s/$mark/$self->Param($replace)/ge;
    }
  }

  return $text;

}

sub getTitle {
  my $self = shift;
  my $action = $self->getAction();
  my $title = undef;

  if ($self->isAction($action) && exists $NagiosGrapher::HTML::SITETITLES{$action}) {
    $title = $self->replaceMarks(
      $NagiosGrapher::HTML::SITETITLES{$action}
    );
  }

  return $title;
}

sub SelfURL {
  my $self = shift;
  my %args = (
    'Type' => undef,
    @_
  );

  my $url = $self->Q->self_url;

  return $url unless($args{Type});

  if ($args{Type} eq 'old') {
    $url =~ s/;/&/g;
  }
  elsif ($args{Type} eq 'script') {
    $url = $self->{_Script};
  }

  return $url;
}

sub getAction {
  my $self = shift;

  my $action = lc( $self->Param('action') );
  $action = 'graph' unless ($action);

  if ($self->Service eq '* All Services') {
    $action = 'hostgraph';
  }

  return $action;
}

sub isAction {
  my $self = shift;

  my $check = $self->getAction();

  foreach (@_) {
    my $action = lc( $_ );
    if (exists $NagiosGrapher::HTML::ACTIONS{$check}
      && exists $NagiosGrapher::HTML::ACTIONS{$action}
      && $action eq $check ) {

      return 1;
    }
  }

  # default
  return undef;
}

sub checkUseTimefilter {
  my $self = shift;

  if ($self->NG) {
    return $self->NG->get_ngraphercfg_value('fe_use_timefilter');
  }

  return undef;
}

sub checkPageBrowserAuth {
  my $self = shift;

  if ($self->NG && !defined $self->{'_FE_BROWSER_AUTH'}) {
    $self->{'_FE_BROWSER_AUTH'} = 0;

    # check auth all
    my $all = $self->NG->get_ngraphercfg_value('fe_use_browser_all') || undef;

    # check url overwrite
    my $overwrite = $self->Param('browser');
    my $auth = $overwrite if ($overwrite && $self->NG->get_ngraphercfg_value('fe_use_browser_url') eq '1');

    if ($auth && $auth eq '1') { # overwrite
      $self->{'_FE_BROWSER_AUTH'} = 1;
    }
    elsif ($self->NG->AuthCheckType($ENV{'REMOTE_USER'}, 'fe_use_browser_for')) { # user
      $self->{'_FE_BROWSER_AUTH'} = 1;
    }
    elsif ($all && $all eq '1')  { # all
      $self->{'_FE_BROWSER_AUTH'} = 1;
    }
    else { # failed
      $self->{'_FE_BROWSER_AUTH'} = 0;
    }

    # rewrite the params for the browser ...
    if ($self->{'_FE_BROWSER_AUTH'} eq '1') {
      $self->Param('old_host', $self->Param('host'));
      $self->Param('old_service', $self->Param('service'));
    }

  }

  return $self->{'_FE_BROWSER_AUTH'};
}

sub htmlGraphImage {
  my $self = shift;
  my %args = (
    'Title' => undef,
    'Start' => undef,
    'End' => undef,
    'Host' => undef,
    'Service' => undef,
    @_
  );

  my (@args, @format_links);
  my @formats = qw(pdf svg eps);

  push @args, sprintf('host=%s', $args{'Host'});
  push @args, sprintf('service=%s', $args{'Service'});
  push @args, sprintf('width=%d', $self->Param('width'));
  push @args, sprintf('height=%d', $self->Param('height'));

  if ($self->getPages() && $self->Param('page_act')) {
    push @args, sprintf('act_page=%s', $self->Param('page_act'));
  }
  push @args, sprintf('type=%s', $self->Param('type'));

  push @args, sprintf('title=%s', $args{'Title'}) if ($args{'Title'});
  push @args, sprintf('start=%s', $args{'Start'}) if ($args{'Start'});
  push @args, sprintf('end=%s', $args{'End'}) if ($args{'End'});

  my $url = $NagiosGrapher::HTML::GRAPH_SCRIPT. '?'. join(';', @args);

  foreach (@formats) {
    push @format_links, $self->Q->a({-href=>sprintf('%s;imageformat=%s', $url, $_), -target=>'_blank'}, uc($_));
  }

  my $out = undef;
  $out .= '<tr><td width="45">&nbsp;</td><td>';
  $out .= '<table class="rrdheader"><tr>';
  $out .= '<td>[&nbsp;'. $self->Q->a({name=>$args{'Title'}, -href=>$self->SelfURL. '#top'}, 'To top');
  $out .= '&nbsp;]&nbsp;'. $args{'Title'}. ' graph'. '</td>';
  $out .= '<td align="right">View as:&nbsp;'. join(',&nbsp;', @format_links);
  $out .= '</td></tr></table>';
  $out .= $self->Q->a({href => $url, target => '_blank'}, $self->Q->img({src => $url, alt => $args{Title}}));
  $out .= '</td></tr>';
  $out .= q(<tr><td colspan="2" style="height: 1px; font-size: 1px;">&nbsp;</td></tr>);

  print $out;
}

sub htmlDefaultImageSelector {
  my $self = shift;

  print '<tr>',
    '<td>Width:</td><td>',
    $self->Q->textfield(-name=>"width", -default=>$self->Param('width'), -override=>1, -size=>4),
    '</td><td>Height:</td><td>',
    $self->Q->textfield(-name=>"height", -default=>$self->Param('height'), -override=>1, -size=>4),
    '</td><td>Refresh:</td><td>',
    $self->Q->popup_menu (
      -name		=> 'refresh',
      -default	=> $self->Param('refresh'),
      -values		=> [keys(%NagiosGrapher::HTML::PAGE_REFRESH)],
      -labels		=> \%NagiosGrapher::HTML::PAGE_REFRESH,
      -onchange	=> 'this.form.submit();'
    ),
    '</td></tr>';
}

sub htmlRRDTypeCombo {
  my $self = shift;

  print '<tr>',
  '<td>Type:</td><td colspan="5">',
  $self->Q->popup_menu (
    -name		=> 'type',
    -default	=> $self->Param('type'),
    -values		=> [keys(%NagiosGrapher::HTML::RRD_TYPEVALUES)],
    -labels		=> \%NagiosGrapher::HTML::RRD_TYPEVALUES,
    -style		=> 'width: 287px; background-color: #bbccff;',
    -onchange	=> 'this.form.submit();'
  ),
  '</td></tr>';
}

sub htmlTimeFilter {
  my $self = shift;

  if ($self->checkUseTimefilter()) {
    print q(<tr><td colspan="6" style="height: 1px; font-size: 1px;">&nbsp;</td></tr>);

    print '<tr><td>',
    $self->Q->a({
      href => '#',
      onclick => "javascript:window.open('". $self->SelfURL(Type => 'script'). "?action=calendar;field=fld-start', 'calendar', 'width=250,height=215');"
    }, 'Start'), ':</td><td colspan="6">',
    $self->Q->textfield (
      -name => 'start',
      -id => 'fld-start',
      -value => $self->Param('start'),
      -size=>18,
      -style=>"width: 287px;"
    ),
    '</td></tr>';

    print '<tr><td>',
    $self->Q->a({
      href => '#',
      onclick => "javascript:window.open('". $self->SelfURL(Type => 'script'). "?action=calendar;field=fld-stop', 'calendar', 'width=250,height=215');"
    }, 'End'), ':</td><td colspan="6">',
    $self->Q->textfield (
      -name => 'stop',
      -id => 'fld-stop',
      -value => $self->Param('stop'),
      -size=>18,
      -style=>"width: 287px;"
    ),
    '</td></tr>';

    if ($self->isAction('hostgraph')) {

      my %timeframes = (
        '' => '',
        '-5400' => 'Actual',
        '-129600' => 'Daily',
        '-907200' => 'Weekly',
        '-4017600' => 'Monthly',
        '-48211200' => 'Yearly',
      );

      print '<tr><td>Timeframe:</td><td colspan="6">',
      $self->Q->popup_menu (
        -name => 'timeframe',
        -value => $self->Param('timeframe'),
        -style =>"width: 287px;",
        -values => [sort keys(%timeframes)],
        -labels => \%timeframes,
		-onchange	=> 'this.form.submit();'
      ),
      '</td></tr>';

    }

    print q(<tr><td colspan="1">&nbsp;</td><td colspan="5">Only values defined by the <a target="_blank" href="http://oss.oetiker.ch/rrdtool/doc/rrdfetch.en.html">AT-STYLE TIME SPECIFICATION</a></td></tr>);
    print q(<tr><td colspan="6" style="height: 1px; font-size: 1px;">&nbsp;</td></tr>);
  }
}

sub htmlPageSelector {
  my $self = shift;

  my %page_hash = $self->getPages();

  if (keys (%page_hash) > 1) {

    my $page_act = $self->Param('page_act');
    $page_act=[keys(%page_hash)]->[1] unless ($page_act);
    $page_act=undef if ($page_act eq '-ALL');

    $self->Param('page_act', $page_act);

    print '<tr><td>page:</td><td colspan="5">',
      $self->Q->popup_menu(
        -name		=> 'page_act',
        -default	=> $page_act,
        -values		=> [keys(%page_hash)],
        -labels		=> \%page_hash,
        -style		=> 'width: 287px; background-color: #fc8888;',
        -onchange	=> 'this.form.submit();'
      ),
      '</td></tr>';
  }
  else {
    $self->Param('page_act', '');
  }
}

sub htmlHostServiceBrowser {
  my $self = shift;

  if ($self->checkPageBrowserAuth()) {
    my $xmlhash = $self->getXMLHashWithMultigraph();

    print '<tr><td>Host:</td><td colspan="5">',
    $self->Q->popup_menu (
      -name		=> 'host',
      -default	=> $self->Host() || undef,
      -values		=> [sort(keys(%{$xmlhash->{host}}))],
      -style		=> 'width: 287px; background-color: #ffff00;',
      -onchange	=> 'this.form.submit();'
    ), '</td></tr>';

    print '<tr><td>Service:</td><td colspan="5">',
    $self->Q->popup_menu (
      -name		=> 'service',
      -default	=> $self->Service() || undef,
      -values		=> [sort(keys(%{$xmlhash->{host}->{$self->Host}->{service}}))],
      -style		=> 'width: 287px; background-color: #ffff00;',
      -onchange	=> 'this.form.submit();'
    ), '</td></tr>';

  }

}

sub htmlHiddenFields {
  my $self = shift;
  my $out = undef;

  foreach my $name (@NagiosGrapher::HTML::FORM_HIDDENFIELDS) {
    my $default = $name;

    next if ($default =~ m/old_/ && !$self->checkPageBrowserAuth());
    next if (($default eq 'host' || $default eq 'service') && $self->checkPageBrowserAuth());

    $default =~ s/^old_//g;

    if ($self->Param($default)) {
      # reset if the fucking '-default' not works ...
      $self->Param($name, $self->Param($default));

      $out .= $self->Q->hidden (
        -name => $name,
        -default => $self->Param($default)
      );
    }

  }

  print $out;
}

sub htmlHeader {
  my $self = shift;

  $self->Q->default_dtd( '-//W3C//DTD HTML 4.01 Transitional//EN' );

  my $out = undef;
  my $meta = undef;

  if ($self->Param('refresh')) {
    $meta = {
      -http_equiv => 'refresh',
      -content => $self->Param('refresh'). '; URL='. $self->SelfURL(Type => 'old'),
    };
  }

  $out .= $self->Q->header(-type=>'text/html', -expires=>'+1s');
  $out .= $self->Q->start_html (
    -lang	=> 'de-DE',
    -title	=> $self->getTitle(),
    -style	=> { -code => $self->getCSS() },
    -head	=> meta($meta),
  );

  $out .= $self->Q->a({-name=>'#top'}, '');

  print $out;
}

sub getCSS {
  # Default Stylesheet
  my $css =<<ECSS;
      body { font-family: 'Arial'; font-size: 8pt; color: #666666; background-color: #f1f1f1; }

      td { font-size: 8pt; color: #333333; }

      a { font-size: 8pt; text-decoration: none; color: #4d8fd2; font-weight: normal; }

      a:hover { text-decoration: underline; }

      .head2 { color: #333333; font-size: 8pt; font-weight: normal; }

      .head2 code { color: #cc0000; font-size: 8pt; font-weight: normal; }

      .head1 { color: #333333; font-size: 12pt; font-weight: bolder; }

      .table1 {
          margin-left: 10px;
          background-color: #ffffff;
          border-collapse: collapse;
          border-color: #f1f1f1;
      }

      .table1 td {
          border-color: #f1f1f1;
      }

      table.rrdheader {
        width: 100%;
        background-color: #f1f1f1;
        border-left: 2px #c0c0c0 solid;
        border-right: 2px #c0c0c0 solid;
        border-top: 2px #c0c0c0 solid;
      }

      table.rrdheader td { padding: 2px; }

      div.line { height: 1px; font-size: 1px; background-color: #000080; margin: 5px 0px 5px 0px; }
ECSS
  return $css;
}

1;
