/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>

#include "MRI.h"
#include "MRW_Loader.h"

char	*progname;

void
Usage()
{
	fprintf (stderr, "Usage: %s mrwfile ... > pnmfile\n", progname);
	fprintf (stderr, "Command line options:\n");
	PrintOptions (stderr);
	exit (1);
}

struct FlatWorldData {
	int	width;
	int	height;
	int	freedata;
	int	doGain;
	double  **gain;
	int	nGain;
	double  **base;
	int	nBase;
	unsigned short  **res;
	int	n;
};

void
FlatWorldStart (void *private, int width, int height, int freedata)
{
	struct FlatWorldData *wd = private;
	int i;

	if (wd->width == 0) {
		wd->width = width;
		wd->height = height;
		wd->gain = (double **)malloc(sizeof(double *) * height);
		wd->base = (double **)malloc(sizeof(double *) * height);
		wd->res = (unsigned short **)malloc(sizeof(unsigned short *) * height);
		for (i = 0; i < height; i++) {
			wd->gain[i] = (double *)malloc(sizeof(double) * width);
			memset (wd->gain[i], 0, sizeof(double)*width);
			wd->base[i] = (double *)malloc(sizeof(double) * width);
			memset (wd->base[i], 0, sizeof(double)*width);
			wd->res[i] = (unsigned short *)malloc(sizeof(unsigned short) * width);
		}
	}
	else if (width != wd->width || height != wd->height) {
		fprintf (stderr, "Images are from different cameras! That can't work!\n");
		exit (1);
	}
	wd->freedata = freedata;
	wd->n = 0;
}

void
FlatWorldRow (void *private, void *data)
{
	struct FlatWorldData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short *R, *G, *B;
	int x;

	if (sl->scanLineType != LINETYPE_SHORT) {
		fprintf (stderr, "FlatWorldRow: Unexpected linetype %d\n", sl->scanLineType);
		exit (1);
	}
	R = (unsigned short *)sl->R;
	G = (unsigned short *)sl->G;
	B = (unsigned short *)sl->B;
	for (x = 0; x < wd->width; x++) {
		/* Exactly one is set, we don't care which. */
		if (wd->doGain) {
			int i = R[x] + G[x] + B[x];
			if (i > wd->base[wd->n][x])
				wd->gain[wd->n][x] += i - wd->base[wd->n][x];
		}
		else
			wd->base[wd->n][x] += R[x] + G[x] + B[x];
	}
	wd->n++;
	if (wd->freedata) {
		MRI_FreeScanLine (sl);
	}
}

void
FlatWorldClose (void *private)
{
}

struct link *
GenFlatWorld (struct FlatWorldData *wd)
{
	struct link *ep = malloc (sizeof (*ep));
	if (ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = FlatWorldStart;
	ep->row = FlatWorldRow;
	ep->close = FlatWorldClose;
	ep->private = wd;
	return ep;
}

int GM[25] = {
 1, 0, 1, 0, 1,
 0, 1, 0, 1, 0,
 1, 0, 1, 0, 1,
 0, 1, 0, 1, 0,
 1, 0, 1, 0, 1
};

int RBM[25] = {
 1, 0, 1, 0, 1,
 0, 0, 0, 0, 0,
 1, 0, 1, 0, 1,
 0, 0, 0, 0, 0,
 1, 0, 1, 0, 1
};

void
ComputeG (struct FlatWorldData *fwd)
{
	int	i, j;

	for (i = 0; i < fwd->height; i++)
	for (j = 0; j < fwd->width; j++)
	if ((i & 1) != (j & 1)) {
			int count = 0;
			double avg = 0.0;
			int l, k;
			for (k = -2; k <= 2; k++)
				for (l = -2; l <= 2; l++)
				if (i+k >= 0 && i+k < fwd->height &&
				    j+l >= 0 && j+l < fwd->width && 
				    GM[(k+2)*5+l+2] &&
				    (k != 0 || l != 0)) {
					    count++;
					    avg += fwd->gain[i+k][j+l];
				    }
#if 0
			fprintf (stderr, "%d %d: count = %d avg=%g cntr=%g\n",
				 i, j, count, avg/count, fwd->gain[i][j]);
#endif
			fwd->res[i][j] = 32768.0 * fwd->gain[i][j] * count / avg;
		}
}

void
ComputeR (struct FlatWorldData *fwd)
{
	int	i, j;

	for (i = 0; i < fwd->height; i++)
	for (j = 0; j < fwd->width; j++)
	if (!(i & 1) && !(j & 1)) {
			int count = 0;
			double avg = 0.0;
			int l, k;
			for (k = -2; k <= 2; k++)
				for (l = -2; l <= 2; l++)
				if (i+k >= 0 && i+k < fwd->height &&
				    j+l >= 0 && j+l < fwd->width && 
				    RBM[(k+2)*5+l+2] &&
				    (k != 0 || l != 0)) {
					    count++;
					    avg += fwd->gain[i+k][j+l];
				    }
			fwd->res[i][j] = 32768.0 * fwd->gain[i][j] * count / avg;
		}
}

void
ComputeB (struct FlatWorldData *fwd)
{
	int	i, j;

	for (i = 0; i < fwd->height; i++)
	for (j = 0; j < fwd->width; j++)
	if ((i & 1) && (j & 1)) {
			int count = 0;
			double avg = 0.0;
			int l, k;
			for (k = -2; k <= 2; k++)
				for (l = -2; l <= 2; l++)
				if (i+k >= 0 && i+k < fwd->height &&
				    j+l >= 0 && j+l < fwd->width && 
				    RBM[(k+2)*5+l+2] &&
				    (k != 0 || l != 0)) {
					    count++;
					    avg += fwd->gain[i+k][j+l];
				    }
			fwd->res[i][j] = 32768.0 * fwd->gain[i][j] * count / avg;
		}
}

ComputeBase (struct FlatWorldData *wd)
{
	fprintf (stderr, "%s: %d base images accummulated\n", progname, wd->nBase);
	if (wd->nBase > 0) {
		double scale =  1.0 / wd->nBase;
		double totbase = 0.0;
		int i, j;
		for (i = 0; i < wd->height; i++)
			for (j = 0; j < wd->width; j++) {
				wd->base[i][j] *= scale;
				totbase += wd->base[i][j];
			}
		fprintf (stderr, "Average base: %g\n", totbase/(wd->width * wd->height));
	}
}

int
main (int argc, char *argv[])
{
	MRI *mri;
	struct link *head;
	char	*errmsg;
	int	width, height;
	unsigned short minres, maxres;
	int	i, j;
	double	totbase;
	struct FlatWorldData *wd = malloc (sizeof (struct FlatWorldData));

	if (wd == (struct FlatWorldData *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}

	progname = argv[0];
	memset (wd, 0, sizeof(*wd));

	if (argc < 2) {
		Usage ();
	}

	for (i = 1; i < argc; i++) {
		FILE	*f;
		
		if (strcmp (argv[i], ".") == 0) {
			wd->doGain = 1;
			ComputeBase (wd);
			continue;
		}
		f = fopen (argv[i], "r");

		if (f == NULL) {
			fprintf (stderr, "%s: cannot open %s for reading.\n", progname, argv[i]);
			continue;
		}
		mri = MRW_Loader (f, &errmsg);
		fclose (f);
		if (mri == (MRI *)0) {
			fprintf (stderr, "%s: cannot read image from file %s: %s.\n", progname, argv[i], errmsg);
			continue;
		}
		head = GenFlatWorld (wd);
        	MRI_ProcessImage (head, mri, 0);
		MRI_FlushPipeline (head);
		MRI_Free (mri);
		if (wd->doGain) wd->nGain++; else wd->nBase++;
	}

	fprintf (stderr, "%s: %d gain images accummulated\n", progname, wd->nGain);

	ComputeG (wd);
	ComputeB (wd);
	ComputeR (wd);

#define SIXTEENBITS
	fprintf (stdout, "P6 %d %d %d\n", wd->width, wd->height, 65535);
	minres = maxres = wd->res[0][0];
	totbase = 0.0;
	for (i = 0; i < wd->height; i++)
		for (j = 0; j < wd->width; j++) {
			unsigned val;

			val = wd->res[i][j];
#if 0
			fprintf (stderr, "%d %d: %g %g %d\n", i, j,
				wd->base[i][j],
				wd->gain[i][j],
				val);
#endif

			if (val > maxres) maxres = val;
			if (val < minres) minres = val;
			if (val > 65535) val = 65535;

			fputc ((val >> 8) & 0xFF, stdout);
#ifdef SIXTEENBITS
			putc (val && 0xFF, stdout);
#endif

			val = wd->base[i][j];
			totbase += val;
			fputc ((val >> 8) & 0xFF, stdout);
#ifdef SIXTEENBITS
			putc (val && 0xFF, stdout);
#endif

			val = 0;
			fputc ((val >> 8) & 0xFF, stdout);
#ifdef SIXTEENBITS
			putc (val && 0xFF, stdout);
#endif
		}
	fprintf (stderr, "Average base: %g %\n", totbase/(wd->width * wd->height));
	fprintf (stderr, "Maximum gain: %g %\n", maxres/327.68);
	fprintf (stderr, "Minimum gain: %g %\n", minres/327.68);
	exit (0);
}
