/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>

#include "MRI.h"

extern char *progname;

int
Reversed(int rotate)
{
	return (rotate == 90) || (rotate == 180);
}

int
MRI_RGFirst (const MRI *mri, int angle)
{
	switch (angle) {
	case 0:
	case 90:
		return TRUE;
	case 180:
	case 270:
		return FALSE;
	default: {
		fprintf (stderr, "%s: Limitation: rotation by %d degrees not implemented\n", progname, angle);
		fprintf (stderr, "    Implemented rotations are 0, 90, 180, and 270.\n");
		exit (1);
	    }
	}
}

void
MRI_ProcessImage (struct link *head, MRI *mri, int rotate)
{
	MRI_Region r;

	r.x = r.y = 0;
	r.width = MRI_GetWidth (mri);
	r.height = MRI_GetHeight (mri);
	MRI_ProcessImageRegion (head, mri, rotate, &r);
}

void
MRI_ProcessImageRegion (struct link *head, MRI *mri, int rotate, MRI_Region *r)
{
	/* The processing pipeline is ready. We now scan the rows in the input
	 * and pass them through the pipeline.
	 */
	switch (rotate) {
	case 0: {
		int y;

		(*head->start) (head->private, r->width, r->height, TRUE);
		for (y = 0; y < r->height; y++) {
			struct MRI_ScanLine *sl = MRI_GetRow (mri, y+r->y, r->x, r->width);
			(*head->row) (head->private, sl);
		}
		break;
	    }
	case 90: {
		int	x;

		(*head->start) (head->private, r->height, r->width, TRUE);
		for (x = 0; x < r->width; x++) {
			struct MRI_ScanLine *sl = MRI_GetCol (mri, x+r->x, r->y, r->height);
			(*head->row) (head->private, sl);
		}
		break;
	    }
	case 180: {
		int y;

		(*head->start) (head->private, r->width, r->height, TRUE);
		for (y = r->height-1; y >= 0; y--) {
			struct MRI_ScanLine *sl = MRI_GetRow (mri, y+r->y, r->x, r->width);
			(*head->row) (head->private, sl);
		}
		break;
	    }
	case 270: {
		int	x;

		(*head->start) (head->private, r->height, r->width, TRUE);
		for (x = r->width-1; x >= 0; x--) {
			struct MRI_ScanLine *sl = MRI_GetCol (mri, x+r->x, r->y, r->height);
			(*head->row) (head->private, sl);
		}
		break;
	    }
	default: {
		fprintf (stderr, "%s: rotation by %d degrees not implemented\n", progname, rotate);
	    }
	}
}

struct _processIterator {
     struct link *head;
     MRI *mri;
     int rotate;
     MRI_Region r;
     int posn;
};

MRI_ProcessIterator *
MRI_NewProcessIterator (struct link *head, MRI *mri, int rotate, MRI_Region *r)
{
	MRI_ProcessIterator *rval = (MRI_ProcessIterator *)malloc(sizeof(*rval));

	if (rval == (void *)0) {
		fprintf (stderr, "MRI_NewProcessIterator: out of memory\n");
		_exit(1);
	}
	rval->head = head;
	rval->mri = mri;
	rval->rotate = rotate;
	rval->r = *r;

	/* The processing pipeline is ready. We now scan the rows in the input
	 * and pass them through the pipeline.
	 */
	switch (rotate) {
	case 0: {
		(*head->start) (head->private, r->width, r->height, TRUE);
		rval->posn = 0;
		break;
	    }
	case 90: {
		(*head->start) (head->private, r->height, r->width, TRUE);
		rval->posn = 0;
		break;
	    }
	case 180: {
		(*head->start) (head->private, r->width, r->height, TRUE);
		rval->posn = r->height-1;
		break;
	    }
	case 270: {
		(*head->start) (head->private, r->height, r->width, TRUE);
		rval->posn = r->width-1;
		break;
	    }
	default: {
		fprintf (stderr, "%s: rotation by %d degrees not implemented\n", progname, rotate);
	    }
	}
	return rval;
}

int
MRI_DoProcessIterator (MRI_ProcessIterator *pip)
{
	/* The processing pipeline is ready. We now scan the rows in the input
	 * and pass them through the pipeline.
	 */
	switch (pip->rotate) {
	case 0: {
		struct MRI_ScanLine *sl = MRI_GetRow (pip->mri, pip->posn+pip->r.y, pip->r.x, pip->r.width);
		(*pip->head->row) (pip->head->private, sl);
		pip->posn++;
		return pip->posn < pip->r.height;
	    }
	case 90: {
		struct MRI_ScanLine *sl = MRI_GetCol (pip->mri, pip->posn+pip->r.x, pip->r.y, pip->r.height);
		(*pip->head->row) (pip->head->private, sl);
		pip->posn++;
		return pip->posn < pip->r.width;
	    }
	case 180: {
		struct MRI_ScanLine *sl = MRI_GetRow (pip->mri, pip->posn+pip->r.y, pip->r.x, pip->r.width);
		(*pip->head->row) (pip->head->private, sl);
		pip->posn--;
		return pip->posn >= 0;
	    }
	case 270: {
		struct MRI_ScanLine *sl = MRI_GetCol (pip->mri, pip->posn+pip->r.x, pip->r.y, pip->r.height);
		(*pip->head->row) (pip->head->private, sl);
		pip->posn--;
		return pip->posn >= 0;
	    }
	default: {
		fprintf (stderr, "%s: rotation by %d degrees not implemented\n", progname, pip->rotate);
		_exit(1);
	    }
	}
}

void
MRI_FreeProcessIterator (MRI_ProcessIterator *pip)
{
    MRI_FlushPipeline (pip->head);
    free (pip);
}
