/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <string.h>
#include "MRI.h"
#include "vmedian.h"

struct HistogramData {
	struct link *next;
	int	width;
	int	N;
	int	*counts;
	double	*Lmin, *Lmax;
	double	Rmin, Rmax;
};

static void
HistogramStart (void *private, int width, int height, int freedata)
{
	struct HistogramData *wd = private;
	int i;

	wd->width = width;
	(*wd->next->start) (wd->next->private, width, height, freedata);
	if (wd->counts != NULL) {
	    for (i = 0; i < wd->N; i++)
		wd->counts[i] = 0;
	}
	wd->Rmin = 200;
	wd->Rmax = -200;
}

static void
HistogramRow (void *private, void *data)
{
	struct HistogramData *wd = private;
	struct MRI_ScanLine *fp = data;
	double *R = (double *)(fp->R);
	int x;

	for (x = 0; x < wd->width; x++) {
		int b;

		if (R[x] > wd->Rmax) wd->Rmax = R[x];
		if (R[x] < wd->Rmin) wd->Rmin = R[x];
		if (wd->counts != NULL) {
		    b = R[x] * wd->N / 100.0;
		    if (b < 0) b = 0;
		    if (b >= wd->N) b = wd->N-1;
		    wd->counts[b]++;
		}
	}
	(*wd->next->row) (wd->next->private, fp);
}

static void
HistogramClose (void *private)
{
	struct HistogramData *wd = private;

	if (wd->Lmin != NULL) *wd->Lmin = wd->Rmin;
	if (wd->Lmax != NULL) *wd->Lmax = wd->Rmax;
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
MRI_GenHistogramCounter (int N, int *counts, double *Lmin, double *Lmax, struct link *next)
{
	struct HistogramData *wd = malloc (sizeof (struct HistogramData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct HistogramData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = HistogramStart;
	ep->row = HistogramRow;
	ep->close = HistogramClose;
	ep->private = wd;
	wd->next = next;
	wd->N = N;
	wd->counts = counts;
	wd->Lmin = Lmin;
	wd->Lmax = Lmax;

	return ep;
}
