#include <math.h>
#include "MRI.h"
#include "vmedian.h"
#include <lcms.h>

VEC3 oneWhitePt = {{ 1.0, 1.000000, 1.0 }};
#if 0
MAT3 dimageRawPrim = {{
	{{ 0.581818, 0.333633, 0.048737 }},
	{{ 0.241455, 0.924133, -0.165588 }},
	{{ 0.123322, -0.236435, 0.938004 }}
}};
VEC3 mrwWhitePt = {{ 0.963577, 1.000000, 0.823654 }};

cmsCIEXYZTRIPLE dimagePrimaries = {
	{ 0.581818, 0.333633, 0.048737 },
	{ 0.241455, 0.924133,-0.165588 },
	{ 0.123322,-0.236435, 0.938004 }
};
cmsCIEXYZ dimageWhitePt = { 0.963577, 1.000000, 0.823654 };

MAT3 mrw2xyz = {{
	{{ 0.5816653, 0.3334802, 0.04858435 }},
	{{ 0.2414550, 0.9241330,-0.16558800 }},
	{{ 0.1230129,-0.2367444, 0.93769495 }}
}};

void
buildDimage2XYZMat (LPMAT3 mat)
{
	cmsCIExyY wPt;
	cmsCIExyYTRIPLE primaries;

	cmsXYZ2xyY (&wPt, &dimageWhitePt);
	cmsXYZ2xyY (&primaries.Red, &dimagePrimaries.Red);
	cmsXYZ2xyY (&primaries.Green, &dimagePrimaries.Green);
	cmsXYZ2xyY (&primaries.Blue, &dimagePrimaries.Blue);
	cmsBuildRGB2XYZtransferMatrix (mat, &wPt, &primaries);
	cmsAdaptMatrixToD50 (mat, &wPt);
}

void
dimage2xyz (LPVEC3 in, LPVEC3 out, LPMAT3 mat)
{
	MAT3eval (out, mat, in);
}

#endif

MAT3 xyz2sRGBprim = {{
	{{ 3.240479, -1.537150, -0.498535 }},
	{{ -0.969256, 1.875992, 0.041556 }},
	{{ 0.055648, -0.204043, 1.057311 }}
}};

static inline void
convert (LPVEC3 in, LPVEC3 out, LPMAT3 mat)
{
	out->n[0] = in->n[0] * mat->v[0].n[0] + in->n[1] * mat->v[0].n[1] + in->n[2] * mat->v[0].n[2];
	out->n[1] = in->n[0] * mat->v[1].n[0] + in->n[1] * mat->v[1].n[1] + in->n[2] * mat->v[1].n[2];
	out->n[2] = in->n[0] * mat->v[2].n[0] + in->n[1] * mat->v[2].n[1] + in->n[2] * mat->v[2].n[2];
}

extern BOOL cmsAdaptMatrixToD50 (LPMAT3 r, LPcmsCIExyY source);

void
createXYZtoRGBmat(LPMAT3 mat, const MRI *mri)
{
	cmsCIExyY D65;
	cmsCIExyYTRIPLE Rec709Primaries = {
		{ 0.64, 0.33, 1.0 },
		{ 0.30, 0.60, 1.0 },
		{ 0.15, 0.06, 1.0 }
	};
	cmsCIExyY wPt;
	MAT3 tmp;

	cmsXYZ2xyY (&wPt, MRI_GetWhitePoint(mri));
	cmsWhitePointFromTemp (6504, &D65);
	cmsBuildRGB2XYZtransferMatrix (&tmp, &D65, &Rec709Primaries);
	cmsAdaptMatrixToD50 (mat, &D65);
	MAT3inverse (mat, &tmp);
}

void
xyz2sRGB (LPVEC3 in, LPVEC3 out, LPMAT3 mat)
{
	MAT3eval (out, mat, in);

	if (out->n[0] < 0.0) out->n[0] = 0.0;
	if (out->n[0] > 1.0) out->n[0] = 1.0;
	if (out->n[1] < 0.0) out->n[1] = 0.0;
	if (out->n[1] > 1.0) out->n[1] = 1.0;
	if (out->n[2] < 0.0) out->n[2] = 0.0;
	if (out->n[2] > 1.0) out->n[2] = 1.0;
}

#define OneThird 0.3333333333333333333333333333

static inline double f (double t)
{
	if (t > 0.008856)
		return pow (t, OneThird);
	else
		return 7.787 * t + 16.0/116.0;
}

void
xyz2Lab (LPVEC3 in, LPVEC3 out, LPVEC3 wtpt)
{
	double XXn = in->n[0] / wtpt->n[0];
	double YYn = in->n[1] / wtpt->n[1];
	double ZZn = in->n[2] / wtpt->n[2];
	double y3;
	if (YYn > 0.008856) {
		y3 = pow (YYn, OneThird);
		out->n[0] = 116.0 * y3 - 16;
	}
	else {
		out->n[0] = 903.3 * YYn;
		y3 = 7.787 * YYn + 16.0/116.0;
	}
	out->n[1] = 500.0 * (pow(XXn, OneThird) - y3);
	out->n[2] = 200.0 * (y3 - pow(ZZn, OneThird));
}

void
Lab2xyz (LPVEC3 in, LPVEC3 out, LPVEC3 wtpt)
{
	double P = (in->n[0] + 16.0) / 116.0;
	out->n[0] = wtpt->n[0] * pow (P + in->n[1] / 500.0, 3.0);
	out->n[1] = wtpt->n[1] * pow (P, 3.0);
	out->n[2] = wtpt->n[2] * pow (P - in->n[2] / 200.0, 3.0);
}

void
Lab2sRGB (LPVEC3 in, LPVEC3 out, LPMAT3 mat)
{
	VEC3 tmp;
	Lab2xyz (in, &tmp, &oneWhitePt);
	xyz2sRGB (&tmp, out, mat);
}

struct MRIStFConvertData {
	struct link *next;
	int	width;
	int	height;
	int	freedata;
	int	useLab;
	MAT3    mat;
	MRI_balance balance;
};

void
MRIStFConvertStart (void *private, int width, int height, int freedata)
{
	struct MRIStFConvertData *wd = private;

	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, width, height, TRUE);
}

void
MRILEtFConvertRow (void *private, void *data)
{
	struct MRIStFConvertData *wd = private;
	struct MRI_ScanLine *sl = data;
	struct MRI_ScanLine *fp;
	int j;

	fp = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
	if (sl->scanLineType == LINETYPE_SHORT) {
	    for (j = 0; j < wd->width; j++) {
		unsigned short rgb[3];
		cmsCIELab Lab;
		rgb[0] = ((unsigned short *)(sl->R))[j];
		rgb[1] = ((unsigned short *)(sl->G))[j];
		rgb[2] = ((unsigned short *)(sl->B))[j];
		cmsLabEncoded2Float (&Lab, rgb);
		((double *)(fp->R))[j] = Lab.L * 1.0;
		((double *)(fp->G))[j] = Lab.a * 1.0;
		((double *)(fp->B))[j] = Lab.b * 1.0;
	    }
	}
	else if (sl->scanLineType == LINETYPE_FLOAT) {
	    float *R = sl->R;
	    float *G = sl->G;
	    float *B = sl->B;
	    for (j = 0; j < wd->width; j++) {
		unsigned short rgb[3];
		cmsCIELab Lab;
		rgb[0] = R[j] < 0 ? 0 : R[j] > 65535.0 ? 65535 : R[j];
		rgb[1] = G[j] < 0 ? 0 : G[j] > 65535.0 ? 65535 : G[j];
		rgb[2] = B[j] < 0 ? 0 : B[j] > 65535.0 ? 65535 : B[j];
		cmsLabEncoded2Float (&Lab, rgb);
		((double *)(fp->R))[j] = Lab.L * 1.0;
		((double *)(fp->G))[j] = Lab.a * 1.0;
		((double *)(fp->B))[j] = Lab.b * 1.0;
	    }
	}
	if (wd->freedata)
		MRI_FreeScanLine (sl);

	(*wd->next->row) (wd->next->private, fp);
}

void
MRIStFConvertRow (void *private, void *data)
{
	struct MRIStFConvertData *wd = private;
	struct MRI_ScanLine *sl = data;
	int j;
	struct MRI_ScanLine *fp;
	double rscale = wd->balance.rgain / (256.0 * 655.35);
	double gscale = wd->balance.ggain / (256.0 * 655.35);
	double bscale = wd->balance.bgain / (256.0 * 655.35);

	fp = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
	if (sl->scanLineType == LINETYPE_SHORT) {
	    for (j = 0; j < wd->width; j++) {
		((double *)(fp->R))[j] = ((unsigned short *)(sl->R))[j] * rscale;
		((double *)(fp->G))[j] = ((unsigned short *)(sl->G))[j] * gscale;
		((double *)(fp->B))[j] = ((unsigned short *)(sl->B))[j] * bscale;
		
	    }
	}
	else if (sl->scanLineType == LINETYPE_FLOAT) {
	    for (j = 0; j < wd->width; j++) {
		((double *)(fp->R))[j] = ((float *)(sl->R))[j] * rscale;
		((double *)(fp->G))[j] = ((float *)(sl->G))[j] * gscale;
		((double *)(fp->B))[j] = ((float *)(sl->B))[j] * bscale;
		
	    }
	}
	if (wd->freedata)
		MRI_FreeScanLine (sl);

	(*wd->next->row) (wd->next->private, fp);
}

void
MRIFtLEConvertRow (void *private, void *data)
{
	struct MRIStFConvertData *wd = private;
	struct MRI_ScanLine *fp = data;
	int j;
	struct MRI_ScanLine *sl;
	cmsCIELab Lab;
	unsigned short rgb[3];

	sl = MRI_NewScanLine (LINETYPE_SHORT, wd->width);

	for (j = 0; j < wd->width; j++) {
		Lab.L = ((double *)(fp->R))[j];
		Lab.a = ((double *)(fp->G))[j];
		Lab.b = ((double *)(fp->B))[j];
		cmsFloat2LabEncoded (rgb, &Lab);
		((unsigned short *)(sl->R))[j] = rgb[0];
		((unsigned short *)(sl->G))[j] = rgb[1];
		((unsigned short *)(sl->B))[j] = rgb[2];
	}
	if (wd->freedata)
		MRI_FreeScanLine (fp);

	(*wd->next->row) (wd->next->private, sl);
}

void
MRIFtSConvertRow (void *private, void *data)
{
	struct MRIStFConvertData *wd = private;
	struct MRI_ScanLine *fp = data;
	int j;
	VEC3 in, out;
	struct MRI_ScanLine *sl;
	double t;

	sl = MRI_NewScanLine (LINETYPE_SHORT, wd->width);

	for (j = 0; j < wd->width; j++) {
		in.n[0] = ((double *)(fp->R))[j];
		in.n[1] = ((double *)(fp->G))[j];
		in.n[2] = ((double *)(fp->B))[j];
		if (wd->useLab) {
			Lab2sRGB (&in, &out, &wd->mat);
			t = out.n[0] * 65535.0;
			((unsigned short *)(sl->R))[j] = t >= 65534.5 ? 65535 : t;
			t = out.n[1] * 65535.0;
			((unsigned short *)(sl->G))[j] = t >= 65534.5 ? 65535 : t;
			t = out.n[2] * 65535.0;
			((unsigned short *)(sl->B))[j] = t >= 65534.5 ? 65535 : t;
		}
		else {
			t = in.n[0] * 655.35;
			((unsigned short *)(sl->R))[j] = t >= 65534.5 ? 65535 : t;
			t = in.n[1] * 655.35;
			((unsigned short *)(sl->G))[j] = t >= 65534.5 ? 65535 : t;
			t = in.n[2] * 655.35;
			((unsigned short *)(sl->B))[j] = t >= 65534.5 ? 65535 : t;
		}
	}
	if (wd->freedata)
		MRI_FreeScanLine (fp);

	(*wd->next->row) (wd->next->private, sl);
}

void
MRIStFConvertClose (void *private)
{
	struct MRIStFConvertData *wd = private;
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
GenMRIStFConverter ( struct link *next, MRI_balance balance, int useLab)
{
	struct MRIStFConvertData *wd = malloc (sizeof (struct MRIStFConvertData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct MRIStFConvertData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIStFConvertStart;
	ep->row = useLab ? MRILEtFConvertRow : MRIStFConvertRow;
	ep->close = MRIStFConvertClose;
	ep->private = wd;
	wd->next = next;
	wd->balance = balance;
	wd->useLab = useLab;

	return ep;
}

struct link *
MRI_GenFtSConverter ( struct link *next, const MRI *mri, int useLab)
{
	struct MRIStFConvertData *wd = malloc (sizeof (struct MRIStFConvertData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MRIStFConvertData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIStFConvertStart;
	ep->row = useLab ? MRIFtLEConvertRow : MRIFtSConvertRow;
	ep->close = MRIStFConvertClose;
	ep->private = wd;
	wd->next = next;
	wd->useLab = useLab;
	if (useLab)
	    createXYZtoRGBmat(&wd->mat, mri);

	return ep;
}
