/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef mrpt_utils_types_H
#define mrpt_utils_types_H

#include <vector>
#include <list>
#include <set>
#include <map>
#include <string>
#include <stdexcept>

// Standard elemental types:
#include "pstdint.h"  // The "portable stdint header file"

#define __STDC_FORMAT_MACROS
#if HAVE_INTTYPES_H
#	include <inttypes.h>
#elif defined(_MSC_VER)
#	include	<mrpt/utils/msvc_inttypes.h>
#endif

// SSE2, SSE3 types:
#if MRPT_HAS_SSE2
	#include <emmintrin.h>
	#include <mmintrin.h>
#endif

namespace mrpt
{
	typedef std::vector<int8_t>  vector_signed_byte;	//!<  A type for passing a vector of signed bytes.
	typedef std::vector<int16_t> vector_signed_word;	//!<  A type for passing a vector of 16-bits signed numbers.
	typedef std::vector<int32_t> vector_int;	//!< A type for passing a vector of ints.
	typedef std::vector<int64_t> vector_long;	//!< A type for passing a vector of longs.
	typedef std::vector<float>   vector_float;	//!< A type for passing a vector of floats.
	typedef std::vector<double>  vector_double;	//!< A type for passing a vector of doubles.

	typedef std::vector<size_t> vector_size_t;	//!<  A type for vectors of size_t's.
	typedef std::vector<uint8_t> vector_byte;	//!<  A type for passing a vector of bytes.
	typedef std::vector<uint16_t> vector_word;	//!<  A type for passing a vector of 16-bits unsigned numbers.
	typedef std::vector<uint32_t> vector_uint;	//!<  A type for passing a vector of 32-bits unsigned numbers.
	
	typedef std::vector<std::string> vector_string;	//!<  A type for passing a vector of strings.
	typedef std::vector<bool> vector_bool;	//!<  A type for passing a vector of boolean values.

	namespace utils
	{
		// --------------------------
		// Functors:
		//  Ret: NO  In:1-3
		// --------------------------

		/** A generic functor type for functions accepting 1 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_1inputs)(const void *);

		/** A generic functor type for functions accepting 2 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_2inputs)(const void *,const void *);

		/** A generic functor type for functions accepting 3 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_3inputs)(const void *,const void *,const void *);

		// --------------------------
		// Functors:
		//  Ret: double  In:1-3
		// --------------------------

		/** A generic functor type for functions accepting 1 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_1inputs)(const void *);

		/** A generic functor type for functions accepting 2 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_2inputs)(const void *,const void *);

		/** A generic functor type for functions accepting 3 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_3inputs)(const void *,const void *,const void *);

		// --------------------------
		// Functors:
		//  Ret: vector  In: vector
		// --------------------------

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecDbl_inpVecDbl)(const vector_double &in, vector_double &out);

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecFlt_inpVecFlt)(const vector_float &in, vector_float &out);

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecInt_inpVecInt)(const vector_int &in, vector_int &out);

		// -----------------------------
		// Functors:
		//  Ret: vector  In: 2 x vector
		// -----------------------------

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecDbl_inp2VecDbl)(const vector_double &x,const vector_double &y, vector_double &out);

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecFlt_inp2VecFlt)(const vector_float &x,const vector_float &y, vector_float &out);

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecInt_inp2VecInt)(const vector_int &x,const vector_int &y, vector_int &out);

		// --------------------------
		// Functors:
		//  Ret: double  In: vectors
		// --------------------------

		/** A generic functor type for functions accepting 1 vector and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp1VecDbl)(const vector_double &in1);

		/** A generic functor type for functions accepting 2 vectors and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp2VecDbl)(const vector_double &in1,const vector_double &in2);

		/** A generic functor type for functions accepting 3 vectors and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp3VecDbl)(const vector_double &in1,const vector_double &in2,const vector_double &in3);


		/** For performing type casting from a pointer to its numeric value.
		*/
		#if defined(_MSC_VER) && (_MSC_VER>=1300)
			typedef unsigned long long POINTER_TYPE;
		#else
			typedef unsigned long POINTER_TYPE;
		#endif

		/** A RGB color - 8bit */
		struct MRPTDLLIMPEXP TColor
		{
			TColor(uint8_t r=0,uint8_t g=0,uint8_t b=0, uint8_t alpha=255) : R(r),G(g),B(b),A(alpha) { }
			uint8_t R,G,B,A;
			/** Operator for implicit conversion into an int binary representation 0xRRGGBB */
			operator int(void) const { return (((int)R)<<16) | (((int)G)<<8) | B; }

			static TColor red; //!< Predefined colors
			static TColor green;//!< Predefined colors
			static TColor blue;//!< Predefined colors
			static TColor white;//!< Predefined colors
			static TColor black;//!< Predefined colors
			static TColor gray;	//!< Predefined colors
		};

		/** A RGB color - floats in the range [0,1] */
		struct MRPTDLLIMPEXP TColorf
		{
			TColorf(float r=0,float g=0,float b=0, float alpha=1.0f) : R(r),G(g),B(b),A(alpha) { }
			float R,G,B,A;
		};

		/** A pair (x,y) of pixel coordinates (subpixel resolution). */
		struct MRPTDLLIMPEXP TPixelCoordf
		{
			double x,y;

			/** Default constructor: undefined values of x,y */
			TPixelCoordf() : x(),y() {}

			/** Constructor from x,y values */
			TPixelCoordf(const double _x,const double _y) : x(_x), y(_y) { }
		};

		/** A pair (x,y) of pixel coordinates (integer resolution). */
		struct MRPTDLLIMPEXP TPixelCoord
		{
			TPixelCoord() : x(0),y(0) { }
			TPixelCoord(const int _x,const int _y) : x(_x), y(_y) { }

			int x,y;
		};

		typedef TPixelCoord TImageSize; //!< A type for image sizes.

		/** For usage when passing a dynamic number of (numeric) arguments to a function, by name.
		  *  \code
		  *    TParameters<double> p;
		  *    p["v_max"] = 1.0;  // Write
		  *    ...
		  *    cout << p["w_max"]; // Read, even if "p" is const.
		  *  \endcode
		  */
		template <typename T>
		struct TParameters : public std::map<std::string,T>
		{
			TParameters() : std::map<std::string,T> () { }
			virtual ~TParameters() { std::map<std::string,T>::clear(); }
			bool has(const std::string &s) const {
				return std::map<std::string,T>::end()!=std::map<std::string,T>::find(s);
			}
			/** A const version of the [] operator, for usage as read-only.
			  * \exception std::logic_error On parameter not present. Please, check existence with "has" before reading.
			  */
			T operator[](const std::string &s) const {
				typename std::map<std::string,T>::const_iterator it =std::map<std::string,T>::find(s);
				if (std::map<std::string,T>::end()==it)
					throw std::logic_error(std::string("Parameter '")+s+std::string("' is not present.").c_str());
				return it->second;
			}
			/** The write (non-const) version of the [] operator. */
			T & operator[](const std::string &s) {
				return std::map<std::string,T>::operator[](s);
			}
		};
	} // end namespace
}

#endif

