/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  MRPT_OS_H
#define  MRPT_OS_H

#include <mrpt/config.h>

#include <cstdarg>
#include <cstdlib>
#include <cstring>
#include <deque>
#include <vector>

// Duplicated here since <mrpt/system/os.h> is the only header that cannot include "utils_defs.h"
#define _IAMINUTILSDEFS_H
#include <mrpt/utils/utils_impexp.h>  // DLL import/export definitions
#undef _IAMINUTILSDEFS_H

#include <mrpt/utils/types.h>  // This must be AFTER <utils_impexp.h>

// Define a decl. modifier for printf-like format checks at compile time:
#ifdef __GNUC__
#	define MRPT_printf_format_check(_FMT_,_VARARGS_)  __attribute__ ((__format__ (__printf__, _FMT_,_VARARGS_)))
#else
#	define MRPT_printf_format_check(_FMT_,_VARARGS_)
#endif

// Define a decl. modifier for scanf-like format checks at compile time:
#ifdef __GNUC__
#	define MRPT_scanf_format_check(_FMT_,_VARARGS_)  __attribute__ ((__format__ (__scanf__, _FMT_,_VARARGS_)))
#else
#	define MRPT_scanf_format_check(_FMT_,_VARARGS_)
#endif


/** Used after member declarations */
#define MRPT_NO_THROWS		throw()

/** Represents an invalid timestamp, where applicable.
  */
#define INVALID_TIMESTAMP (0)

namespace mrpt
{
	/** This namespace provides a OS-independent interface to many useful functions: filenames manipulation, time and date, string parsing, file I/O, threading, memory allocation, etc.
	 *  \sa mrpt::system::os
	 */
	namespace system
	{
		/** This namespace provides a OS-independent interface to low-level functions.
		 *   Most of these functions are converted into calls to standard functions, unless we are into Visual Studio 2005 (or newer). In that case the secure version
		 *     of the standard library functions (prefix "_s") are used instead.
		 */
		namespace os
		{
			/** An OS-independent version of sprintf (Notice the bufSize param, which may be ignored in some compilers)
			  *  \sa utils::format
			  */
			int MRPTDLLIMPEXP sprintf(char *buf, size_t bufSize, const char *format, ...) MRPT_NO_THROWS MRPT_printf_format_check(3,4);

			/** An OS-independent version of vsprintf (Notice the bufSize param, which may be ignored in some compilers)
			  */
			int MRPTDLLIMPEXP vsprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS;

			/** An OS-independent version of vsnprintf (Notice the bufSize param, which may be ignored in some compilers)
			  */
			int MRPTDLLIMPEXP vsnprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS;

			/** An OS-independent version of fopen.
			  * \return It will always return NULL on any error.
			  */
			FILE MRPTDLLIMPEXP *fopen(const char *fileName,const char *mode) MRPT_NO_THROWS;

			/** An OS-independent version of fopen (std::string version)
			  * \return It will always return NULL on any error.
			  */
			FILE MRPTDLLIMPEXP *fopen(const std::string &fileName,const char *mode) MRPT_NO_THROWS;

			/** An OS-independent version of fprintf
			  */
			int MRPTDLLIMPEXP fprintf(FILE *fil, const char *format, ...)  MRPT_NO_THROWS MRPT_printf_format_check(2,3);

			/** An OS-independent version of fscanf
			  * \return The number of fields correctly assigned
			  */
			int MRPTDLLIMPEXP fscanf(FILE *fil, const char *format, ...)  MRPT_NO_THROWS MRPT_scanf_format_check(2,3);

			/** An OS-independent version of fclose.
			  * \exception std::exception On trying to close a NULL file descriptor.
			  */
			void MRPTDLLIMPEXP fclose(FILE *f);

			/** An OS-independent version of strcat.
			  * \return It will always return the "dest" pointer.
			  */
			char MRPTDLLIMPEXP * strcat(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS;

			/** An OS-independent version of strcpy.
			  * \return It will always return the "dest" pointer.
			  */
			char  MRPTDLLIMPEXP *strcpy(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS;

			/** An OS-independent version of strcmp.
			  * \return It will return 0 when both strings are equal, casi sensitive.
			  */
			int MRPTDLLIMPEXP _strcmp(const char*str1,const char*str2) MRPT_NO_THROWS;

			/** An OS-independent version of strcmpi.
			  * \return It will return 0 when both strings are equal, casi insensitive.
			  */
			int MRPTDLLIMPEXP _strcmpi(const char*str1,const char*str2) MRPT_NO_THROWS;

			/** An OS-independent version of strtoll.
			  */
			int64_t MRPTDLLIMPEXP _strtoll(const char *nptr, char **endptr, int base);

			/** An OS-independent version of strtoull.
			  */
			uint64_t MRPTDLLIMPEXP _strtoull(const char *nptr, char **endptr, int base);

			/** An OS-independent version of timegm (which is not present in all compilers): converts a time structure into an UTM time_t */
			time_t MRPTDLLIMPEXP timegm(struct tm *tm);

			/** An OS and compiler independent version of "memcpy"
			  */
			void MRPTDLLIMPEXP memcpy(
				void		*dest,
				size_t		destSize,
				const void	*src,
				size_t		copyCount ) MRPT_NO_THROWS;

			/** An OS-independent version of getch, which waits until a key is pushed.
			  * \return The pushed key code
			  */
			int MRPTDLLIMPEXP getch() MRPT_NO_THROWS;

			/** An OS-independent version of kbhit, which returns true if a key has been pushed.
			  */
			bool MRPTDLLIMPEXP kbhit() MRPT_NO_THROWS;

		}	// end namespace "os"

        /** Returns true if the number is NaN. */
        bool MRPTDLLIMPEXP  isNaN(float  f) MRPT_NO_THROWS;

        /** Returns true if the number is NaN. */
        bool  MRPTDLLIMPEXP isNaN(double f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity. */
        bool MRPTDLLIMPEXP  isFinite(float f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity.  */
        bool  MRPTDLLIMPEXP isFinite(double f) MRPT_NO_THROWS;

#ifdef HAVE_LONG_DOUBLE
		/** Returns true if the number is NaN. */
        bool  MRPTDLLIMPEXP isNaN(long double f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity. */
        bool MRPTDLLIMPEXP  isFinite(long double f) MRPT_NO_THROWS;
#endif

		/** Shows the message "Press any key to continue" (or other custom message) to the current standard output and returns when a key is pressed.
		  */
		void MRPTDLLIMPEXP pause(const std::string &msg = std::string("Press any key to continue...") ) MRPT_NO_THROWS;

		/** Clears the console window */
		void MRPTDLLIMPEXP clearConsole();

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<float> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<double> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<int> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		* \sa vectorToBinaryFile
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<size_t> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** Saves a vector directly as a binary dump to a file:
		* \return Returns false on any error, true on everything OK.
		* \sa loadBinaryFile
		*/
		bool MRPTDLLIMPEXP vectorToBinaryFile( const vector_byte &vec, const std::string &fileName );

		/** Loads a entire file as a vector of bytes.
		* \return Returns false on any error, true on everything OK.
		* \sa vectorToBinaryFile
		*/
		bool MRPTDLLIMPEXP loadBinaryFile( vector_byte &out_data, const std::string &fileName );

		/** Returns the MRPT compilation date
		  */
		std::string MRPTDLLIMPEXP MRPT_getCompilationDate();

		/** Returns a string describing the MRPT version including the SVN number.
		  */
		std::string MRPTDLLIMPEXP MRPT_getVersion();

		/** Call this to register handlers for fatal erros (memory access,etc) that show useful debug information (It is called automatically normally, no need for the user to explicitly call this method.).
		  */
		void MRPTDLLIMPEXP registerFatalExceptionHandlers();

		/** Dumps the current program stack with detailed information of source files and lines.
		  *  This function requires MRPT linked against wxWidgets. Otherwise, an empty string is returned.
		  *  File names and lines won't be available in release builds.
		  */
		std::string MRPTDLLIMPEXP stack_trace(bool calling_from_exception = false);

		/** For use in  setConsoleColor */
		enum TConsoleColor
		{
			CONCOL_NORMAL = 0,
			CONCOL_BLUE   = 1,
			CONCOL_GREEN  = 2,
			CONCOL_RED    = 4
		};

		/** Changes the text color in the console for the text written from now on.
		  * The parameter "color" can be any value in TConsoleColor.
		  *
		  * By default the color of "cout" is changed, unless changeStdErr=true, in which case "cerr" is changed.
		  */
		void MRPTDLLIMPEXP setConsoleColor( TConsoleColor color, bool changeStdErr=false );

	} // End of namespace

} // End of namespace

#endif
