/* ***** BEGIN LICENSE BLOCK *****
 *	 Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is moz-gnome-pm.
 *
 * The Initial Developer of the Original Code is
 * Canonical Ltd.
 * Portions created by the Initial Developer are Copyright (C) 2010
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 * Chris Coulson <chris.coulson@canonical.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 * 
 * ***** END LICENSE BLOCK ***** */

const Ci = Components.interfaces;
const Cc = Components.classes;
const Cr = Components.results;
const Cu = Components.utils;

const nsISupports 			= Ci.nsISupports;
const IGnomeSessionInhibitorService 	= Ci.IGnomeSessionInhibitorService;
const nsIComponentRegistrar		= Ci.nsIComponentRegistrar;
const nsIFactory			= Ci.nsIFactory;
const nsIProcess			= Ci.nsIProcess;
const nsILocalFile			= Ci.nsILocalFile;
const nsIProperties			= Ci.nsIProperties;
const nsIFile				= Ci.nsIFile;
const nsIObserverService		= Ci.nsIObserverService;

const CLASS_ID = Components.ID("{af7a4e5d-a7e2-4617-9ce0-2d7d18e2b099}");
const CLASS_NAME = "GNOME session inhibitor service";
const CONTRACT_ID = "@canonical.com/gnome-session-inhibitor-service;1";
const MY_ID = "{a03390fe-612a-4cfb-9861-ce9369d54a0b}";

try {
	// Gecko >= 2.0 only
	Cu.import("resource://gre/modules/AddonManager.jsm");
} catch(ex) {}

try {
	// Gecko < 2.0 only
	const nsIExtensionManager		= Ci.nsIExtensionManager;
} catch (ex) {}

function InhibitorReq(flags, reason, cookie) {
	this.flags = flags;
	this.reason = reason;
	this.cookie = cookie;
}

// The inhibitor list maintains a list of registered inhibitors for each inhibit
// type. Inhibits are registered one at a time on the session manager in a FIFO fashion
// to avoid cluttering the UI when the inhibit dialog is displayed.

function Inhibitor(flag, helper) {
	this._init(flag, helper);

}

Inhibitor.prototype = {
	_init: function(flag, helper) {
		this.flag = flag;
		this._inhibitors_by_cookie = new Array();
		this._inhibitor_queue = new Array();
		this._helper = Cc["@mozilla.org/process/util;1"].createInstance(nsIProcess);
		this._terminator = Cc["@mozilla.org/process/util;1"].createInstance(nsIProcess);
		this._current = null;

		this._helper_is_stopping = false;
		this._helper_wants_restart = false;

		this._helper.init(helper);

		var file = Cc["@mozilla.org/file/local;1"].createInstance(nsILocalFile);
		file.initWithPath("/bin/kill");
		this._terminator.init(file);
	},

	observe: function(aSubject, aTopic, aData) {
		if((aTopic == "process-finished") || (aTopic == "process-failed")) {
			if(this._helper_is_stopping == true) {
				this._helper_is_stopping = false;
				if(this._helper_wants_restart == true) {
					this._helper_wants_restart = false;
					this._inhibit();
				}
				return;
			}

			if(this._helper.exitValue != 0) {
				window.dump("gnome-session-inhibitor-helper.py exited with exit status " + this._helper.exitValue);
			} else {
				// Try to restart it again
				this._inhibit();
			}
		}
	},

	_inhibit: function() {
		// We have to spawn a helper process to talk to the session manager. This is because
		// trying to get ctypes to work is about as much fun as stepping in a giant steaming turd
		if(this._helper.isRunning == true) {
			var kill_args = new Array();
			kill_args[0] = "-TERM";
			kill_args[1] = this._helper.pid;
			this._helper_is_stopping = true;
			this._helper_wants_restart = true;
			this._terminator.run(false, kill_args, kill_args.length);
			return;
		}
		var file = Cc["@mozilla.org/file/directory_service;1"].getService(nsIProperties).get("ProfD", nsIFile);
		file.append(".parentlock");
		var args = new Array();
		args[0] = "-a";
		// FIXME: This should match the name of the desktop file that launched us. How do we do this
		// on versioned builds like firefox-4?
		args[1] = "firefox";
		args[2] = "-r";
		args[3] = this._current.reason;
		args[4] = "-f";
		args[5] = this.flag;
		args[6] = "-l";
		args[7] = file.path;
		this._helper.runAsync(args, args.length, this);
	},

	_uninhibit: function() {
		if(this._helper.isRunning == true) {
			// This sucks. nsIProcess.kill() sends SIGKILL, which the helper cannot catch
			// in order to exit cleanly. So, I have to run /bin/kill to ensure we send SIGTERM
			// Yuck yuck yuck!
			var kill_args = new Array();
			kill_args[0] = "-TERM";
			kill_args[1] = this._helper.pid;
			this._helper_is_stopping = true;
			this._terminator.run(false, kill_args, kill_args.length);
		}
	},

	add: function(inhibitor) {
		this._inhibitors_by_cookie[inhibitor.cookie] = inhibitor;
		this._inhibitor_queue.push(inhibitor.cookie);
		this._maybe_inhibit();
	},

	remove: function(cookie) {
		// Is this the current inhibitor?
		if(this._current != null) {
			if(cookie == this._current.cookie) {
				delete this._current;
			}
		}

		// Remove it from the list if it exists
		delete this._inhibitors_by_cookie[cookie];
		this._maybe_inhibit();
	},

	_maybe_inhibit: function() {
		// If there is a current inhibitor, then there's no need to do anything
		if(this._current != null) {
			return;
		}

		// If there's not a current inhibitor, find one in the queue
		do {
			var new_inhibit = this._inhibitor_queue.shift();
			// The queue is empty, so we want to remove the inhibitor from the session manager
			if(new_inhibit == null) {
				this._uninhibit();
				return;
			}
		} while (this._inhibitors_by_cookie[new_inhibit] == null);

		this._current = this._inhibitors_by_cookie[new_inhibit];
		this._inhibit();
	}
}

function GnomeSessionInhibitorService() {
	this._init();
	this.wrappedJSObject = this;
}

// The service maintains a list of inhibitors for each inhibitor type
GnomeSessionInhibitorService.prototype = {
	QueryInterface: function(aIID) {
		if (!aIID.equals(nsISupports) 
			&& !aIID.equals(IGnomeSessionInhibitorService)) {
				throw Components.results.NS_ERROR_NO_INTERFACE;
		}
		return this;
	},

	observe: function(aSubject, aTopic, aData) {
		if(aTopic == "quit-application") {
			// Free all of the inhibitors, which will make sure the helpers exit cleanly
			for(var i in this._cookies) {
				this.uninhibit(i);
			}
		}
	},

	_init: function() {
		var obj = this;
		if(typeof AddonManager == "undefined") {
			// Gecko < 2.0
			var em = Cc["@mozilla.org/extensions/manager;1"].getService(nsIExtensionManager);
			this._helper = em.getInstallLocation(MY_ID).getItemFile(MY_ID, "gnome-session-inhibit-helper.py");
			this._init2();
		} else {
			// Gecko >= 2.0
			AddonManager.getAddonByID(MY_ID, function(addon) {
  				obj._helper = addon.getResourceURI("gnome-session-inhibit-helper.py").QueryInterface(Components.interfaces.nsIFileURL).file;
				obj._init2();
			});  
		}
	},

	_init2: function () {
		// Urgh! The add-on manager doesn't set the executable bit on the helper, so we need to do it here
		var new_permissions = this._helper.permissions | 0111;
		if(this._helper.permissions != new_permissions) {
			this._helper.permissions = new_permissions;
		}

		// Create the list for each inhibitor type
		this._inhibitor_list = new Array();
		this._inhibitor_list[GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_LOGOUT] = 
			new Inhibitor(GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_LOGOUT, this._helper);
		this._inhibitor_list[GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SWITCH_USER] = 
			new Inhibitor(GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SWITCH_USER, this._helper);
		this._inhibitor_list[GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SUSPEND] = 
			new Inhibitor(GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SUSPEND, this._helper);
		this._inhibitor_list[GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_IDLE] = 
			new Inhibitor(GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_IDLE, this._helper);

		// We keep a list of cookies so we don't create the same cookie twice
		this._cookies = new Array();
		this._cookie_count = 0;
		var os = Cc["@mozilla.org/observer-service;1"].getService(nsIObserverService);
		os.addObserver(this, "quit-application", false);
	},

	_generate_cookie: function() {
		var cookie;
		if(this._cookie_count >= 100) {
			// We don't want to spend too much time looking for a free cookie,
			// and 100 should be more than enough here
			throw Cr.NS_ERROR_FAILURE;
		}

		do {
			cookie = Math.floor(Math.random() * 65536);
		} while (this._cookies[cookie] != null);

		this._cookies[cookie] = 1;
		this._cookie_count++;
		return cookie;
	},

	_free_cookie: function(cookie) {
		if(this._cookies[cookie] == null) {
			throw Cr.NS_ERROR_INVALID_ARG;
		}

		delete this._cookies[cookie];
		this._cookie_count--;

		if(this._cookie_count < 0) {
			// This shouldn't happen
			throw Cr.NS_ERROR_UNEXPECTED;
		}
	},

	_remove_inhibitor_req: function(cookie) {
		for(var i in this._inhibitor_list) {
			var iter = this._inhibitor_list[i];
			iter.remove(cookie);
		}
	},

	// Add the inhibitor request to the appropriate inhibitor
	_add_inhibitor_req: function(inhibitor_req) {
		for(var i in this._inhibitor_list) {
			var iter = this._inhibitor_list[i];
			if((inhibitor_req.flags & iter.flag) != 0) {
				iter.add(inhibitor_req);
			}
		}
	},

	inhibit: function(flags, reason) {
		var cookie = this._generate_cookie();
		var inhibitor_req = new InhibitorReq(flags, reason, cookie);
		this._add_inhibitor_req(inhibitor_req);

		return cookie;
	},

	uninhibit: function(cookie) {
		this._remove_inhibitor_req(cookie);
		this._free_cookie(cookie);
	}
}

// GsmInhibitor flags
GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_LOGOUT 		= 1;
GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SWITCH_USER	= 2;
GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_SUSPEND 	= 4;
GnomeSessionInhibitorService.GSM_INHIBITOR_FLAG_IDLE 		= 8;

// All the stuff to register the service goes below here. This should be easy with XPCOMUtils.jsm,
// but I want to create a service rather than a component that can be instantiated multiple times,
// and XPCOMUtils.jsm doesn't seem to support this, unless I'm just being a retard
// Factory
var GnomeSessionInhibitorFactory = {
	singleton: null,
	createInstance: function (aOuter, aIID) {
		if (aOuter != null)
			throw Components.results.NS_ERROR_NO_AGGREGATION;
		if (this.singleton == null)
			this.singleton = new GnomeSessionInhibitorService();
		return this.singleton.QueryInterface(aIID);
	}
};

// Module (only for Gecko < 2.0)
var GnomeSessionInhibitorModule = {
	registerSelf: function(aCompMgr, aFileSpec, aLocation, aType) {
		aCompMgr = aCompMgr.QueryInterface(nsIComponentRegistrar);
		aCompMgr.registerFactoryLocation(CLASS_ID, CLASS_NAME, CONTRACT_ID, aFileSpec, aLocation, aType);
	},

	unregisterSelf: function(aCompMgr, aLocation, aType) {
		aCompMgr = aCompMgr.QueryInterface(nsIComponentRegistrar);
		aCompMgr.unregisterFactoryLocation(CLASS_ID, aLocation);				
	},
	
	getClassObject: function(aCompMgr, aCID, aIID) {
		if (!aIID.equals(nsIFactory))
			throw Components.results.NS_ERROR_NOT_IMPLEMENTED;

		if (aCID.equals(CLASS_ID))
			return GnomeSessionInhibitorFactory;

		throw Components.results.NS_ERROR_NO_INTERFACE;
	},

	canUnload: function(aCompMgr) { return true; }
};

//module initialization Gecko < 2.0
function NSGetModule(aCompMgr, aFileSpec)
{
	return GnomeSessionInhibitorModule;
}

//module initialization Gecko >= 2.0
function NSGetFactory(cid)
{
	if (cid.equals(CLASS_ID)) {
		return GnomeSessionInhibitorFactory;
	}
	throw Cr.NS_ERROR_FACTORY_NOT_REGISTERED;
}

