/***************************************************************************
 *   Copyright (C) 2005 by Dmitry Nezhevenko                               *
 *   dionua@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "p2kproc.h"

#include <time.h>

#include <qthread.h>
#include <qmutex.h>
#include <qapplication.h>
#include <qdatetime.h>
// #include <qptrlist.h>

#include <usb.h>
#include <string.h>


#include <termios.h>
#include <fcntl.h>
#include <errno.h>

#include "custmsg.h"
#include "appconfig.h"

// Intefaces for Motorola P2k phones
#define INTERFACE_ACCESSORIES 0x05
#define INTERFACE_DATA_LOGGING_MCU 0x06
#define INTERFACE_TEST_COMMAND 0x08

// Transfer direction
#define DIR_IN 0
#define DIR_OUT 1

#ifdef MY_DEBUG
#define FUNC(val)					\
	char FUNC_NAME[]=val;				\
	printf("=== [P2k API: %s]\n", val);
#else
#define FUNC(val)					\
	char FUNC_NAME[]=val;
#endif
// 	printf("Launch: %s\n",FUNC_NAME);

#define RAISE(...)					\
{ 							\
	qDebug("(E_%s: %s)", FUNC_NAME, __VA_ARGS__);	\
	return(-1);					\
}

extern appConfig *globalConfig;

INT16 getInt16(unsigned char *s)
{
	INT16 res;
	res=s[0];
	res<<=8; res+=s[1];
// 	printf("[!! getInt16 !!!]: %02x %02x -> %04x\n",s[0],s[1],res);
	return (res);
}

INT32 getInt32(unsigned char *s)
{
	INT32 res;
	res=s[0];
	res<<=8; res+=s[1];
	res<<=8; res+=s[2];
	res<<=8; res+=s[3];
	return (res);
}

void setInt16(unsigned char *s, INT16 value)
{
	s[1]=value%0x100; value/=0x100;
	s[0]=value%0x100;
}

void setInt32(unsigned char *s, INT32 value)
{
	s[3]=value%0x100; value/=0x100;
	s[2]=value%0x100; value/=0x100;
	s[1]=value%0x100; value/=0x100;
	s[0]=value%0x100;
}

P2kProc::P2kProc(QObject * o)
		: parent(o)
{
	drv_initUsb();
	phoneMode=-1;
	phoneHandle=0;
	phone=0;
	state=CONNECT_NONE;
	freeID=0;
	isFirstRead=0;
	lastControlStatus=0;
	lastCnt=-1;

	devlst[0].vendor=0x00;
	devlst[0].product=0x00;
	strcpy(devlst[0].name,"NONE");

	devlst[1].vendor=DEF_PHONE_AT_VENDOR;
	devlst[1].product=DEF_PHONE_AT_PRODUCT;
	strcpy(devlst[1].name,"AT");

	devlst[2].vendor=DEF_PHONE_P2K_VENDOR;
	devlst[2].product=DEF_PHONE_P2K_PRODUCT;
	strcpy(devlst[2].name,"P2K");
}


void P2kProc::setATconfig (unsigned int vendor, unsigned int product)
{
	devlst[1].vendor=vendor;
	devlst[1].product=product;
}

void P2kProc::setP2kconfig (unsigned int vendor, unsigned int product)
{
	devlst[2].vendor=vendor;
	devlst[2].product=product;
}

void P2kProc::setDevice(char * cmd)
{
	strcpy(ACMdev,cmd);
}

void P2kProc::stopThread()
{
	isRunning=0;
}

void P2kProc::postMessage(int style, const QString &msg)
{
	msg_P2kLog * m = new msg_P2kLog(style, msg);
	QApplication::postEvent(parent, m);
}

void P2kProc::postProgress(int pg)
{
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_PROGRESS, QString("%1").arg(pg)));
}

int P2kProc::isBusy()
{
	if (task)
		postMessage(MSGSTYLE_ERROR, QObject::tr("Phone is busy. Please try later"));
	return(task);
}

// ===================== Thread main loop =======================

void P2kProc::run()
{
	isRunning=1;
	task=0;
	while (isRunning)
	{
		doUpdateMode();
		if (task)
		{
			lastStr="";
			switch (task)
			{
			case TASK_CONNECT: doConnect(); break;
			case TASK_DISCONNECT: doDisconnect(); break;
			case TASK_REBOOT: doReboot(); break;
			case TASK_SUSPEND: doSuspend(); break;
			case TASK_GETINFO: doGetInfo(); break;
			case TASK_FILELIST: doGetFileList(); break;
			case TASK_DOWNLOADFILES: doDownloadFiles(); break;
			case TASK_UPLOADFILES: doUploadFiles(); break;
			case TASK_DELETEFILES: doDeleteFiles(); break;
			case TASK_CHANGEATTR: doChangeAttr(); break;
			case TASK_READSEEM: doReadSeem(); break;
			case TASK_WRITESEEM: doWriteSeem(); break;
			case TASK_BACKUP: doBackup(); break;
			case TASK_CREATEDIRECTORY: doCreateDirectory(); break;
			case TASK_REMOVEDIRECTORY: doRemoveDirectory(); break;
			case TASK_SEARCH: doSearch(); break;
			default: postMessage(MSGSTYLE_ERROR, QObject::tr("Unknown task ignored: %1").arg(task));
			}
			QApplication::postEvent(parent, new msg_ParamInfo(PARAM_COMPLETE,
			                        QString("%1 %2").arg(task).arg(lastStr)));
			task=0;
		}
		else msleep(100);
	}
}

// ======================  Action Implementation

void P2kProc::setState(int st)
{
	if (state == st) return;
	state=st;
	switch (state)
	{
	case CONNECT_NONE:
		{
			postMessage(MSGSTYLE_INFO, QObject::tr("Phone disconnected"));
			phone=0;
			phoneHandle=0;
			break;
		}
	case CONNECT_P2K: postMessage(MSGSTYLE_INFO, QObject::tr("Phone connected as P2K")); break;
	default:;
	}

	msg_StatusChanged * msg = new msg_StatusChanged(state);
	QApplication::postEvent(parent, msg);
}

int P2kProc::getState()
{
	return state;
}

void P2kProc::doUpdateMode()
{
	QMutexLocker lock ( &mtxInUse );
	int pm;

	pm=drv_findPhone();

	if (pm!=phoneMode)
	{
		msg_ModeChanged * msg = new msg_ModeChanged(pm);
		QApplication::postEvent(parent, msg);
	}

	if ((phoneMode==-1) && (pm==PHONE_NONE))
	{
		postMessage(MSGSTYLE_INFO, QObject::tr("Phone is unpluged. Please connect it"));
		phoneMode=0;
	}
	if (pm!=phoneMode)
	{
		if (pm!=state)
			setState(CONNECT_NONE);
		qDebug("New mode: %d", pm);
		if (pm==0)
			postMessage(MSGSTYLE_INFO, QObject::tr("Phone is unpluged"));
		else
			postMessage(MSGSTYLE_INFO, QObject::tr("Phone pluged as %1").arg(devlst[pm].name));
		phoneMode=pm;
	}
}

void P2kProc::doConnect()
{
	qDebug("P2kProc::doConnect()");
	postMessage(MSGSTYLE_NONE, QObject::tr("Try to connect"));
	if (drv_connect()>0)
	{
		doUpdateMode();
		setState(CONNECT_P2K);
		//doGetInfo();
	}
	else
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to connect"));
}

void P2kProc::doDisconnect()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Try to disconnect"));
	if (drv_closePhone()>0)
		setState(CONNECT_NONE);
	else
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to disconnect"));
}

void P2kProc::doReboot()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Try to reboot"));
	if (drv_reboot()>0)
	{
		msg_StatusChanged * msg = new msg_StatusChanged(CONNECT_NONE);
		QApplication::postEvent(parent, msg);
		postMessage(MSGSTYLE_NONE, QObject::tr("Phone rebooted"));
	}
	else
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to reboot"));
}

void P2kProc::doSuspend()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Try to suspend"));
	if (drv_suspend()>0)
	{
		msg_StatusChanged * msg = new msg_StatusChanged(CONNECT_NONE);
		QApplication::postEvent(parent, msg);
		postMessage(MSGSTYLE_NONE, QObject::tr("Phone suspended"));
	}
	else
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to suspend"));
}

void P2kProc::doGetInfo()
{
	doGetPhoneModel();
	doGetDriveName();
	doGetFileCount();
	doGetFreeSpace();
	if (memCnt == 1) doGetFileList();
}

void P2kProc::doGetFileCount()
{
	int i;
	i=drv_fileCount();
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get file count"));
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_FILECOUNT, QString("%1").arg(i)));
}

void P2kProc::doGetDriveName()
{
	int i;
	char s[256];
	i=drv_getDriveName((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get drive name"));
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_DRIVENAME, QString("%1").arg(s)));
}

void P2kProc::doGetFreeSpace()
{
	long i;
	char s[256];

	if (drv_getDriveName((unsigned char *) s)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get drive name"));
		return;
	}
	i=drv_freeSpace((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get free space"));
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_FREESPACE, QString("%1").arg(i)));
}

void P2kProc::doGetPhoneModel()
{
	int i;
	char s[256];
	i=drv_getPhoneModel((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get phone model"));
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_PHONEMODEL, QString("%1").arg(s)));
}

void P2kProc::doGetFileList()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Getting file list"));
	int i;
	//i=drv_fileList();
	i=drv_getFullFileList();
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get file list"));
		return;
	}
	else
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));

	// 	postMessage(MSGSTYLE_NONE, "AAA");
}

void P2kProc::doDownloadFiles()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Downloading %1 file(s)").arg(memCnt));
	if (drv_downloadFiles(memLst, memCnt, memDir)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to download files"));
	}
	else
		postMessage(MSGSTYLE_NONE, QObject::tr("Downloading complete"));
	free(memLst);
}

void P2kProc::doUploadFiles()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Uploading %1 file(s)").arg(memCnt));
	if (drv_uploadFiles(memLst, memCnt, memDir, memDir2)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to upload files"));
	}
	else
		postMessage(MSGSTYLE_NONE, QObject::tr("Uploading complete"));
	free(memLst);
}

void P2kProc::doDeleteFiles()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Deleting %1 file(s)").arg(memCnt));
	if (drv_deleteFiles(memLst, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to delete files"));
	}
	else
		postMessage(MSGSTYLE_NONE, QObject::tr("Deleting complete"));
	free(memLst);
}

void P2kProc::doChangeAttr()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Changing attributes  for %1 file(s)").arg(memCnt));
	if (drv_changeAttr(memLst, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to change attributes"));
	}
	else
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));
	free(memLst);
}

void P2kProc::doReadSeem()
{
	//FIXME: doReadSeem
	QString sX=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	postMessage(MSGSTYLE_NONE, QObject::tr("Reading seem %1_%2.seem").arg(sX).arg(sY));
	int t=*memSize;
	*memSize=drv_read_seem(memX, memY, memBuf, t);
	if (*memSize<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to read seem"));
	}
	else
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));
		QApplication::postEvent(parent, new msg_ParamInfo(PARAM_SEEMREADY, "1"));
	}
}

void P2kProc::doWriteSeem()
{
	QString sX=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	postMessage(MSGSTYLE_NONE, QObject::tr("Writing seem %1_%2.seem").arg(sX).arg(sY));
	if (drv_write_seem(memX, memY, memBuf, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to write seem"));
	}
	else
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));
		QApplication::postEvent(parent, new msg_ParamInfo(PARAM_SEEMREADY, "2"));
	}

}

void P2kProc::doBackup()
{
	QString sX1=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY1=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	QString sX2=QString("%1").arg(memX1,0,16).rightJustify(4,'0');
	QString sY2=QString("%1").arg(memY1,0,16).rightJustify(4,'0');

	postMessage(MSGSTYLE_NONE, QObject::tr("Doing backup of seem %1_%2.seem to %3_%4.seem").arg(sX1).arg(sY1).arg(sX2).arg(sY2));

	if (drv_backup(memX, memY, memX1, memY1, (unsigned char*)memDir) <0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to backup seem"));
	}
	else
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Backup complete"));
	}
}

void P2kProc::doCreateDirectory()
{
	if (memDir[strlen(memDir)-1]=='/') memDir[strlen(memDir)-1]=0x00;
	postMessage(MSGSTYLE_NONE, QObject::tr("Creating directory %1").arg(memDir));
	if (FSAC_CreateDirectory(memDir)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to create directory"));
	}
	else
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));
	}
}

void P2kProc::doRemoveDirectory()
{
	if (memDir[strlen(memDir)-1]=='/') memDir[strlen(memDir)-1]=0x00;
	postMessage(MSGSTYLE_NONE, QObject::tr("Deleting directory %1").arg(memDir));
	if (FSAC_RemoveDirectory(memDir)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to delete directory"));
	}
	else
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Complete"));
	}
}

void P2kProc::doSearch()
{
	postMessage(MSGSTYLE_NONE, QObject::tr("Searching for %1").arg(memDir));
	int cnt=drv_search(memDir);
	if (cnt<=0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Bad request or no files found"));
	}
	else
	{
		postMessage(MSGSTYLE_INFO, QObject::tr("Found %1 files").arg(cnt));
		drv_fileList();
	}
}

// ======================= Public methods ============================

int P2kProc::phGetDevList(devInfo * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return 0;

	if (state != CONNECT_NONE) doDisconnect();

	usb_find_busses();
	usb_find_devices();

	struct usb_bus *bus;
	struct usb_device *dev;
	usb_dev_handle *udev;

	int idx=0;
	unsigned int devVendor;
	unsigned int devProduct;
	char devManufacturerStr[200];
	char devProductStr[200];
	int ret;

	for (bus = usb_busses; bus; bus = bus->next)
	{
		for (dev = bus->devices; dev; dev = dev->next)
		{
			devVendor=dev->descriptor.idVendor;
			devProduct=dev->descriptor.idProduct;
			devManufacturerStr[0]=0;
			devProductStr[0]=0;

			udev = usb_open(dev);
			if (udev)
			{
				if (dev->descriptor.iManufacturer)
					ret = usb_get_string_simple(udev, dev->descriptor.iManufacturer,
					                            devManufacturerStr, sizeof(devManufacturerStr));

				if (dev->descriptor.iProduct)
					ret = usb_get_string_simple(udev, dev->descriptor.iProduct,
					                            devProductStr, sizeof(devProductStr));
				usb_close (udev);
			}

			lst[idx].vendor=devVendor;
			lst[idx].product=devProduct;
			strcpy(lst[idx].manufacturerStr, devManufacturerStr);
			strcpy(lst[idx].productStr, devProductStr);
			if (++idx > cnt-1)
			{
				qDebug("phGetDevList: Please, increase cnt!!!");
				return idx;
			}
		}
	}

	return idx;

	//task=TASK_CONNECT;
	//
}

void P2kProc::phConnect()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_CONNECT;
}

void P2kProc::phDisconnect()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_DISCONNECT;
}

void P2kProc::phReboot()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_REBOOT;
}

void P2kProc::phSuspend()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_SUSPEND;
}

void P2kProc::phGetInfo(int what)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memCnt=what;
	task=TASK_GETINFO;
}

void P2kProc::phGetFileList()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_FILELIST;
}

void P2kProc::phDownloadFiles(P2kFile * lst, int cnt, const char * dir)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	strcpy(memDir, dir);
	task=TASK_DOWNLOADFILES;
}

void P2kProc::phUploadFiles(P2kFile * lst, int cnt, const char * dirLocal, const char * dirPhone)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	strcpy(memDir, dirLocal);
	strcpy(memDir2, dirPhone);
	task=TASK_UPLOADFILES;
}

void P2kProc::phDeleteFiles(P2kFile * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	task=TASK_DELETEFILES;
}

void P2kProc::phChangeAttr(P2kFile * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	task=TASK_CHANGEATTR;
}

void P2kProc::phReadSeem(int x, int y, unsigned char * buf, int * size)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memBuf=buf;
	memSize=size;
	memX=x;
	memY=y;
	task=TASK_READSEEM;
}

void P2kProc::phWriteSeem(int x, int y, unsigned char * buf, int size)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memBuf=buf;
	memCnt=size;
	memX=x;
	memY=y;
	task=TASK_WRITESEEM;

}

void P2kProc::phBackup(int x1, int y1, int x2, int y2, unsigned char * dir)
{
	//FIXME:ph
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memX=x1;
	memY=y1;
	memX1=x2;
	memY1=y2;
	strcpy((char *)memDir, (char *)dir);
	task=TASK_BACKUP;
}

void P2kProc::phCreateDirectory(const char *dir)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	strcpy((char *)memDir, (char *)dir);
	task=TASK_CREATEDIRECTORY;
}

void P2kProc::phRemoveDirectory(const char *dir)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	strcpy((char *)memDir, (char *)dir);
	task=TASK_REMOVEDIRECTORY;
}

void P2kProc::phSearch(const char *request)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	strcpy((char *)memDir, (char *)request);
	task=TASK_SEARCH;
}

// ======================= libUsb interface ===========================

// Initialize libUsb
void P2kProc::drv_initUsb()
{
	usb_init();
}

// Detect current phone state
struct usb_device * P2kProc::drv_findDevice(int vendor, int product)
{
	struct usb_bus *busses;
	struct usb_bus *bus;

	usb_find_busses();
	usb_find_devices();

	busses = usb_get_busses();

	for (bus = busses; bus; bus = bus->next)
	{
		struct usb_device *dev;

		for (dev = bus->devices; dev; dev = dev->next)
			if ((dev->descriptor.idVendor==vendor) && (dev->descriptor.idProduct==product))
				return(dev);

	}
	return(0);
}

// Return Current phone state
int P2kProc::drv_findPhone()
{
	for (int i=1; i<=2; i++)
	{
		phone=(drv_findDevice(devlst[i].vendor, devlst[i].product));
		if (phone)  return(i);
	}
	return(PHONE_NONE);
}

// Return !=0 if switch is ok.
int P2kProc::drv_switchP2K(char * st)
{
	long acm;
	char ACMdevice[256];
	char ACMcmd[50];
	QTime elTime;

	strcpy(ACMdevice, (st == 0) ? ACMdev : st);


	postMessage(MSGSTYLE_INFO, QObject::tr("Switching device %1 to P2K mode...").arg(ACMdevice));

	acm=open(ACMdevice, O_RDWR | O_NOCTTY | O_NONBLOCK );
	if (acm<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to open device"));
		postMessage(MSGSTYLE_ERROR, QObject::tr("Please check preferences"));
		return 0;
	}

	struct termios tio;
	bzero(&tio, sizeof(tio));
	tio.c_cflag= CRTSCTS | CS8 | CLOCAL | CREAD | O_NDELAY;
	cfsetispeed(&tio, 115200);
	tio.c_cflag &= ~CRTSCTS;
	tio.c_iflag = IGNPAR;
	tio.c_oflag = 0;
	tio.c_lflag = 0;
	tio.c_cc[VTIME] = 1;
	tio.c_cc[VMIN] = 0;
	tcflush(acm, TCIOFLUSH);
	tcsetattr(acm, TCSANOW, &tio);


	char readBuf[1024];
	int nread;
	
	strcpy(ACMcmd,"AT E0\r\n");
	write(acm, ACMcmd, strlen(ACMcmd));
	tcdrain(acm);
	
	nread=0; elTime.start();
	while (1)
	{
	    nread=read(acm,readBuf,sizeof(readBuf)-1);
	    if (nread>0) break;
	    if (elTime.elapsed()>2000) break;
	}
	if (nread>0)
	{
	    readBuf[nread]=0;
	    postMessage(MSGSTYLE_INFO, QObject::tr("%1 answer: %2").arg(ACMcmd).arg(readBuf));
	}
	

	strcpy(ACMcmd,"AT+MODE=8\r\n");
	write(acm, ACMcmd, strlen(ACMcmd));
	tcdrain(acm);
	
	nread=0; elTime.start();
	while (1)
	{
	    nread=read(acm,readBuf,sizeof(readBuf)-1);
	    if (nread>0) break;
	    if (elTime.elapsed()>2000) break;
	}

	if (nread>0)
	{
	    readBuf[nread]=0;
	    postMessage(MSGSTYLE_INFO, QObject::tr("Phone answer: %1").arg(readBuf));
	}
	else
	{
	    perror("Unable to connect");
	    postMessage(MSGSTYLE_DEBUG, QObject::tr("nread=%1, errno=%2").arg(nread).arg(errno));
	}
	close(acm);
	sleep(1);
	return 1;
}

// Open phone
int P2kProc::drv_openPhone()
{
	FUNC("openPhone");

	phoneHandle=0;
	phoneHandle = usb_open(phone);
	
	// Motorola E680i bugfix
	usb_close(phoneHandle);
	phoneHandle = usb_open(phone);
	
	//Detach kernel driver. Useful for new motorola phones like Razr V3x/V3c
#ifdef LIBUSB_HAS_DETACH_KERNEL_DRIVER_NP
	if (globalConfig->cfgDetachDriver)
	{
	    char oldDriver[1024];
	    int ret;
	    //Just hack.. will fix this later.
	    for (int i=0; i<20; i++)
	    {
		ret=usb_get_driver_np(phoneHandle,i,oldDriver,sizeof(oldDriver)-1);
		if (ret>=0)
		{
		    //Interface is claimed;
		    postMessage(MSGSTYLE_INFO, QObject::tr("Interface %1 is claimed by %2.").arg(i).arg(oldDriver));
		    ret=usb_detach_kernel_driver_np(phoneHandle, i);
		    if (ret>=0)
	    	    {
			postMessage(MSGSTYLE_INFO, QObject::tr("Driver %1 detached from interface %2").arg(oldDriver).arg(i));
		    }
		    else
		    {
		        postMessage(MSGSTYLE_ERROR,QObject::tr("Unable to detach: %1").arg(usb_strerror()));
		    }	
		}
	    }
	}
#endif	
	if (phoneHandle==0) RAISE("Unable to open phone\n");
	//if (usb_set_configuration(phoneHandle,1)) RAISE("Unable to set configuration");
	//if (usb_claim_interface(phoneHandle, INTERFACE_TEST_COMMAND)) RAISE("Unable to claim the interface");
	return(1);
}

// Close phone handle
int P2kProc::drv_closePhone()
{
	FUNC("closePhone");
	if (phoneHandle<=0) return(-1);
	if (usb_close(phoneHandle)) RAISE("Unable to close phone");
	return(1);
}

// Connect to phone.
int P2kProc::drv_connect()
{
	FUNC("drv_connect");
	int ph=drv_findPhone();
	if (ph==PHONE_NONE)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("No phone found. Check preferences for AT Vendor/Product ID"));
		RAISE("no phone")
	}
	if (ph==PHONE_AT)
	{
		postMessage(MSGSTYLE_INFO, QObject::tr("AT phone found"));
		if (!drv_switchP2K())
			return -1;
	}

	int t;
	t=time(NULL);

	while ((time(NULL)-t<8) && (ph!=PHONE_P2K))
	{
		usb_find_devices();
		ph=drv_findPhone();
		usleep(10000);
	}
	if (ph!=PHONE_P2K) return(-1);
	return(drv_openPhone());
}

// ====================== Debug methods ==============================

// Out buffer content to standart output
void P2kProc::showArr(unsigned char *bytes, int size)
{
#define BYTES_IN_LINE 16

	int i;
	if (size==0) return;

	printf("   -------");
	for (i=0; i<16; i++)
	{
		printf("%3X",i);
	}
	printf("\n");

	for (i=0; i<size; i++)
	{
		if (i % 16 ==0)
		{
			if (i!=0) printf("\n");
			printf("   %06x: ",i);
		}
		printf("%02x ",bytes[i]);
	}
	printf("\n");
}


// ====================== Main USB IO ================================

int P2kProc::get_cmd_size(unsigned char * packetCount)
{
	INT16 cnt;
	INT16 size;
	int bufsize;
	cnt=getInt16(packetCount);
	size=getInt16(packetCount+2);
	bufsize=2*cnt+size+4;
	return(bufsize);
}

// Send control message to device, but with debug loging and error handling
// Return !=0 if error
int P2kProc::sendControl(int dir, usb_dev_handle *dev, int requesttype, int request, int value,
                         int index, char *bytes, int size, int timeout)
{
	FUNC("sendControl");
	if (dev==0) RAISE("no connection");
	int ret;
#ifdef MY_DEBUG
	if (dir==DIR_IN) printf("<==== "); else printf("====> ");
	printf("Control Message\n");
	printf("   requesttype=0x%02x, request=0x%02x, value=0x%02x, index=0x%02x, size=0x%02x (%04d), timeout=%04d\n", requesttype, request, value, index, size, size, timeout);

	if (dir==DIR_OUT) showArr((unsigned char *)bytes, size);
#endif
	
	ret=usb_control_msg(dev, requesttype, request, value, index, bytes, size, timeout);
	lastControlStatus=ret;
	
	if (ret<0)
	{
	    // Some P2k phones sometimes fail. message should be resubmited.
	    postMessage(MSGSTYLE_INFO, QObject::tr("Sending control message failed.. Retry..."));
	    QTime elTime;
	    elTime.start();
	    while (elTime.elapsed()<5000)
	    {
		ret=usb_control_msg(dev, requesttype, request, value, index, bytes, size, timeout);
		lastControlStatus=ret;
		if (ret>=0) break;
		msleep(100);
	    }
	    if (ret>0)
	    {
		postMessage(MSGSTYLE_INFO, QObject::tr("Message sent"));
	    }
	    
	}
		
#ifdef MY_DEBUG
	printf("   result=%04d",ret);
#endif
	if (ret<0) printf("sendControl Error:[%s]\n",usb_strerror());
#ifdef MY_DEBUG
	else printf("\n");
	if ((dir==DIR_IN) && (ret>0)) showArr((unsigned char *)bytes,ret);
	printf("\n");
#endif
	return(ret);
}

// Add header and set data to phone via sendControl
int P2kProc::outData(unsigned char * data, unsigned int size)
{
	/*41 02 00 00 08 00 x1 x1		- Setup packet
	  x2 x2 x3 x3 x4 x4 00 00		- data
	x1 -    
	x2 - ID -  .  p2kman     0.
	x3 - .  restart - 22
	x4 -    .*/

	// Set Packet ID
	freeID++;
	freeID&=0xFFF;
	setInt16(data,freeID);

	return(sendControl(DIR_OUT, phoneHandle, 0x41, 0x02, 0x00, 0x08, (char*) data, size,1000));
}

// Request Packet Count
int P2kProc::inpSize(unsigned char * cmd, int size)
{
	/*  -      -:
	1 00 00 00 08 00 08 00 */
	int t;
	int ret;

	t=time(NULL);

	while (time(NULL)-t<5)
	{
		ret = sendControl(DIR_IN, phoneHandle, 0xc1, 0x00, 0x00, 0x08, (char *)cmd, size, 1000);
		if ((ret<0) || (ret>2)) break;
#ifdef MY_DEBUG
		else printf("Error answer. try again\n");
#endif
		usleep(1000);
	}
	return(ret);

}

// Get data from phone and perform error handling
int P2kProc::inpData(unsigned char * data, unsigned int count, unsigned int size)
{
	FUNC("inpData");
	/*   :
	1 01 x1 x1 08 00 x2 x2
	x1 - - ,     .
	x2 -  . 

	 :
	  0  1  2  3  4  5   6  7  8  9  0  1  2  3  4  5  6  6  7
	    01 00 00 00 00 87  80 03 80 4a 00 7f 00 00 10
	 01 00 00 00 00 09 80  04  80  4a 00 01 00 00 08
	 x1 ?? ?? ?? x2 x2 [x3 x3 x4 x4 x5 x5 ?? ?? yy yy yy yy yy] ...
	x1 - -  .
	x2 -   (   +8)
	x3 - PacketID or 8000h    PacketID      !
	x4 -  ... ( 1 ) 60 - ok,   , 80 - ok,   ,  - .
	x5 -   
	yy -  
	[] -   1 .    -  .
	*/

	int ret=sendControl(DIR_IN , phoneHandle, 0xc1, 0x01, 0x01, 0x08, (char*)data, size, 1000);
	if (ret<0) return (ret);

	int err;
	INT16 x2;
	x2=getInt16(data+4);
	err=0;
	if (err+=(data[0]!=count)) RAISE("E001");
	if (err+=(x2!=size-6)) RAISE("E002");
	if (err) return(-999); else return(ret);
}

int P2kProc::check_packet_header(unsigned char * buf, int bufsize, int adr, char needdata, char checky)
{
	FUNC("CPH");

	INT16 packId;
	INT16 packSize;
	int myid;
	unsigned char packType;
	unsigned char iserr;

	packId=getInt16(buf+adr);
	myid= packId & 0xFFF;

	packType=buf[adr+2];
	packSize=getInt16(buf+adr+4);
	iserr=buf[adr+8];

	if (bufsize<0) RAISE("E00");

	if (myid != freeID) RAISE("E01");
	if (needdata)
	{
		if (packType!=0x80) RAISE("E02a");
	}
	else
	{
		if (packType!=0x60) RAISE("E02b");
	}
	if (checky && iserr) RAISE("E03");

	/*	if ((myid != freeID.wordId) || (packType!=0x80) || (iserr))
		{
			printf("(E_CPH_01)");
			return(-1);
		}
	*/
	return(packSize);
}

// ==================== P2k Mode methods

// Reboot phone
int P2kProc::drv_reboot()
{
	FUNC("drv_reboot");
	//This is reboot p2k command
	unsigned char cmd1[]={0xFF, 0xFF, 0x00, 0x22, 0x00, 0x00, 0x00, 0x00};
	unsigned char cmd2[0x0e];
	if (outData(cmd1, sizeof(cmd1))<0) RAISE("E001");
	if (inpSize(packetCount, sizeof(packetCount))<0) RAISE("E002");
	if (inpData(cmd2, 1, sizeof(cmd2))<0) RAISE("E003");
	return(1);
}

// Suspend phone
int P2kProc::drv_suspend()
{
	FUNC("drv_suspend");
	unsigned char cmd1[]={0xFF, 0xFF, 0x00, 0x36, 0x00, 0x01, 0x00, 0x00,0x00};
	unsigned char * tmp;
	int sz;
	if (outData(cmd1, sizeof(cmd1))<0) RAISE("E01");
	if (inpSize(packetCount, sizeof(packetCount))<0) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	free(tmp);
	return(1);
}

// Get drive information
int P2kProc::drv_getDriveName(unsigned char * buf)
{
	FUNC("getDriveName");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A};
	// 	unsigned char tmp[1024];
	unsigned char * tmp;
	int sz;
	int buf_ps;
	int ps;
	// 0x00 0x4A -  

	outData(cmd, sizeof(cmd));
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E001");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E002");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E003");
	buf_ps=0;
	ps=15;
	while (tmp[ps]!=0)
	{
		if (tmp[ps]==0xfe) tmp[ps]=32;
		buf[buf_ps++]=tmp[ps];
		ps+=2;
	}
	buf[buf_ps++]=0;
	// 	printf("%02X",tmp[ps]);

	free(tmp);
	return(1);
}

int P2kProc::drv_getPhoneModel(unsigned char * buf)
{
	FUNC("getPhoneName");

	unsigned char cmd[0x10]={0xFF, 0xFF, 0x00, 0x20, 0x00, 0x08, 0x00, 0x00, 0x01, 0x17, 0x00, 0x01, 0x00, 0x00, 0x00};
	// 00 02 00 20 00 08 00 00 01 17 00 01 00 00 00 00
	// 	unsigned char tmp[1024];
	unsigned char * tmp;
	int sz;
	int buf_ps;
	int ps;
	// 0x00 0x4A -  

	outData(cmd, sizeof(cmd));
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E001");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E002");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E003");
	buf_ps=0;
	ps=0x10;
	while (tmp[ps]!=0)
	{
		buf[buf_ps++]=tmp[ps];
		ps+=2;
	}
	buf[buf_ps++]=0;
	// 	printf("%02X",tmp[ps]);

	free(tmp);
	return(1);
}

int P2kProc::drv_freeSpace(unsigned char * dev)
{
	FUNC("freeSpace");

	unsigned char cmd[0x208] ={0xFF, 0xFF, 0x00, 0x4A, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0B, 0x00};
	unsigned char * tmp;
	int i;
	int ps;
	int sz;
	int size=strlen((char*)dev);

	INT32 lng;

	ps=13;
	for (i=0; i< size; i++)
	{
		cmd[ps++]=dev[i];
		cmd[ps++]=0;
	}
	cmd[ps++]=0;
	cmd[ps++]=0;

	if (outData(cmd, sizeof(cmd))<=0) RAISE("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E003");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E004");

// 	lng.id[3]=tmp[14]; lng.id[2]=tmp[15]; lng.id[1]=tmp[16]; lng.id[0]=tmp[17];
	lng=getInt32(tmp+14);
	return(lng);

	free(tmp);
}

int P2kProc::drv_fileCount()
{
	FUNC("fileCount");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07};
	unsigned char * tmp;
	int sz;
	INT16 t;

	if (outData(cmd, sizeof(cmd))<=0) RAISE("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE("E003");

	// 	This package has incorrect yy. So we can not check it.
	// 	if (check_packet_header(tmp,sz,6)<0) RAISE("E004");
	t=getInt16(tmp+14);
	free(tmp);
	return(t);
}

int P2kProc::drv_fileList()
{
	FUNC("fileList");

	// Now not need.
	/*	struct filerec
		{
			char name[0x104];
			char b1;
			unsigned char attr;
			char b2;
			unsigned char owner;
			INT32 size;
		};
	*/

	int fCnt;
	unsigned char cmd[] = {0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x05, 0x00, 0x00, 0x00,  0x00, 0x00, 0x08, 0x03};
	unsigned char * tmp;
	unsigned char * recAddr;
	int sz;
	int sz1;
	INT16 rCnt;
	int inRec;
	int i;
	int curidx=0;
	int fnameSize;
	int recSize;

	if (lastCnt>0) fCnt=lastCnt; else
		fCnt=drv_fileCount();
	if (fCnt<0) RAISE("E000");
	sz=-1;

	msg_FileList * msg = new msg_FileList(fCnt);

	while (curidx < fCnt)
	{
		if (outData(cmd, sizeof(cmd))<=0) RAISE ("E001");
		if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
		sz1=get_cmd_size(packetCount);
		if (sz1>sz)
		{
			if (sz!=-1) free(tmp);
			sz=sz1;
			tmp=(unsigned char *) malloc (sz);
		}

		if (inpData(tmp, 0x01, sz1)<0) RAISE ("E003");
		if (check_packet_header(tmp,sz,6)<0) RAISE("E004");

		rCnt=tmp[0x0f];

		inRec=rCnt;
		if ((sz1-0x12) % inRec)
		{
			printf("(BUG) Unsupported filelist answer: sz1=%d, inRec=%d. Please Report!!!", sz1, inRec);
		}
		recSize=(sz1-0x12)/inRec;
		fnameSize=recSize-8;

		recAddr=tmp+0x12;
		for (i=0; i<inRec; i++)
		{
// 			swapLong((INT32 *)(recAddr+fnameSize+4));
			curidx++;
			msg->addItem(curidx, QString((char *) recAddr),getInt32(recAddr+fnameSize+4),
			             recAddr[fnameSize+3], recAddr[fnameSize+1]);
			postProgress(100*(curidx+1)/fCnt);
			recAddr+=recSize;
		}
	}
	postProgress(-1);
	QApplication::postEvent(parent, msg);

	free(tmp);
	return(1);
}

int P2kProc::drv_search(char *request)
{
	FUNC ("drw_search");
	unsigned char cmd[0x108]={0xFF, 0xFF, 0x00, 0x4A, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07};
	unsigned char * tmp;
	int sz;
	INT16 t;

	lastCnt=-1;

	int l;
	int ps=12;
	// Parsing query
	l=strlen(request);
	for (int i=0; i<l; i++)
	{
		if (request[i]=='|')
		{
			cmd[ps++]=0xFF;
			cmd[ps++]=0xFE;
		}
		else
		{
			cmd[ps++]=0x00;
			cmd[ps++]=request[i];
		}
	}
	while (ps<108)
		cmd[ps++]=0x00;

	if (outData(cmd, sizeof(cmd))<=0) RAISE("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE("E003");

	// 	This package has incorrect yy. So we can not check it.
	if (check_packet_header(tmp,sz,6,1,0)<0) RAISE("E004");
	t=getInt16(tmp+14);
	free(tmp);
	lastCnt=t;
	return(t);
}

int P2kProc::drv_getFullFileList()
{
	char drives[256];
	if (drv_getDriveName((unsigned char*)drives)<0)
	{
		postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to get drive list"));
		return 0;
	}
	strcat(drives," ");
	postMessage(MSGSTYLE_INFO, QObject::tr("Found drives: [ %1]").arg(drives));
	char t[256];
	t[0]=0;
	int tp=0;
	int ps=0;
	int cnt;

	while (ps<strlen(drives))
	{
		if (drives[ps]!=' ')
		{
			t[tp++]=drives[ps];
		}
		else
		{
			t[tp]=0x00;
			//postMessage(MSGSTYLE_DEBUG, QString("Drive %1:").arg(t));
			strcat(t,"/|*");
			postMessage(MSGSTYLE_INFO, QObject::tr("Search request: [%1]").arg(t));
			cnt=drv_search(t);
			if (cnt>0)
			{
				postMessage(MSGSTYLE_INFO, QObject::tr("Found %1 files").arg(cnt));
				drv_fileList();
			}
			else
			{
				postMessage(MSGSTYLE_DEBUG, QObject::tr("Unable to execute search request"));
			}
			tp=0;
			t[tp]=0;
		}
		ps++;
	}
	return 1;
}

// ====================== FSAC Methods =========================

int P2kProc::FSAC_Delete(char * fname)
{
	FUNC("FSAC_Delete")
	//                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05};
	unsigned char * buf;
	unsigned char * tmp;
	int bufsize;
	int sz;

	bufsize=sizeof(cmd)+strlen(fname);
	buf=(unsigned char *) malloc (bufsize);
	memcpy(buf,cmd,sizeof(cmd));
	memcpy(buf+sizeof(cmd), fname, strlen(fname));
	setInt16(buf+0x04,bufsize-8);

	if (outData(buf, bufsize)<=0) RAISE ("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);

	tmp=(unsigned char*) malloc (sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE ("E003");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E004");

	free(tmp);
	free(buf);
	return(1);

}

int P2kProc::FSAC_Open(char * fname, unsigned char attr)
{
	FUNC("FSAC_Open");


	//   00 7B 00 4A 00 13 00 00 00 00 00 00 00 00 00 04   .{.J............
	//   2F 61 2F 6F 69 63 71 2E 63 66 67

	/*x2 x2 x3 x3 x4 x4 00 00		- data
	x1 -    
	x2 - ID -  .  p2kman     0.
	x3 - .  restart - 22
	x4 -    .*/

	// 						 0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08
	// 					    0    1     2     3     4     5     6     7       8     9     A     B       C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0xAA};
	int bufsize;
	unsigned char * buf;
	unsigned char * tmp;
	int sz;

	bufsize=sizeof(cmd)+strlen(fname);
	buf=(unsigned char *)malloc(bufsize);
	memcpy(buf,cmd, sizeof(cmd));
	memcpy(buf+sizeof(cmd),fname,strlen(fname));

	setInt16(buf+0x04,bufsize-8);
	buf[0x0F]=attr;

	if (outData(buf, bufsize)<=0) RAISE ("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);

	tmp=(unsigned char*) malloc (sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE ("E003");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E004");

	free(tmp);
	free(buf);
	return(1);
}

int P2kProc::FSAC_Close()
{
	FUNC("FSAC_Close");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04};
	unsigned char * tmp;
	int sz;

	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(1);
}

int P2kProc::FSAC_Seek(unsigned long offset, char dir)
{
	FUNC("FSAC_Seek");

	// 	4  - offset  seek, 5 -   - 0 -  , 1 -   , 2 -  .

	//  					    0    1     2     3     4     5     6     7    8     9     A     B       C     D     E     F
	unsigned char cmd[]={0x00, 0x7C, 0x00, 0x4A, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xAA, 0xAA, 0xAA, 0xAA, 0x00 };
	unsigned char * tmp;
	int sz;

// 	t.longId=offset;
// 	swapLong(&t);

// 	memcpy(cmd+0x0C,&t,4);
	setInt32(cmd+0x0C,offset);
	cmd[0x10]=dir;

	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(1);
}

int P2kProc::FSAC_Read(unsigned char * buf, unsigned short size)
{
	FUNC("FSAC_Read");
	//                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0xAA, 0xAA};
	//00 7D 00 4A 00 08 00 00 00 00 00 01 00 00 00 0C
	int sz;
	unsigned char * tmp;
	int ret;

	if (size>0x400) RAISE("E00");
// 	t.wordId=size;


// 	cmd[0x0e]=t.id[1];
// 	cmd[0x0f]=t.id[0];
	setInt16(cmd+0x0e,size);
	
	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)!=size+0xE) RAISE("E03");
	if ((ret=check_packet_header(tmp,sz,6,1,0))!=size) RAISE("E04");

	memcpy(buf, tmp+0x0E, size);

	free(tmp);
	return(1);
}

int P2kProc::FSAC_Write(unsigned char * tbuf, int size)
{
	FUNC("FSAC_Write");
	//	                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[] ={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xAA, 0xAA};
	unsigned char * buf;
	unsigned char * tmp;
	int bufsize;
	int sz;

	if (size>0x400) RAISE("E00");

	bufsize=size+sizeof(cmd);
	buf=0;
	buf = (unsigned char *) malloc (bufsize);
	memcpy(buf, cmd, sizeof(cmd));
	memcpy(buf+0x10, tbuf, size);

// 	t.wordId=size+8;
// 	buf[0x04]=t.id[1];
// 	buf[0x05]=t.id[0];
	setInt16(buf+0x04,size+8);

// 	t.wordId=size;
// 	buf[0x0E]=t.id[1];
// 	buf[0x0F]=t.id[0];
	setInt16(buf+0x0e,size);

	if (outData(buf, bufsize)<0) RAISE("E01");
	//	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=0;
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");

	free(buf);
	free(tmp);

	return(1);
}

int P2kProc::FSAC_CreateDirectory(char * dirname)
{
	FUNC("FSAC_createDir");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0E,
	                     0x00, 0x00, 0x00, 0x10};
	unsigned char *buf;
	unsigned char *tmp;
	int sz;

	sz=strlen(dirname)+0x10;
	buf=(unsigned char*) malloc(sz);
	memcpy(buf,cmd,sizeof(cmd));
	memcpy(buf+0x10, dirname, strlen(dirname));
	buf[0x4]=(sz-8) / 0x100;
	buf[0x5]=(sz-8) % 0x100;

	if (outData(buf, sz)<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(0);
}

int P2kProc::FSAC_RemoveDirectory(char * dirname)
{
	FUNC("FSAC_removeDir");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F,
	                     0x00, 0x00, 0x00, 0x10};
	unsigned char *buf;
	unsigned char *tmp;
	int sz;

	sz=strlen(dirname)+0x10;
	buf=(unsigned char*) malloc(sz);
	memcpy(buf,cmd,sizeof(cmd));
	memcpy(buf+0x10, dirname, strlen(dirname));
	buf[0x4]=(sz-8) / 0x100;
	buf[0x5]=(sz-8) % 0x100;

	if (outData(buf, sz)<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(0);
}


// =============================================================

int P2kProc::drv_downloadOneFile(char * fn, char attr, unsigned long fsize, unsigned char * tbuf,
                                 unsigned long curSize, unsigned long allSize)
{
	FUNC("downloadOneFile");

#define READ_BUF_SIZE 0x400

	unsigned char * buf;
	unsigned long numreaded;
	unsigned short rec_size;


	// 	FSAC_Close();

	// 	return(1);
	if (FSAC_Open(fn,attr)<0) RAISE("E001");
	if (FSAC_Seek(0x0,0)<0)
	{
		FSAC_Close();
		RAISE("E002");
	}
	numreaded=0; buf=tbuf;
	while (numreaded<fsize)
	{
		if (fsize-numreaded >=READ_BUF_SIZE) rec_size=READ_BUF_SIZE; else
		{
			rec_size=fsize-numreaded;
			// 			printf("Last record.\n");
		}

		// 		printf("!:numread=%d, fsize=%d, rec_size=%d\n",numreaded,fsize,rec_size);

		if (FSAC_Read(buf,rec_size)<0)
		{
			FSAC_Close();
			// 			RAISE("E003");
		}
		buf+=rec_size;
		numreaded+=rec_size;

		postProgress(100*(curSize+numreaded)/allSize);
	}

	if (FSAC_Close()<0) RAISE("E004");
	lastStr=fn;
	return(1);
}

int P2kProc::drv_downloadFiles(P2kFile * lst, int cnt, char * dir)
{
	FUNC("drv_downloadFiles");
	unsigned char * buf;
	unsigned long bufsize;
	unsigned long totalSize;
	unsigned long currentSize;
	int i;
	FILE * fp;
	char fn[1024];
	char * ts;

	if (cnt==0) return(1);

	postProgress(0);
	totalSize=0; currentSize=0;
	for (i=0; i<cnt; i++)
		totalSize+=lst[i].size;

	for (i=0; i< cnt; i++)
	{
		bufsize=lst[i].size;
		buf=(unsigned char *) malloc (bufsize);
		postMessage(MSGSTYLE_NONE, QObject::tr("Downloading: %1 (%2 bytes)").arg(QString::fromLocal8Bit(lst[i].name)).arg(lst[i].size));
		if (drv_downloadOneFile(lst[i].name, lst[i].attr, lst[i].size, buf, currentSize, totalSize)<0) RAISE("E01");
		currentSize+=lst[i].size;
		if (dir[strlen(dir)-1]!='/')
			strcpy(fn,dir);  //dir is already file name
		else
		{
			ts=strrchr(lst[i].name, '/')+1;
			strcpy(fn,dir);
			strcat(fn,ts);
		}
		fp=fopen(fn,"w");
		if (fp==NULL) RAISE("E02");
		if (fwrite(buf, 1, bufsize, fp)!=bufsize) RAISE("E03");
		fclose(fp);
		free(buf);
	}
	postProgress(-1);
	return(1);
}

int P2kProc::drv_uploadOneFile(char * phoneFn, unsigned char attr, unsigned long fsize, unsigned char * tbuf,
                               unsigned long curSize, unsigned long allSize)
{
	FUNC("writeFile");
#define WRITE_BUF_SIZE 0x400

	unsigned char *buf;
	// 	unsigned long bufpos;
	unsigned long numread;
	unsigned long totalread=0;

	if (FSAC_Open(phoneFn,attr)<0) RAISE("E02");

	buf=tbuf;
	// 	bufpos=0;
	numread=WRITE_BUF_SIZE;

	while (numread == WRITE_BUF_SIZE)
	{
		if (totalread+WRITE_BUF_SIZE<fsize)
			numread=WRITE_BUF_SIZE;
		else
			numread=fsize-totalread;
		if (!numread) continue;
		if (FSAC_Write(buf, numread)<0) RAISE("E02");
		buf=buf+numread;
		totalread+=numread;
		postProgress(100*(curSize+totalread)/allSize);
	}

	if (FSAC_Close()<0) RAISE("E03");
	return(1);
}

int P2kProc::drv_uploadFiles(P2kFile * lst, int cnt, const char * dirLocal, const char * dirPhone)
{
	FUNC("drv_uploadFiles");
	if (cnt==0) return(1);

	unsigned char * buf;
	unsigned long bufsize=0;
	unsigned long totalSize=0;
	unsigned long currentSize=0;
	int i;
	FILE * fp;
	char s[1024];
	char s1[1024];


	postProgress(0);

	for  (i=0; i<cnt;i++)
	{
		strcpy(s,dirLocal);
		strcat(s,lst[i].name);
		if ((fp=fopen(s, "r"))==NULL) RAISE("E01");
		fseek(fp, 0L, SEEK_END);
		lst[i].size=ftell(fp);
		totalSize+=lst[i].size;
		fclose(fp);
	}

	for (int i=0; i<cnt; i++)
	{
		bufsize=lst[i].size;
		buf=(unsigned char *) malloc (bufsize);
		strcpy(s,dirLocal);
		strcat(s,lst[i].name);

		strcpy(s1,dirPhone);
		strcat(s1,lst[i].name);

		fp=fopen(s,"r");
		if (fp==NULL) RAISE("E02");
		if (fread(buf, 1, bufsize, fp)!=bufsize) RAISE("E03");
		fclose(fp);

		postMessage(MSGSTYLE_NONE, QObject::tr("Uploading: %1 (%2 bytes)").arg(QString::fromLocal8Bit(lst[i].name)).arg(lst[i].size));
		if (drv_uploadOneFile( s1, lst[i].attr, lst[i].size, buf, currentSize, totalSize)<0) RAISE("E01");
		currentSize+=lst[i].size;
		free(buf);
	}

	postProgress(-1);
	doGetFreeSpace();
	doGetFileCount();
	return(1);
}

int P2kProc::drv_deleteFiles(P2kFile * lst, int cnt)
{
	FUNC("drv_deleteFiles");
	if (cnt==0) return 1;
	postProgress(0);
	for (int i=0; i<cnt; i++)
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Deleting file: %1").arg(QString::fromLocal8Bit(lst[i].name)));
		if (FSAC_Delete(lst[i].name)<0) RAISE("E01");
		lastStr=lst[i].name;
		postProgress(100*(i+1)/cnt);
	}
	postProgress(-1);
	doGetFreeSpace();
	doGetFileCount();
	return 1;
}

int P2kProc::drv_changeAttr(P2kFile * lst, int cnt)
{
	FUNC("drv_changeAttr");
	if (cnt==0) return 1;
	postProgress(0);
	for (int i=0; i<cnt; i++)
	{
		postMessage(MSGSTYLE_NONE, QObject::tr("Changing attributes for: %1").arg(QString::fromLocal8Bit(lst[i].name)));
		if (FSAC_Open(lst[i].name, lst[i].attr)<0) RAISE("E001");
		if (FSAC_Close()<0) RAISE("E002");
		postProgress(100*(i+1)/cnt);
	}
	postProgress(-1);

	return 1;
}

int P2kProc::drv_read_seem(int x, int y, unsigned char * seemBuf, int seemBufSize)
{
	unsigned char cmd1[0x10]={0xFF, 0xFF, 0x00, 0x20, 0x00, 0x08, 0x00, 0x00, 0xAA, 0xAA, 0xBB, 0xBB, 0x00, 0x00, 0x00, 0x00};\
	// 	unsigned char * mem;
	INT16 cnt;
	INT16 size;
	INT16 packId;
	int bufsize;
	unsigned char * buf;
	int adr;
	// 	int dadr;
	unsigned short myid;
	INT16 packSize;
	unsigned char packType;
	int seemPos;
	int j;
	unsigned char iserr;

// 	xxxx.wordId=x;	cmd1[8]=xxxx.id[1]; cmd1[9]=xxxx.id[0];
	setInt16(cmd1+8,x);
	
// 	yyyy.wordId=y;	cmd1[10]=yyyy.id[1]; cmd1[11]=yyyy.id[0];
	setInt16(cmd1+10,y);
	
	outData(cmd1,sizeof(cmd1));
	// 	usleep(100000);
	// 	sleep(1);
	if (inpSize(packetCount, sizeof(packetCount))<=2)
	{
		printf("(E003)");
		return(-1);
	}
	// 	sleep(1);
// 	cnt.id[0]=packetCount[1]; cnt.id[1]=packetCount[0];
	cnt=getInt16(packetCount);
	
// 	size.id[0]=packetCount[3]; size.id[1]=packetCount[2];
	size=getInt16(packetCount+2);
	
	bufsize=2*cnt+size+4;
	buf = (unsigned char *) malloc (bufsize);
	if (inpData(buf, cnt, bufsize)!=bufsize)
	{
		printf("(E004)");
		return(-1);
	}

	adr=6;
	seemPos=0;

	for (int i=0; i<cnt; i++)
	{
// 		packId.id[0]=buf[adr+1];
// 		packId.id[1]=buf[adr];
		packId=getInt16(buf+adr);
		
		
		myid= packId & 0xFFF;
		packType=buf[adr+2];
// 		packSize.id[0]=buf[adr+5];
// 		packSize.id[1]=buf[adr+4];
		packSize=getInt16(buf+adr+4);
		iserr=buf[adr+8];

		if ((myid != freeID) || (packType!=0x80) || (iserr))
		{
			if (packType==0x60) adr+=7;	else adr+=8+packSize;
			// 			printf("Packet Error. myid=0x%04X, needID=0x%04X, packID=0x%04X packType=0x%02X, yy=0x%02X\n",myid, freeID.wordId, packId, packType, iserr);
			printf("(E01)");
			return(-1);
		}
		for (j=0; j<packSize-1; j++)
		{
			if (seemPos>=seemBufSize)
			{
				printf("(E_RS_01_%04d_%04d\n",seemBufSize,packSize-1);
				return(-1);
			}
			seemBuf[seemPos++]=buf[adr+9+j];
		}
		break;
	}
	return(seemPos);
}

int P2kProc::drv_write_seem(int x, int y, unsigned char * seemBuf, int seemSize)
{
#define MAX_SEEM_SIZE 102400

	unsigned char * temp_seem;
	unsigned char * temp_packet;

	temp_packet=(unsigned char *) malloc (MAX_SEEM_SIZE);
	temp_seem=temp_packet+16;

	int actualSize;
	int ret;

	// Check seem size by re-reading it from phone

	// 	if (!isFirstRead)
	// 	{
	actualSize=drv_read_seem(x,y, temp_seem, MAX_SEEM_SIZE-16);
	actualSize=seemSize;

	if (actualSize != seemSize)
	{
		printf("(E_WS_01_%04d)",actualSize);
		free(temp_packet);
		return(-1);
	}

// 	INT16 wrd;

	memcpy(temp_seem,seemBuf,seemSize);

// 	wrd.wordId=seemSize+8;

	//	 0	1  2  3  4  5  6  7  8  9 10 11 12 13 14 15
	//  00 05 00 2F 00 88 00 00 02 06 00 01 00 00 00 80
	//	FF FF 00 2F AA AA 00 00 XX XX YY YY 00 00 ZZ ZZ


	/*	FF FF - ID . AUTO
		00 2F -  seem
		AA AA -  
		XX XX - Seem XXXX
		YY YY - Seem YYYY
		ZZ ZZ - Seem size
	*/

	temp_packet[2]=0x00; temp_packet[3]=0x2F;
// 	wrd.wordId=seemSize+8; temp_packet[4]=wrd.id[1]; temp_packet[5]=wrd.id[0];
	setInt16(temp_packet+4,seemSize+8);
	
	temp_packet[6]=0; temp_packet[7]=0;
// 	wrd.wordId=x; temp_packet[8]=wrd.id[1]; temp_packet[9]=wrd.id[0];
	setInt16(temp_packet+8,x);
// 	wrd.wordId=y; temp_packet[10]=wrd.id[1]; temp_packet[11]=wrd.id[0];
	setInt16(temp_packet+10,y);	
	
	temp_packet[12]=0; temp_packet[13]=0;
// 	wrd.wordId=seemSize; temp_packet[14]=wrd.id[1]; temp_packet[15]=wrd.id[0];
	setInt16(temp_packet+14,seemSize);

	ret=outData(temp_packet,seemSize+16);
	if (ret!=seemSize+16)
	{
		printf("(E_WS_02)");
		return(-1);
	}

	// 	usleep(100000);

	if (inpSize(packetCount, sizeof(packetCount))<=2)
	{
		printf("(E_WS_03)");
		return(-1);
	}

	INT16 pcnt;
	INT16 psize;
	int bufsize;


// 	pcnt.id[0]=packetCount[1]; pcnt.id[1]=packetCount[0];
	pcnt=getInt16(packetCount);
	
// 	psize.id[0]=packetCount[3]; psize.id[1]=packetCount[2];
	psize=getInt16(packetCount+2);
	bufsize=2*pcnt+psize+4;

	if (inpData(temp_packet, pcnt, bufsize)!=bufsize)
	{
		printf("(E_WS_04)");
		return(-1);
	}

	// 	unsigned char * buf, int bufsize, int adr, char needdata=1, char checky=1
	ret=check_packet_header(temp_packet, pcnt, 6);
	if (ret<0)
	{
		printf("(E_WS_05)");
		return(-1);
	}
	free(temp_packet);
	return(1);
}

int P2kProc::drv_backup(int x1, int y1, int x2, int y2, unsigned char * dir)
{

	qDebug("drv_backup");

	int x;
	int y;
	unsigned char buf[102400];
	int ret;
	FILE * fp;
	char s[256];
	// 	int idx=0;
	int all=(x2-x1);
	if (all==0) all=1;
	x=x1;

	postProgress(0);


	while (x<=x2)
	{
		if(x==x1) y=y1; else y=0x01;

		while (1)
		{
			ret=drv_read_seem(x,y,buf,sizeof(buf));

			if (ret<0)
			{
				if (lastControlStatus>=0) break;
				if (lastControlStatus==-110)
				{
					postMessage(MSGSTYLE_NONE, QObject::tr("Phone reboot detected. Try to restore (Timeout:30 sec)"));
					setState(CONNECT_NONE);
					sleep(2);
					int tm=time(NULL);
					int ok=0;
					while (time(NULL)-tm<30)
					{
						doUpdateMode();
						if (drv_connect()>0)
						{
							ok=1;
							break;
						}
						else
							sleep(1);
					}
					doUpdateMode();
					if (!ok)
					{
						postMessage(MSGSTYLE_ERROR, QObject::tr("Unable to restore connection"));
						return(-1);
					}
					else
					{
						postMessage(MSGSTYLE_NONE, QObject::tr("Connection restored"));
						setState(CONNECT_P2K);
					}

					x=x-11; if (x<x1) x=x1-1;
					break;
				}

				postMessage(MSGSTYLE_NONE, QObject::tr("Unknown Error. USB IO code=%1").arg(lastControlStatus));
				return(-1);

			}

			sprintf(s,"%s%04x_%04x.seem",dir,x,y);
			fp=fopen(s,"w+");
			if (fp==NULL)
			{
				postMessage(MSGSTYLE_NONE, QObject::tr("Unable to open file [%1]").arg(s));
				return(-1);
			}

			fwrite(buf,1, ret, fp);
			fclose(fp); fp=NULL;
			y++;

			if ((x==x2) && (y==y2)) break;
			if (y>0xff) break;
		}
		x++;
		postProgress(100*(x-x1)/all);
	}
	postProgress(-1);
	return(1);
}
//
