// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License (Ms-PL).
// Please see http://go.microsoft.com/fwlink/?LinkID=131993 for details.
// All other rights reserved.

using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Reflection;
using System.Windows.Controls.Data.Test.DataClassSources;
using System.Windows.Controls.Test;
using System.Windows.Media;
using Microsoft.Silverlight.Testing;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Mono.Moonlight.UnitTesting;

namespace System.Windows.Controls.Data.Test
{
    /// <summary>
    /// Tests that should have no restrictions on the data source.
    /// </summary>
    public partial class DataGridUnitTest_Unrestricted<TDataClass, TDataClassSource> : DataGridUnitTest<TDataClass>
        where TDataClass : new()
        where TDataClassSource : DataClassSource<TDataClass>, new()
    {
        [TestMethod]
        [Description("Create a DataGrid control.")]
        public virtual void CreateDataGrid()
        {
            DataGrid dg = new DataGrid();
            Assert.IsNotNull(dg);
        }

        [TestMethod]
        [Asynchronous]
        [Description("Check Default Values.")]
        public virtual void DefaultValues()
        {
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            TestPanel.Children.Add(dataGrid);
            this.EnqueueYieldThread();

            this.EnqueueCallback(delegate
            {
                //Verify Default Grid Values
                Assert.AreEqual(true, dataGrid.CanUserResizeColumns, "CanUserResizeColumns wrong default");
                Assert.AreEqual(false, dataGrid.IsReadOnly, "IsReadOnly wrong default");
                Assert.AreEqual(DataGridLength.Auto, dataGrid.ColumnWidth, "ColumnWidth wrong default");
                Assert.IsTrue(double.IsNaN (dataGrid.RowHeight), "RowHeight wrong default");
                Assert.IsTrue(double.IsNaN (dataGrid.ColumnHeaderHeight), "ColumnHeaderHeight wrong default");
                Assert.IsTrue(double.IsNaN (dataGrid.RowHeaderWidth), "RowHeaderWidth wrong default");
                Assert.AreEqual(null, dataGrid.SelectedItem, "SelectedItem wrong default");
                Assert.AreEqual(((IList<TDataClass>)(new List<TDataClass>())).Count, dataGrid.SelectedItems.Count, "SelectedItems wrong default");
                Assert.AreEqual(DataGridSelectionMode.Extended, dataGrid.SelectionMode, "SelectionMode wrong default");
                Assert.AreEqual(null, dataGrid.CurrentColumn, "CurrentColumn wrong default");
                Assert.AreEqual(false, dataGrid.AreRowDetailsFrozen, "AreRowDetailsFrozen wrong default");
                Assert.AreEqual(true, dataGrid.AutoGenerateColumns, "AutoGenerateColumns wrong default");
                Assert.AreEqual(DataGridRowDetailsVisibilityMode.VisibleWhenSelected, dataGrid.RowDetailsVisibilityMode, "RowDetailsVisibilityMode wrong default");
            });
            EnqueueTestComplete();
        }

        #region AutoGenenerate
        [TestMethod]
        [Asynchronous]
        [Description("Test autogenerated columns.")]
        public virtual void AutogeneratedColumns()
        {
            IEnumerable emptyNonGeneric = new NonGenericEnumerable_0<TDataClass>();
            IEnumerable nonEmptyNonGeneric = new NonGenericEnumerable_1<TDataClass>();
            IEnumerable<TDataClass> emptyList = new GenericEnumerable_0<TDataClass>();

            IList emptyNonGenericList = new NonGenericList_0<TDataClass>();

            IList nonEmptyNonGenericList = new NonGenericList_1<TDataClass>();

            IList emptyNonGenericListINCC = new NonGenericListINCC_0<TDataClass>();

            IEnumerable<TDataClass> nonEmptyGeneric = new GenericEnumerable_25<TDataClass>();
            DataGrid dg = new DataGrid();
            Assert.IsNotNull(dg);
            TestPanel.Children.Add(dg);
            int i;

            // empty non-generics don't trigger AGC
            dg.ItemsSource = emptyNonGeneric;
            this.EnqueueYieldThread();
            this.EnqueueCallback(delegate
            {
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after assigning an empty non-generic sequence");

                // non-empty non-generics do
                dg.ItemsSource = nonEmptyNonGeneric;
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after assigning a non-empty, non-generic sequence");

                // in order to get autogeneration to happen as a result of adding an item to the list, the list needs to be INCC
                dg.ItemsSource = emptyNonGenericList;
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after assigning an empty non-generic list");

                emptyNonGenericList.Add(new TDataClass());
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after adding an item to an empty, non-generic list");

                // test column count after removing from a non-empty, non-INCC
                dg.ItemsSource = nonEmptyNonGenericList;
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after assigning a non-empty, non-generic list");
                Assert.AreEqual(1, dg.RowCount, "Wrong number of rows in datagrid after assigning a non-empty, non-generic list.");

                // we're non-INCC, so the grid won't be seeing this remove
                nonEmptyNonGenericList.RemoveAt(0);
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after removing item from a non-empty, non-generic list");
                Assert.AreEqual(1, dg.RowCount, "Wrong number of rows in datagrid after removing item from a non-empty, non-generic list.");

                // an INCC data source
                dg.ItemsSource = emptyNonGenericListINCC;
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after assigning an empty non-generic INCC list");

                // ... adding an item triggers AGC
                emptyNonGenericListINCC.Add(new TDataClass());
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after adding an item to an empty, non-generic INCC list");
                Assert.AreEqual(1, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(emptyNonGenericListINCC.Count, dg.RowCount, "Row count does not match item count");

                // ... removing an item doesn't clear the columns
                emptyNonGenericListINCC.RemoveAt(0);
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after removing the item from an empty, non-generic INCC list");
                Assert.AreEqual(0, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(emptyNonGenericListINCC.Count, dg.RowCount, "Row count does not match item count");

                // ... clearing the columns manually does
                dg.Columns.Clear();
                Assert.AreEqual(0, dg.Columns.Count, "Wrong column count after clearing all columns");
                Assert.AreEqual(0, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(emptyNonGenericListINCC.Count, dg.RowCount, "Row count does not match item count");

                // ... adding a new (different type) item to an ItemsSource for which columns have been generated regenerates the autogenerated columns
                // ... also, the row doesn't show since the columns don't exist
                emptyNonGenericListINCC.Add(this.GetType());
                Assert.AreEqual(16, dg.Columns.Count, "Wrong column count after adding this an empty, non-generic INCC list which has already had columns generated and cleared");
                Assert.AreEqual(1, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(1, emptyNonGenericListINCC.Count, "Wrong list count");

                // ... setting AutogenerateColumns to false clears the AutogeneratedColumns
                dg.AutoGenerateColumns = false;
                Assert.AreEqual(0, dg.Columns.Count, "Wrong column count after adding this an empty, non-generic INCC list which has already had columns generated and cleared");
                Assert.AreEqual(0, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(1, emptyNonGenericListINCC.Count, "Wrong list count");

                // ... setting AutogenerateColumnsBack to true regenerates the columns
                dg.AutoGenerateColumns = true;
                Assert.AreEqual(16, dg.Columns.Count, "Wrong column count after adding this an empty, non-generic INCC list which has already had columns generated and cleared");
                Assert.AreEqual(1, dg.RowCount, "Wrong number of rows in datagrid.");
                Assert.AreEqual(1, emptyNonGenericListINCC.Count, "Wrong list count");

                // clear the dg
                dg.ItemsSource = emptyNonGeneric;
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after assigning an empty non-generic sequence");

                // strongly-typed enumerables trigger AGC
                dg.ItemsSource = emptyList;
                i = 0;
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after assigning an empty generic sequence");
                foreach (PropertyInfo prop in properties)
                {
                    DataGridColumn col = dg.Columns[i++];
                    Assert.AreEqual(prop.Name, col.Header);

                    PropertyTestExpectedResultsAttribute expectedResults = PropertyTestExpectedResultsAttribute.GetExpectedResults(prop, "AutogeneratedColumns");

                    Assert.AreEqual(expectedResults.IsReadOnly, col.IsReadOnly, "IsReadOnly wrong value: " + prop.Name);
                }

                // clear the dg
                dg.ItemsSource = emptyNonGeneric;
                Assert.AreEqual(0, dg.Columns.Count, "Non-zero column count after assigning an empty non-generic sequence");

                // another strongly-typed enumerable
                dg.ItemsSource = nonEmptyGeneric;
                i = 0;
                Assert.AreEqual(numberOfProperties, dg.Columns.Count, "Wrong column count after assigning a non-empty generic sequence");
                foreach (PropertyInfo prop in properties)
                {
                    DataGridColumn col = dg.Columns[i++];
                    Assert.AreEqual(prop.Name, col.Header);

                    PropertyTestExpectedResultsAttribute expectedResults = PropertyTestExpectedResultsAttribute.GetExpectedResults(prop, "AutogeneratedColumns");

                    Assert.AreEqual(expectedResults.IsReadOnly, col.IsReadOnly, "IsReadOnly wrong value: " + prop.Name);
                }
            });
            EnqueueTestComplete();
        }

        #endregion

        #region CheckColumnHeaderVisibility
        [TestMethod]
        [Asynchronous]
        [Description("Test ColumnHeaderVisibilty for various ItemsSource conditions")]
        public virtual void CheckColumnHeaderVisibility()
        {
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            ObservableCollection<Person> people = new ObservableCollection<Person>();
            dataGrid.ItemsSource = people;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.Columns.Count > 0, "Columns were not generated for IEnumerable<T>");
                // Make sure the Headers are visible since we have columns even though the list is empty
                Assert.IsTrue(dataGrid.ColumnHeaders.Visibility == Visibility.Visible, "ColumnHeaders should be visible for an empty list");
                people.Add(new Person { Name = "Square guy" });
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.ColumnHeaders.Visibility == Visibility.Visible, "ColumnHeaders should be visible for a non-empty list");
                people.RemoveAt(0);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.ColumnHeaders.Visibility == Visibility.Visible, "ColumnHeaders should be visible even after deleting all items");
                people.Clear();
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.ColumnHeaders.Visibility == Visibility.Visible, "ColumnHeaders should be visible even after Clear (NotifyCollectionChangedAction.Reset)");
                dataGrid.ItemsSource = null;
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.Columns.Count == 0, "Columns were not cleared when ItemsSource was set to null");
            });
            EnqueueTestComplete();
        }
        #endregion CheckColumnHeaderVisibility

        #region CheckLoadRow
        List<TDataClass> gridList = new List<TDataClass>();
        int gridListItems = 0;

        [Asynchronous]
        [TestMethod]
        [Description("Test that all data in the list is also in the grid.")]
        public virtual void CheckLoadRow()
        {
            gridList.Clear();
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            dataGrid.Width = 300;
            dataGrid.Height = 250;
            Assert.IsNotNull(dataGrid);
            dataGrid.LoadingRow += new EventHandler<DataGridRowEventArgs>(dataGrid_LoadingRow);
            TestPanel.Children.Add(dataGrid);
            dataGrid.ItemsSource = boundList;

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                foreach (TDataClass st in boundList)
                {
                    dataGrid.ScrollIntoView(st, null);
                }
            });
            
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(boundList.Count() <= gridList.Count);

                foreach (TDataClass st in boundList)
                {
                    Assert.IsTrue(gridList.Contains(st));
                }
            });

            EnqueueTestComplete();
        }

        void dataGrid_LoadingRow(object sender, DataGridRowEventArgs e)
        {
            gridListItems++;
            gridList.Add((TDataClass)e.Row.DataContext);
        }
        #endregion

        [TestMethod]
        [Asynchronous]
        [Description("Test adding items to underlying collection and make sure grid gets updated.")]
        public virtual void AddItemsToUnderlyingData()
        {
            IEnumerable listSomeType = new TDataClassSource();
            int sizeOfList = listSomeType.Count();
            DataGrid dataGrid1 = new DataGrid();
            dataGrid1.ItemsSource = listSomeType;
            TestPanel.Children.Add(dataGrid1);
            this.EnqueueYieldThread();

            //List should have sizeOfList items in it.
            int countOfItems = 0;
            EnqueueCallback(delegate
            {
                foreach (TDataClass i in dataGrid1.ItemsSource)
                {
                    countOfItems++;
                    //This will fail if the item isn't in the grid.
                    dataGrid1.SelectedItems.Add(i);
                }
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.AreEqual(sizeOfList, countOfItems);
                Assert.AreEqual(sizeOfList, dataGrid1.SelectedItems.Count,
                    String.Format("Should be {0} items in SelectedItems", sizeOfList));
                //Can only add items with IList
                if (listSomeType is IList)
                {
                    TDataClass newItem = new TDataClass();
                    ((IList)listSomeType).Add(newItem);

                    if (listSomeType is INotifyCollectionChanged)
                    {
                        dataGrid1.SelectedItems.Add(newItem);
                        Assert.AreEqual(sizeOfList + 1, dataGrid1.SelectedItems.Count,
                            String.Format("Should be {0} items in SelectedItems", sizeOfList + 1));


                        //
                    }
                }
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the HorizontalScrollBarVisibility property after the DataGrid was populated with rows.")]
        public virtual void SetHorizontalScrollBarVisibility()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Make sure default value is ScrollBarVisibility.Auto
                Assert.AreEqual(ScrollBarVisibility.Auto, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Disabled
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Disabled;
                Assert.AreEqual(ScrollBarVisibility.Disabled, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Visible
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Visible;
                Assert.AreEqual(ScrollBarVisibility.Visible, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Hidden
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                Assert.AreEqual(ScrollBarVisibility.Hidden, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Auto
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Auto;
                Assert.AreEqual(ScrollBarVisibility.Auto, dataGrid.HorizontalScrollBarVisibility);
            });
            EnqueueCallback(delegate
            {
                // Make the vertical scrollbar appear too
                boundList = new TDataClassSource();
                dataGrid.ItemsSource = boundList;
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Disabled
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Disabled;
                Assert.AreEqual(ScrollBarVisibility.Disabled, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Visible
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Visible;
                Assert.AreEqual(ScrollBarVisibility.Visible, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Hidden
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                Assert.AreEqual(ScrollBarVisibility.Hidden, dataGrid.HorizontalScrollBarVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change value to ScrollBarVisibility.Auto
                dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Auto;
                Assert.AreEqual(ScrollBarVisibility.Auto, dataGrid.HorizontalScrollBarVisibility);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the HorizontalGridLinesBrush property after the DataGrid was populated with rows.")]
        public virtual void SetHorizontalGridLinesBrush()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Make sure default GridLinesVisibility value is DataGridGridLinesDataGridGridLinesVisibility.Vertical
                Assert.AreEqual(DataGridGridLinesVisibility.Vertical, dataGrid.GridLinesVisibility);

                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.All;

                // Change HorizontalGridLinesBrush value to SolidColorBrush
                SolidColorBrush blueBrush = new SolidColorBrush(Colors.Blue);
                dataGrid.HorizontalGridLinesBrush = blueBrush;
                Assert.AreEqual(blueBrush, dataGrid.HorizontalGridLinesBrush);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility value to DataGridGridLinesVisibility.None
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.None;
                Assert.AreEqual(DataGridGridLinesVisibility.None, dataGrid.GridLinesVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility to DataGridGridLinesVisibility.Horizontal
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.Horizontal;
                Assert.AreEqual(DataGridGridLinesVisibility.Horizontal, dataGrid.GridLinesVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Reset HorizontalGridLinesBrush value
                dataGrid.HorizontalGridLinesBrush = null;
                Assert.IsNull(dataGrid.HorizontalGridLinesBrush);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility to DataGridGridLinesVisibility.All
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.All;
                Assert.AreEqual(DataGridGridLinesVisibility.All, dataGrid.GridLinesVisibility);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the VerticalGridLinesBrush property after the DataGrid was populated with rows.")]
        public virtual void SetVerticalGridLinesBrush()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Make sure default GridLinesVisibility value is DataGridGridLinesVisibility.Vertical
                Assert.AreEqual(DataGridGridLinesVisibility.Vertical, dataGrid.GridLinesVisibility);

                // Change VerticalGridLinesBrush value to SolidColorBrush
                SolidColorBrush blueBrush = new SolidColorBrush(Colors.Blue);
                dataGrid.VerticalGridLinesBrush = blueBrush;
                Assert.AreEqual(blueBrush, dataGrid.VerticalGridLinesBrush);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility value to DataGridGridLinesVisibility.None
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.None;
                Assert.AreEqual(DataGridGridLinesVisibility.None, dataGrid.GridLinesVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility to DataGridGridLinesVisibility.Vertical
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.Vertical;
                Assert.AreEqual(DataGridGridLinesVisibility.Vertical, dataGrid.GridLinesVisibility);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Reset VerticalGridLinesBrush value
                dataGrid.VerticalGridLinesBrush = null;
                Assert.IsNull(dataGrid.VerticalGridLinesBrush);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Change GridLinesVisibility to DataGridGridLinesVisibility.All
                dataGrid.GridLinesVisibility = DataGridGridLinesVisibility.All;
                Assert.AreEqual(DataGridGridLinesVisibility.All, dataGrid.GridLinesVisibility);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the DataGridRowHeader's SeparatorBrush / SeparatorVisibility properties while populating the DataGrid.")]
        public virtual void SetRowHeaderSeparator()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);

            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                for (int i = dataGrid.DisplayData.FirstDisplayedScrollingRow; i <= dataGrid.DisplayData.LastDisplayedScrollingRow; i++)
                {
                    DataGridRow row = dataGrid.DisplayData.GetDisplayedRow(i);
                    row.HeaderCell.SeparatorBrush = new SolidColorBrush(Colors.Cyan);
                    if (row.Index % 2 == 0)
                    {
                        row.HeaderCell.SeparatorVisibility = Visibility.Collapsed;
                    }
                }
            });

            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the DataGridColumnHeader's SeparatorBrush / SeparatorVisibility properties while populating the DataGrid.")]
        public virtual void SetColumnHeaderSeparator()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.RowHeight = 18;
            dataGrid.ColumnWidth = new DataGridLength(30);
            dataGrid.Width = 350;
            dataGrid.Height = 350;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.LoadingRow += new EventHandler<DataGridRowEventArgs>(DataGrid_LoadingRow_ColumnHeaderSeparatorBrush);
            dataGrid.AutoGenerateColumns = true;
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);

            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
                dataGrid.LoadingRow -= new EventHandler<DataGridRowEventArgs>(DataGrid_LoadingRow_ColumnHeaderSeparatorBrush);
            });
            EnqueueTestComplete();
        }

        private void DataGrid_LoadingRow_ColumnHeaderSeparatorBrush(object sender, DataGridRowEventArgs e)
        {
            foreach (DataGridColumn dataGridColumn in e.Row.OwningGrid.Columns)
            {
                if (e.Row.Index % 3 == 0)
                {
                    dataGridColumn.HeaderCell.SeparatorBrush = new SolidColorBrush(Colors.Green);
                }
                else if (e.Row.Index % 5 == 0)
                {
                    dataGridColumn.HeaderCell.SeparatorBrush = null;
                }
                if (dataGridColumn.DisplayIndex % 2 == (e.Row.Index % 2))
                {
                    dataGridColumn.HeaderCell.SeparatorVisibility = Visibility.Collapsed;
                }
                else
                {
                    dataGridColumn.HeaderCell.SeparatorVisibility = Visibility.Visible;
                }
            }
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the FrozenColumnCount property to freeze and unfreeze columns.")]
        public virtual void SetFrozenColumnCount()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.ColumnWidth = new DataGridLength(100);
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.AutoGenerateColumns = true;
            TestPanel.Children.Add(dataGrid);
            dataGrid.ItemsSource = boundList;

            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            if (dataGrid.Columns.Count > 2)
            {
                EnqueueCallback(delegate
                {
                    dataGrid.HorizontalOffset = 20;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(false, dataGrid.Columns[0].IsFrozen);
                    Assert.AreEqual(0, dataGrid.DisplayData.FirstDisplayedScrollingCol);
                    dataGrid.FrozenColumnCount = 2;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    // Horizontal Offset should be set back to 0 because we froze the first column
                    Assert.AreEqual(0, dataGrid.HorizontalOffset);
                    Assert.AreEqual(true, dataGrid.Columns[0].IsFrozen);
                    Assert.AreEqual(true, dataGrid.Columns[1].IsFrozen);
                    Assert.AreEqual(false, dataGrid.Columns[2].IsFrozen);
                    Common.AssertExpectedException(DataGridError.DataGrid.CannotMoveFrozenColumn(),
                        delegate
                        {
                            dataGrid.Columns[0].DisplayIndex = 2;
                        });
                    Common.AssertExpectedException(DataGridError.DataGrid.CannotMoveNonFrozenColumn(),
                        delegate
                        {
                            dataGrid.Columns[2].DisplayIndex = 0;
                        });
                    Assert.AreEqual(2, dataGrid.DisplayData.FirstDisplayedScrollingCol);

                    // Reorder within frozen columns
                    dataGrid.Columns[0].DisplayIndex = 1;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(1, dataGrid.Columns[0].DisplayIndex);

                    dataGrid.FrozenColumnCount = 0;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(false, dataGrid.Columns[0].IsFrozen);
                });
            }

            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that exercises the HeadersVisibility property to show/hide both column and row headers.")]
        public virtual void SetHeadersVisibility()
        {
            IEnumerable listSomeType = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 350;
            dataGrid.ColumnHeaderHeight = 25;
            dataGrid.RowHeaderWidth = 50;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.AutoGenerateColumns = true;
            dataGrid.HeadersVisibility = DataGridHeadersVisibility.None;
            dataGrid.ItemsSource = listSomeType;
            TestPanel.Children.Add(dataGrid);

            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();
            IList list = listSomeType as IList;
            if (list != null && list is INotifyCollectionChanged)
            {
                EnqueueCallback(delegate
                {
                    list.Add(new TDataClass());
                    list.Insert(1, new TDataClass());
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.All;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(25, dataGrid.ColumnHeaders.DesiredSize.Height);
                    Assert.AreEqual(50, dataGrid.ActualRowHeaderWidth);
                    list.Add(new TDataClass());
                    list.Insert(1, new TDataClass());
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.None;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(false, dataGrid.AreRowHeadersVisible);
                    list.RemoveAt(3);
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.Row;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(50, dataGrid.ActualRowHeaderWidth);
                    list.RemoveAt(0);
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.None;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(false, dataGrid.AreRowHeadersVisible);
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.Row;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(50, dataGrid.ActualRowHeaderWidth);
                    list.Add(new TDataClass());
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.Column;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(25, dataGrid.ColumnHeaders.DesiredSize.Height);
                    Assert.AreEqual(false, dataGrid.AreRowHeadersVisible);
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.None;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(false, dataGrid.AreRowHeadersVisible);
                    list.Add(new TDataClass());
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.Column;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(25, dataGrid.ColumnHeaders.DesiredSize.Height);
                    Assert.AreEqual(false, dataGrid.AreRowHeadersVisible);
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.All;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(25, dataGrid.ColumnHeaders.DesiredSize.Height);
                    Assert.AreEqual(50, dataGrid.ActualRowHeaderWidth);
                    Assert.AreEqual(true, dataGrid.AreRowHeadersVisible);
                    list.Add(new TDataClass());
                    dataGrid.HeadersVisibility = DataGridHeadersVisibility.Row;
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(Visibility.Collapsed, dataGrid.ColumnHeaders.Visibility);
                    Assert.AreEqual(50, dataGrid.ActualRowHeaderWidth);
                    Assert.AreEqual(true, dataGrid.AreRowHeadersVisible);
                    list.Clear();
                });
            }
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test scrolling rows into view.")]
        public virtual void ScrollRowsIntoView()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.AutoGenerateColumns = true;
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 99);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 0);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                dataGrid.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 99);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 0);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                dataGrid.VerticalScrollBarVisibility = ScrollBarVisibility.Visible;
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 25);
            });
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                dataGrid.UpdateStateOnMouseLeftButtonDown(null, 0, 5);
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test changing column display index.")]
        public virtual void SetColumnDisplayIndex()
        {
            IEnumerable boundList = new TDataClassSource();
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.Width = 350;
            dataGrid.Height = 250;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            dataGrid.AutoGenerateColumns = true;
            dataGrid.ItemsSource = boundList;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Columns[2].DisplayIndex = 1;
                    Assert.AreEqual(0, dataGrid.Columns[0].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[2].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[1].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Columns[0].DisplayIndex = 2;
                    Assert.AreEqual(0, dataGrid.Columns[2].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[1].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[0].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Columns[0].DisplayIndex = 0;
                    dataGrid.Columns[1].DisplayIndex = 1;
                    Assert.AreEqual(0, dataGrid.Columns[0].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[1].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[2].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Focus();
                    dataGrid.BeginEdit();
                    dataGrid.Columns[2].DisplayIndex = 1;
                    Assert.AreEqual(0, dataGrid.Columns[0].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[2].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[1].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Focus();
                    dataGrid.BeginEdit();
                    dataGrid.Columns[0].DisplayIndex = 2;
                    Assert.AreEqual(0, dataGrid.Columns[2].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[1].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[0].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                if (dataGrid.Columns.Count >= 3)
                {
                    dataGrid.Focus();
                    dataGrid.BeginEdit();
                    dataGrid.Columns[0].DisplayIndex = 0;
                    dataGrid.Columns[1].DisplayIndex = 1;
                    Assert.AreEqual(0, dataGrid.Columns[0].DisplayIndex);
                    Assert.AreEqual(1, dataGrid.Columns[1].DisplayIndex);
                    Assert.AreEqual(2, dataGrid.Columns[2].DisplayIndex);
                }
            });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                //Reset datagrid
                dataGrid.Loaded -= new RoutedEventHandler(dataGrid_Loaded);
            });
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that checks scrollbar visibility and the filler column width")]
        [MoonlightBug]
        public virtual void ScrollBarVisibilityTest()
        {
            DataGrid dataGrid = new DataGrid();
            isLoaded = false;
            dataGrid.Width = 150;
            dataGrid.ColumnHeaderHeight = 25;
            dataGrid.RowHeight = 50;
            dataGrid.HeadersVisibility = DataGridHeadersVisibility.All;
            dataGrid.RowHeaderWidth = 25;
            dataGrid.Height = 150;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            ObservableCollection<Person> people = new ObservableCollection<Person>();
            people.Add(new Person { Name = "Person 1" });
            dataGrid.ItemsSource = people;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();

            double fillerWidth = 0;
            DataGridFillerColumn fillerColumn = null;
            DataGridCell fillerCell = null;
            EnqueueCallback(delegate
            {
                // Check ScrollBar visibility
                Assert.IsNotNull(dataGrid.HorizontalScrollBar);
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Collapsed, "Horizontal ScrollBar should be Collapsed");
                Assert.IsNotNull(dataGrid.VerticalScrollBar);
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");

                // Check the filler column
                fillerColumn = dataGrid.ColumnsInternal.FillerColumn;
                Assert.IsNotNull(fillerColumn);
                fillerWidth = fillerColumn.Width;
                Assert.IsTrue(fillerColumn.IsActive, "FillerColumn should be active");
                Assert.IsTrue(fillerWidth > 0);

                // Check the width of a cell in the filler column
                DataGridRow row = dataGrid.GetRowFromItem(people[0]) as DataGridRow;
                Assert.IsNotNull(row);
                fillerCell = row.FillerCell;
                Assert.IsNotNull(fillerCell);
                Assert.IsTrue(fillerCell.ActualWidth == fillerWidth, "Filler cells should have the same width as the FillerColumn");

                dataGrid.Width = 200;
            });
            this.EnqueueYieldThread();
            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(fillerColumn.Width > fillerWidth, "FillerColumn Width should auto grow with DataGrid");
                Assert.IsTrue(fillerCell.ActualWidth == fillerColumn.Width, "Filler cells should update when FillerColumn width changes");

                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Collapsed, "Horizontal ScrollBar should be Collapsed");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");

                dataGrid.ColumnWidth = new DataGridLength(100);
                dataGrid.Width = 120;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Visible, "Horizontal ScrollBar should be Visible");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");
                dataGrid.Width = 150;
                dataGrid.RowHeaderWidth = 100;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Visible, "Horizontal ScrollBar should be Visible");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");
                dataGrid.RowHeaderWidth = 25;
                dataGrid.RowHeight = 150;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Collapsed, "Horizontal ScrollBar should be Collapsed");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Visible, "Vertical ScrollBar should be Visible");
                dataGrid.RowHeight = 120;
                dataGrid.RowHeaderWidth = 35;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Collapsed, "Horizontal ScrollBar should be Collapsed");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");
                dataGrid.RowHeight = 150;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Vertical ScrollBar causes us to have a Horizontal ScrollBar
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Visible, "Horizontal ScrollBar should be Visible");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Visible, "Vertical ScrollBar should be Visible");

                dataGrid.RowHeight = 120;
                dataGrid.RowHeaderWidth = 50;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                // Horizontal ScrollBar causes us to have a Vertical ScrollBar
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Visible, "Horizontal ScrollBar should be Visible");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Visible, "Vertical ScrollBar should be Visible");

                dataGrid.RowHeaderWidth = 25;
            });

            this.EnqueueYieldThread();
            EnqueueCallback(delegate
            {
                Assert.IsTrue(dataGrid.HorizontalScrollBar.Visibility == Visibility.Collapsed, "Horizontal ScrollBar should be Collapsed");
                Assert.IsTrue(dataGrid.VerticalScrollBar.Visibility == Visibility.Collapsed, "Vertical ScrollBar should be Collapsed");
            });

            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that inserting and removing items correctly updates SelectedItem and SelectedIndex.")]
        public virtual void UpdatesSelectedItemOnRowsChanged()
        {
            TDataClassSource listSomeType = new TDataClassSource();
            if ((listSomeType is INotifyCollectionChanged) && (listSomeType is IList))
            {
                ((IList)listSomeType).Add(new TDataClass());
                TDataClass newItem = new TDataClass();
                DataGrid dataGrid = new DataGrid();
                isLoaded = false;
                dataGrid.ItemsSource = listSomeType;
                dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
                TestPanel.Children.Add(dataGrid);

                EnqueueConditional(delegate { return isLoaded; });
                EnqueueCallback(delegate
                {
                    dataGrid.SelectedIndex = 0;
                    ((IList)listSomeType).Insert(0, newItem);
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(1, dataGrid.SelectedIndex);
                    ((IList)listSomeType).Remove(newItem);
                });
                this.EnqueueYieldThread();
                EnqueueCallback(delegate
                {
                    Assert.AreEqual(0, dataGrid.SelectedIndex);
                });
            }
            EnqueueTestComplete();
        }

        [TestMethod]
        [Asynchronous]
        [Description("Test that counts the number of elements in the visual tree for a 1 column 1 row datagrid")]
        public virtual void VisualTreeCount()
        {
            DataGrid dataGrid = new DataGrid();
            Assert.IsNotNull(dataGrid);
            isLoaded = false;
            dataGrid.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
            dataGrid.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
            dataGrid.Loaded += new RoutedEventHandler(dataGrid_Loaded);
            List<Person> people = new List<Person>();
            people.Add(new Person { Name = "circle guy" });
            dataGrid.ItemsSource = people;

            TestPanel.Children.Add(dataGrid);
            EnqueueConditional(delegate { return isLoaded; });
            this.EnqueueYieldThread();

            EnqueueCallback(delegate
            {
                // Make sure the DataGrid auto fills the space its given
                Assert.IsTrue(dataGrid.ActualWidth == TestPanel.ActualWidth, "The DataGrid's width didn't fill");
                Assert.IsTrue(dataGrid.ActualHeight == TestPanel.ActualHeight, "The DataGrid's height didn't fill");
                Assert.AreEqual(48, CountVisualTreeRecursively(dataGrid), "New number of elements in dataGrid visual tree");
            });

            EnqueueTestComplete();
        }

        public class Person
        {
            public string Name { get; set; }
        }
    }
}

