{
  type v = 
    | ID of string
    | STRING of string
    | MULT of string list
    | NONE
  type stanza = (string * v) list
  type t = stanza list

  let string_buffer = 
    Buffer.create 128

  let rec make_value lex_value lb =
    match lex_value lb with
    | `NL -> NONE
    | `STRING s ->
	begin
	  match make_value lex_value lb with
	  | NONE ->
	      STRING s
	  | STRING s2 ->
	      MULT [ s ; s2 ]
	  | MULT sl ->
	      MULT (s :: sl)
	  | ID _ ->
	      failwith "Basic_io_lexer: value"
	end
    | `ID id ->
	match lex_value lb with
	| `NL -> ID id
	| _ -> failwith "Basic_io_lexer: value"
}

let id    = ['a'-'f' '0'-'9']*
let ident = ['a'-'z' '_']+
let ws    = [' ' '\t']+
let nl    = [ '\n' ]

rule lex = parse
  | ws                           { lex lexbuf }
  | ident as k                   { let v = make_value lex_value lexbuf in
                                   `TOK (k, v) }
  | nl                           { `END_OF_STANZA }
  | eof                          { `EOF }

and nl = parse
  | ws                           { nl lexbuf }
  | nl                           { () }

and lex_value = parse
  | ws                           { lex_value lexbuf }
  | nl                           { `NL }
  | '[' (id as id) ']'           { `ID id }
  | '"'                          { Buffer.clear string_buffer ;
				   `STRING (string lexbuf) }

and string = parse
  | '"'                          { Buffer.contents string_buffer }
  | '\\' ['"' '\\']              { Buffer.add_char 
				     string_buffer
				     (Lexing.lexeme_char lexbuf 1) ;
				   string lexbuf }
  | [^ '"' '\\']+                { let off = lexbuf.Lexing.lex_start_pos in
                                   let len = lexbuf.Lexing.lex_curr_pos - off in
				   Buffer.add_substring 
				     string_buffer 
				     lexbuf.Lexing.lex_buffer
				     off len ;
				   string lexbuf }

{
  let rec _get_stanza acc lb = 
    match lex lb with
    | `TOK ((k, _) as v) -> 
	_get_stanza (v :: acc) lb
    | `END_OF_STANZA when acc = [] ->
	_get_stanza acc lb
    | `EOF
    | `END_OF_STANZA as e -> 
	e, List.rev acc

  let get_stanza lb =
    match _get_stanza [] lb with
    | `EOF, [] ->
	None
    | _, st ->
	Some st

  let rec _parse acc lb =
    match _get_stanza [] lb with
    | `EOF, [] -> 
	List.rev acc
    | `EOF, st -> 
	List.rev (st :: acc)
    | `END_OF_STANZA, st ->
	assert (st <> []) ;
	_parse (st :: acc) lb

  let parse lb =
    _parse [] lb

  let string_of_elem = function
    | MULT (s :: _)
    | STRING s
    | ID s -> s
    | MULT []
    | NONE -> ""
}

