 /* mixer.c: sound mixer capplet
 *
 * Copyright (C) 2007 Intel Corp.
 *
 * Written by: Todd Brandt <todd.e.brandt@intel.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <locale.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <sys/vfs.h>
#include <libosso.h>
#include <glib-object.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

#include "capplet-util.h"
#include "gconf-property-editor.h"
#include <gconf/gconf-client.h>
#include <volume.h>

#define NO_HARDWARE_ERROR "Unable to detect sound device!"
#define CREATE_ID(t, c) ((gpointer)((t)+((c) << 16)))
#define TAB_FROM_ID(id) ((int)((id)&0xffff))
#define CHANNEL_FROM_ID(id) ((int)(((id)>>16)&0xffff))

struct mixerchannel {
	GladeXML *glade;
	GtkLabel *label;
	GtkRange *range;
	GtkToggleButton *toggle;
};

struct mixertab {
	GladeXML *glade;
	GtkLabel *label;
	int num_channels;
	struct mixerchannel *chanlist;
};

static GladeXML *dialog = NULL;
static int num_tabs = 0;
static gboolean channels_in_sync = FALSE;
static struct mixertab *tablist = NULL;

static void cb_volume_scrollbar (GtkVScrollbar *vscrollbar, unsigned int id)
{
	int i, volume;
	int tab=TAB_FROM_ID(id), chan=CHANNEL_FROM_ID(id);
	GtkRange *range;

	range = GTK_RANGE(vscrollbar);
	volume = (int)gtk_range_get_value(range);

	if(channels_in_sync) {
	    set_all_volume(tab, volume);
	    for(i = 0; i < tablist[tab].num_channels; i++)
	    {
		if(i != chan) {
		    gtk_range_set_value(
			tablist[tab].chanlist[i].range, volume);
		}
	    }
	} else {
	    set_volume(tab, chan, volume);
	}
}

static void cb_mute_clicked (GtkToggleButton *toggle, unsigned int id)
{
	int i;
	int tab=TAB_FROM_ID(id), chan=CHANNEL_FROM_ID(id);

	if(channels_in_sync) {
	    set_all_switch(tab, !(toggle->active));
	    for(i = 0; i < tablist[tab].num_channels; i++)
	    {
		if(i != chan) {
		    gtk_toggle_button_set_active(
			tablist[tab].chanlist[i].toggle, toggle->active);
		}
	    }
	} else {
	    set_switch(tab, chan, !(toggle->active));
	}
}

static void cb_sync_clicked (GtkToggleButton *toggle) 
{ 
	channels_in_sync = toggle->active;
}

static void
dialog_response (GtkWidget *widget, gint response_id,
		GConfChangeSet *changeset)
{
int i;

	switch(response_id) {
	case GTK_RESPONSE_CLOSE:
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
	    gtk_widget_destroy(widget);
	    for(i = 0; i < num_tabs; i++)
	    {
		free(tablist[i].chanlist);
	    }
	    free(tablist);
	    tablist = NULL;
	    num_tabs = 0;
	    g_object_unref(dialog);
	    dialog = NULL;
	    gtk_main_quit();
	    break;
	}
}

static int
setup_dialog ()
{
	GtkNotebook *notebook;
	GtkWidget *widget;
	GtkContainer *box;
	int i, j, has_vol, has_mute;
	int volume, active, vmin=0, vmax=100;

	/* create a list of mixertab objects, one for each element */
	num_tabs = num_sound_elements();
	tablist = (struct mixertab *)malloc(num_tabs*sizeof(struct mixertab));

	/* The first element is already in the dialog */
	tablist[0].glade = dialog;

	/* Retrieve the notebook from the dialog */
	widget = WID("notebook1");
	if(!widget||!GTK_IS_NOTEBOOK(widget)) return -1;
	notebook = GTK_NOTEBOOK(widget);

	/* create as many new pages as needed just like the first */
	for(i = 1; i < num_tabs; i++)
	{
	    /* the channelbox is the tab contents */
	    tablist[i].glade = glade_xml_new (MA_GLADE_DIR "/mixer.glade",
		"channelbox", NULL);
	    if(tablist[i].glade == NULL) {
		free(tablist);
		return -1;
	    }
	}

	/* overwrite the first page label */
	widget = WID("tablabel");
	if(!widget||!GTK_IS_LABEL(widget)) 
	{
	    free(tablist);
	    return -1;
	}
	tablist[0].label = GTK_LABEL(widget);
	gtk_label_set_text(tablist[0].label, name_from_index(0));

	/* create new pages for each new element */
	for(i = 1; i < num_tabs; i++)
	{
	    /* grab a pointer to the channelbox instance */
	    widget = glade_xml_get_widget(tablist[i].glade, "channelbox");
	    tablist[i].label = GTK_LABEL(gtk_label_new(name_from_index(i)));
	    gtk_notebook_append_page(notebook, widget, GTK_WIDGET(tablist[i].label));
	}

	/* setup the callbacks from each of the new pages */
	for(i = 0; i < num_tabs; i++)
	{
	    get_support(i, &has_vol, &has_mute);
	    tablist[i].num_channels = num_channels(i);
	    tablist[i].chanlist = (struct mixerchannel *)malloc(
		    (tablist[i].num_channels)*sizeof(struct mixerchannel));
	    tablist[i].chanlist[0].glade = tablist[i].glade;

	    /* create new controls for each new channel */
	    for(j = 1; j < tablist[i].num_channels; j++)
	    {
		tablist[i].chanlist[j].glade = glade_xml_new (
			MA_GLADE_DIR "/mixer.glade",
			"channeltemplate", NULL);
	    }

	    widget = glade_xml_get_widget(
		tablist[i].glade, "channelbox");
	    box = GTK_CONTAINER(widget);

	    if(has_vol) get_volume_range(i, &vmin, &vmax);

	    /* get the channels up and working */
	    for(j = 0; j < tablist[i].num_channels; j++)
	    {
		/* insert the channel into the container */
		widget = glade_xml_get_widget(
		    tablist[i].chanlist[j].glade, "channeltemplate");
		gtk_container_add(box, widget);

		/* extract all the widgets in the controls */
		widget = glade_xml_get_widget(
		    tablist[i].chanlist[j].glade, "channellabel");
		tablist[i].chanlist[j].label = GTK_LABEL(widget);
		widget = glade_xml_get_widget(
		    tablist[i].chanlist[j].glade, "volumeslider");
		tablist[i].chanlist[j].range = GTK_RANGE(widget);
		widget = glade_xml_get_widget(
		    tablist[i].chanlist[j].glade, "mutebutton");
		tablist[i].chanlist[j].toggle = GTK_TOGGLE_BUTTON(widget);

		/* set the label for the channel */
		gtk_label_set_text(tablist[i].chanlist[j].label, 
		    name_from_channel(i, j));

		if(has_mute)
		{
		    /* now set the active switches to their defaults */
		    get_switch(i, j, &active);
		    gtk_toggle_button_set_active(
			tablist[i].chanlist[j].toggle, !active);

		    /* setup a callback on mute button click */
		    GTK_WIDGET_SET_FLAGS(GTK_WIDGET(tablist[i].chanlist[j].toggle), 
			GTK_VISIBLE | GTK_SENSITIVE);
		    g_signal_connect (G_OBJECT (tablist[i].chanlist[j].toggle),
			"toggled", G_CALLBACK (cb_mute_clicked), CREATE_ID(i, j));
		} else {
		    /* destroy the mute button if this channel doesn't support it */
		    gtk_widget_destroy(GTK_WIDGET(tablist[i].chanlist[j].toggle));
		    tablist[i].chanlist[j].toggle = NULL;
		}

		if(has_vol)
		{
		    /* now set the volume sliders to their defaults */
		    gtk_range_set_range(tablist[i].chanlist[j].range, 
			(gdouble)vmin, (gdouble)vmax);
		    get_volume(i, j, &volume);
		    gtk_range_set_value(tablist[i].chanlist[j].range, volume);

		    /* setup a callback on volume slider change */
		    GTK_WIDGET_SET_FLAGS(GTK_RANGE(tablist[i].chanlist[j].range), 
			GTK_VISIBLE | GTK_SENSITIVE);
		    g_signal_connect (G_OBJECT (tablist[i].chanlist[j].range), 
			"value-changed", G_CALLBACK (cb_volume_scrollbar), 
			CREATE_ID(i, j));
		} else {
		    /* destroy the volume slider if this channel doesn't support it */
		    gtk_widget_destroy(GTK_WIDGET(tablist[i].chanlist[j].range));
		    tablist[i].chanlist[j].range = NULL;
	        }
	    }
	}

	widget = WID ("syncbutton");
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(widget), channels_in_sync);
	g_signal_connect (widget, "toggled", 
		G_CALLBACK (cb_sync_clicked), NULL);

	g_signal_connect (WID ("mixer_dialog"), "response",
		G_CALLBACK (dialog_response), NULL);
	return 0;
}

void
catch_fire_and_explode()
{
	GtkWidget *d = gtk_dialog_new_with_buttons(
		"Settings Error",
		NULL,
		GTK_DIALOG_MODAL,
		GTK_STOCK_OK,
		GTK_RESPONSE_NONE,
		NULL);
		
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(d)->vbox),
			   gtk_label_new (_(NO_HARDWARE_ERROR)));
	g_signal_connect_swapped (d, 
				  "response", 
				  G_CALLBACK (gtk_widget_destroy),
				  d);
	gtk_window_set_default_size(GTK_WINDOW(d), 400, 200);
	gtk_widget_show_all(d);
}

int main(int argc, char *argv[], char *env[])
{
	GtkWidget   *widget;
	static GConfClient *client = NULL;
	
	gtk_init(&argc, &argv);

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	if (client == NULL)
	    client = gconf_client_get_default ();

	if (init_alsa_vars() == -1)
	{
	    catch_fire_and_explode();
	    return -1;
	}

	dialog = glade_xml_new (MA_GLADE_DIR "/mixer.glade",
				"mixer_dialog", NULL);
	if(setup_dialog() == -1)
	{
	    catch_fire_and_explode();
	    return -1;
	}

	widget = WID ("mixer_dialog");
	gtk_widget_show_all (widget);
	gtk_main();
	return 0;
}
