/* 
  Copyright (C) 2008 Steffen Basting, André Gaul

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <netinet/tcp.h>
#include <netinet/ip.h>
#include "lib/message.h"
#include "lib/game.h"

#define PONG_KICKED 		(-1)
#define PONG_FAILURE 		(-2)

short volatile signal_exiting= 0;



void sigterm_handler(int signal)
{
	if (signal == SIGPIPE)
		printf("Caught SIGPIPE.\n");
	if (signal == SIGURG)
		printf("Caught SIGURG.\n");
	if (signal == SIGTERM)
		signal_exiting= 1;
}



int main(int argc, char *argv[])
{
	if (argc <= 1) {
		printf("Usage: %s server [port]\n", argv[0]);
		exit(1);
	}

	struct sigaction termact= { .sa_handler= sigterm_handler, .sa_flags= SA_RESETHAND };
	sigemptyset(&termact.sa_mask);
	if ( (sigaction(SIGTERM, &termact, NULL)) ||
		 (termact.sa_flags= SA_RESTART, sigaction(SIGPIPE, &termact, NULL)) ) {
		fprintf(stderr, "Error putting signal handlers in place.\n");
		exit(-1);
	}

	struct netmessage_buffer *recvbuf= NULL, *sendbuf= NULL;
	if ( (netmessage_buffer_init(&recvbuf) != NETMSG_SUCCESS) ||
		 (netmessage_buffer_init(&sendbuf) != NETMSG_SUCCESS) ) {
		 fprintf(stderr, "Error initializing message buffers.\n");
		 exit(-1);
	}

	char *portname= "1212";
	if (argc > 2)
		portname= argv[2];
	struct addrinfo criteria= {
		.ai_family= AF_INET,
		.ai_socktype= SOCK_STREAM,
		.ai_protocol= IPPROTO_TCP,
		.ai_flags= AI_ADDRCONFIG | AI_CANONNAME // | AI_IDN | AI_CANONIDN
	};
	struct addrinfo *peers;
	int stat;
	if ((stat= getaddrinfo(argv[1], portname, &criteria, &peers))) {
		fprintf(stderr, "getaddrinfo(): %s\n", gai_strerror(stat));
		exit(-1);
	}
	char addrstr[INET_ADDRSTRLEN];
	printf("Connecting to %s", peers->ai_canonname);
	if (inet_ntop(peers->ai_addr->sa_family, &((struct sockaddr_in *)(peers->ai_addr))->sin_addr, addrstr, INET_ADDRSTRLEN))
		printf(" (%s:%d)", addrstr, ntohs(((struct sockaddr_in *)(peers->ai_addr))->sin_port));
	printf("\n");

	int sock= socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock == (-1)) {
		perror("socket()");
		exit(-1);
	}

	if (connect(sock, peers->ai_addr, sizeof(*peers->ai_addr)) == (-1)) {
		perror("socket()");
		exit(-1);
	}
	freeaddrinfo(peers);

	int yes= 1, rcvlow= 2;
	int maxseg= sizeof(struct netmessage);
	uint8_t tos= IPTOS_LOWDELAY;
	if ( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &yes, sizeof(yes)) == (-1) ||
		 // we expect to receive short integers
		 setsockopt(sock, SOL_SOCKET, SO_RCVLOWAT, &rcvlow, sizeof(rcvlow)) == (-1) ||
		 setsockopt(sock, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)) == (-1) ||
		 // ideally, we don't send or receive more data than a netmessage contains
		 setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, &maxseg, sizeof(maxseg)) == (-1) ||
		 setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, &yes, sizeof(yes)) == (-1) 
		)
		fprintf(stderr, "Socket options have not been set properly.\n");
	printf("Connected.\n");

	struct gameplay *game= NULL;
	// The reason we do this is to have the game state managed exclusively by the lib
	if ((game= gameplay_create()) == NULL) {
		fprintf(stderr, "Cannot allocate game state.\n");
		exit(PONG_FAILURE);
	}
	// Try setting sensible default values
	if (gameplay_init(game, game->version, gamemode_linear, gamepadprofile_flat) != PONG_SUCCESS)
		fprintf(stderr, "Warning: Init to default game state unsuccessful.\n");

	uint16_t team= 0;
	float pos= 0.5;

	struct netmessage msg;
	int netcode;
	int condition= PONG_SUCCESS;
	
	// make sure to retrieve a full game state from the server when joining the game
	if ( (netcode= netmessage_recv(sock, &msg, sizeof(msg), recvbuf)) == NETMSG_SUCCESS ) {
		switch (msg.hdr.id) {

			case NETMSG_STAT:
				gameplay_apply_state(&msg, game, &team);
				break;
		
			case NETMSG_KICK:
				printf("Got kicked (%s).\n", msg.payload.data);
				condition= PONG_KICKED;
				break;

			default:
				fprintf(stderr, "Unknown message, id == %d ('%c')\n", msg.hdr.id, (msg.hdr.id >' ')? msg.hdr.id : ' ');
				condition= PONG_FAILURE;
				break;
		}
	}
	else {
		fprintf(stderr, "Cannot determine initial game state (code == %d)\n", netcode);
		condition= PONG_FAILURE;
	}

	if (condition != PONG_SUCCESS) {
		free(game);
		exit(condition);
	}

	printf("I'm on team %1d\n", team);
	if (team > 1) team= 1;

	if (fcntl(sock, F_SETFL, O_NONBLOCK) == (-1))
		fprintf(stderr, "Socket I/O is blocking.\n");
	unsigned scoresense[2]= { 0, 0 };

	while(!signal_exiting)
	{
		//handle incoming messages
		while ( (netcode = netmessage_recv(sock, &msg, sizeof(msg), recvbuf)) == NETMSG_SUCCESS) {
			switch (msg.hdr.id) {

				case NETMSG_STAT:
				case NETMSG_UPDT:
					gameplay_apply_state(&msg, game, &team);
					if (team >1) team= 1;
					if (msg.hdr.id == NETMSG_STAT)
					if (scoresense[0] != game->pad_attr[0].score || scoresense[1] != game->pad_attr[1].score) {
						scoresense[0]= game->pad_attr[0].score;
						scoresense[1]= game->pad_attr[1].score;
						printf("The score is now %d : %d\n",scoresense[0], scoresense[1]);
					}
					break;

				case NETMSG_KICK:
					printf("Got kicked (%s)\n", msg.payload.data);
					condition= PONG_KICKED;
					break;

				default:
					fprintf(stderr, "Unknown message, id == %d ('%c')\n", msg.hdr.id, (msg.hdr.id >' ')? msg.hdr.id : ' ');
					break;
			}
		}

		if ((netcode != NETMSG_SUCCESS) && (netcode != NETMSG_FAIL_DELIVER) && (netcode != NETMSG_PARTIAL)) {
			fprintf(stderr, "Socket failure while reading (%d).\n", netcode);
			condition= PONG_FAILURE;
		}

		if (condition != PONG_SUCCESS) break;

		if (sendbuf->len)
			netmessage_buffer_flush(sock, sendbuf);
		if (game->pad[team].mean != game->ball.pos[1]) {
			pos= game->ball.pos[1];
			if (pos + game->pad[team].size / 2 > 1.f)
				pos = 1.f - game->pad[team].size / 2;
			if (pos - game->pad[team].size / 2 < 0.f)
				pos = game->pad[team].size / 2;
		}
		if (pos != game->pad[team].mean) {
			uint16_t sendpos = pos * PONG_RANGE_SPREAD;
			int sendcode = netmessage_send(sock, NETMSG_POS, &sendpos, sizeof(sendpos), sendbuf);
			if ( (sendcode!= NETMSG_SUCCESS) && (sendcode != NETMSG_FAIL_DELIVER) && (sendcode != NETMSG_PARTIAL)) {
				printf("Socket failure while sending (%d).\n", sendcode);
				break;
			}
		}
		usleep(50000);
	}

	netmessage_send(sock, NETMSG_EXIT, NULL, 0, sendbuf);
	close(sock);
	if (game) free(game);
	if (recvbuf) free(recvbuf);
	if (sendbuf) free(sendbuf);
  
	return 0;
}

