/* 
  Copyright (C) 2008 Steffen Basting, André Gaul

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef WIN32
# include <winsock2.h>
# include <ws2tcpip.h>
#else
# include <sys/socket.h>
# include <arpa/inet.h>
# include <netdb.h>
# include <netinet/tcp.h>
# include <netinet/ip.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <caca.h>
#include "lib/message.h"
#include "lib/game.h"

#define PONG_KICKED 		(-1)
#define PONG_FAILURE 		(-2)

#define RTTM_QUERY_INTERVAL 	12000 	// every 12sec
#define DRAW_REFRESH_INTERVAL 	   25 	// every 25msec

#define INFOBARCOLOR 	CACA_BLACK
#define BACKCOLOR 	CACA_DARKGRAY
#define BALL_COLOR 	CACA_YELLOW
#define PADLEFT_COLOR 	CACA_BLUE
#define PADRIGHT_COLOR 	CACA_RED
#define PADHIGHLIGHT_COLOR 	CACA_WHITE
#define SCORESTR "SCORE"

short signal_exiting= 0;
struct gameplay *game= NULL;


#ifndef WIN32
void sigterm_handler(int signal)
{
	if (signal == SIGPIPE)
		printf("Caught SIGPIPE.\n");
	if (signal == SIGURG)
		printf("Caught SIGURG.\n");
	if (signal == SIGTERM)
		signal_exiting= 1;
}
#endif



static void free_game()
{
	if (game) free(game);
}



void draw_paddle(caca_canvas_t *cv, const struct gamepaddle *pad, uint16_t team, uint8_t padcolor)
{
  caca_set_color_ansi(cv, padcolor, BACKCOLOR); 

	float mean = ( 1 - pad->mean ) * (caca_get_canvas_height(cv) -1),
	      size = pad->size * (caca_get_canvas_height(cv) -1);

	if (mean + size/2 > (caca_get_canvas_height(cv) -1)) {
#ifdef DEBUG
		fprintf(stderr, "Paddle[%d] out of bounds (%f).\n", team, pad->mean);
#endif
		mean = caca_get_canvas_height(cv)-size/2;
	}
		 
	if (mean < size/2) {
#ifdef DEBUG
		fprintf(stderr, "Paddle[%d] out of bounds (%f).\n", team, pad->mean);
#endif
		mean = size/2;
	}
	
	if ( !(size/2)) {
#ifdef DEBUG
		fprintf(stderr, "Paddle[%d] size == 0.\n", team);
#endif
		size=2;
	}

  for (uint16_t ypos = mean - size/2; ypos < mean + size/2; ypos++)
	  caca_put_str(cv, (team==0 ? 0 : caca_get_canvas_width(cv)-1), ypos, "#");
}



void draw_everything(caca_canvas_t *cv, const struct gameplay *game, float pos, uint16_t team) 
{
	caca_set_color_ansi(cv, BACKCOLOR, BACKCOLOR);
	caca_clear_canvas(cv);
	// ball
	caca_set_color_ansi(cv, BALL_COLOR, BACKCOLOR);
	uint16_t x = game->ball.pos[0] * caca_get_canvas_width(cv),
	         y = (1 - game->ball.pos[1]) * (caca_get_canvas_height(cv) - 1) +1;
	caca_put_str(cv, x , y, "o"); 
	// paddles
	uint8_t colors[2]= { PADLEFT_COLOR, PADRIGHT_COLOR };
	for (unsigned side= 0; side< 2; side++)
	  draw_paddle(cv, game->pad+side, side, colors[side]);
	struct gamepaddle mypad= { .size= game->pad[team].size, .mean= pos, .var= 0 };
	draw_paddle(cv, &mypad, team, PADHIGHLIGHT_COLOR);
	// score
	caca_set_color_ansi(cv, PADHIGHLIGHT_COLOR, INFOBARCOLOR);
	caca_fill_box(cv, 0, 0, caca_get_canvas_width(cv), 1, ' ');
	caca_put_str(cv, (caca_get_canvas_width(cv) - strlen(SCORESTR)) /2, 0, SCORESTR);
	for (unsigned side= 0; side< 2; side++) {
		caca_set_color_ansi(cv, colors[side], INFOBARCOLOR);
		int putlen= snprintf(NULL, 0, "%u", game->pad_attr[side].score) +1;
		char *putscore= malloc(putlen);
		if (!putscore) continue;
		snprintf(putscore, putlen, "%u", game->pad_attr[side].score);
		caca_put_str(cv, (side)? (caca_get_canvas_width(cv) -putlen +1) : 0, 0, putscore);
		free(putscore);
	}
}



static void timeval_combine(const int lfact, struct timeval *lval, const int rfact, const struct timeval *rval)
{
	// combine linearly
	lval->tv_sec= lval->tv_sec * lfact + rval->tv_sec * rfact;
	lval->tv_usec= lval->tv_usec * lfact + rval->tv_usec * rfact;
	// normalize
	if (lval->tv_usec < 0) {
		lval->tv_sec+= lval->tv_usec / (1000L * 1000L) -1;
		lval->tv_usec-= (lval->tv_usec / (1000L * 1000L) -1) * (1000L * 1000L);
	}
	if (lval->tv_usec >= 1000L * 1000L) {
		lval->tv_sec+= lval->tv_usec / (1000L * 1000L);
		lval->tv_usec-= (lval->tv_usec / (1000L * 1000L)) * (1000L * 1000L);
	}
}



int main(int argc, char *argv[])
{
  caca_canvas_t *cv;
  caca_display_t *dp;

	if (argc <= 1) {
		printf("Usage: %s server [port]\n", argv[0]);
		exit(1);
	}

#ifndef WIN32
	struct sigaction termact= { .sa_handler= sigterm_handler, .sa_flags= SA_RESETHAND };
	sigemptyset(&termact.sa_mask);
	if ( (sigaction(SIGTERM, &termact, NULL)) ||
		 (termact.sa_flags= SA_RESTART, sigaction(SIGPIPE, &termact, NULL)) ) {
		fprintf(stderr, "Error putting signal handlers in place.\n");
		exit(-1);
	}
#else
	// winsock init
	WSADATA wsaData;
	int wsaresult = WSAStartup(MAKEWORD(2,2), &wsaData);
	if (wsaresult != 0) {
    fprintf(stderr, "Error initializing Windows Sockets (code==%d)\n", wsaresult);
    exit(-1);
	}
#endif

	struct netmessage_buffer *recvbuf= NULL, *sendbuf= NULL;
	if ( (netmessage_buffer_init(&recvbuf) != NETMSG_SUCCESS) ||
		 (netmessage_buffer_init(&sendbuf) != NETMSG_SUCCESS) ) {
		 fprintf(stderr, "Error initializing message buffers.\n");
		 exit(-1);
	}
	netmessage_scrambler_init(); 	// conceal the real system clock

	char *portname= "1212";
	if (argc > 2)
		portname= argv[2];
	struct addrinfo criteria= {
		.ai_family= AF_INET,
		.ai_socktype= SOCK_STREAM,
		.ai_protocol= IPPROTO_TCP,
		.ai_flags= AI_ADDRCONFIG | AI_CANONNAME // | AI_IDN | AI_CANONIDN
	};
	struct addrinfo *peers;
	int stat;
	if ((stat= getaddrinfo(argv[1], portname, &criteria, &peers))) {
		fprintf(stderr, "getaddrinfo(): %s\n", gai_strerror(stat));
		exit(-1);
	}
	char addrstr[INET_ADDRSTRLEN];
	printf("Connecting to %s", peers->ai_canonname);
	if (inet_ntop(peers->ai_addr->sa_family, &((struct sockaddr_in *)(peers->ai_addr))->sin_addr, addrstr, INET_ADDRSTRLEN))
		printf(" (%s:%d)", addrstr, ntohs(((struct sockaddr_in *)(peers->ai_addr))->sin_port));
	printf("\n");

	int sock= socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock == (-1)) {
		perror("socket()");
		exit(-1);
	}

	if (connect(sock, peers->ai_addr, sizeof(*peers->ai_addr)) == (-1)) {
		perror("socket()");
		exit(-1);
	}

	int yes= 1, rcvlow= 2;
	int maxseg= sizeof(struct netmessage);
	uint8_t tos= IPTOS_LOWDELAY;
	if ( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &yes, sizeof(yes)) == (-1) ||
		 // we expect to receive short integers
		 setsockopt(sock, SOL_SOCKET, SO_RCVLOWAT, &rcvlow, sizeof(rcvlow)) == (-1) ||
		 setsockopt(sock, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)) == (-1) ||
		 // ideally, we don't send or receive more data than a netmessage contains
		 setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, &maxseg, sizeof(maxseg)) == (-1) ||
		 setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, &yes, sizeof(yes)) == (-1) 
		)
		fprintf(stderr, "Socket options have not been set properly.\n");
	printf("Connected.\n");

	cv = caca_create_canvas(0, 0);
	if(cv == NULL)
	{
		printf("Can't create canvas\n");
		return -1;
	}

	dp = caca_create_display(cv);
	uint16_t team= 0;
	float pos= .5f;

	caca_set_display_time(dp, 800);
	caca_set_mouse(dp, 0);

	if (gameplay_getversion() < sizeof(struct gameplay)) {
		fprintf(stderr, "This client requires a newer version of the mmpong library (>=%u).\n", (unsigned)sizeof(struct gameplay));
		exit(PONG_FAILURE);
	}
	if (atexit(free_game)) {
		fprintf(stderr, "Error adding an exit() handler.\n");
		exit(PONG_FAILURE);
	}
	// The reason we do this is to have the game state managed exclusively by the lib
	if ((game= gameplay_create()) == NULL) {
		fprintf(stderr, "Cannot allocate game state.\n");
		exit(PONG_FAILURE);
	}
	// Try setting sensible default values
	if (gameplay_init(game, game->version, gamemode_linear, gamepadprofile_flat) != PONG_SUCCESS)
		fprintf(stderr, "Warning: Init to default game state unsuccessful.\n");
	struct netmessage msg;
	int netcode= 0;
	
	// make sure to retrieve a full game state from the server when joining the game
	if ( (netcode= netmessage_recv(sock, &msg, sizeof(msg), recvbuf)) == NETMSG_SUCCESS ) {
		switch (msg.hdr.id) {

		case NETMSG_STAT:
			gameplay_apply_state(&msg, game, &team);
			break;
		
		case NETMSG_KICK:
			printf("Got kicked! Do you see what happens, Larry: %s\n", msg.payload.data);
			exit(PONG_KICKED);
			break;

		default:
			fprintf(stderr, "Received unknown message, id == %d ('%c')\n",
				msg.hdr.id, (msg.hdr.id > ' ')? msg.hdr.id : ' ');
			exit(PONG_FAILURE);
			break;
		}
	}
	else {
		fprintf(stderr, "Cannot determine initial game state (code == %d)\n", netcode);
		exit(PONG_FAILURE);
	}

	// determine offset 
	struct timeval peer_offset, relative;
	gettimeofday(&peer_offset, NULL);
	memcpy(&game->stamp, &peer_offset, sizeof(game->stamp));
	memcpy(&game->lasthit, &game->stamp, sizeof(game->lasthit));
	netmessage_get_hdr_stamp(&msg, &relative);
	timeval_combine(1, &peer_offset, (-1), &relative);
	printf("Client to server time offset: ");
	if (peer_offset.tv_sec > 60*60*24) printf("%ldd ", (long)peer_offset.tv_sec / (60*60*24));
	if (peer_offset.tv_sec % (60*60*24) > 60*60) printf("%ldh ", (long)(peer_offset.tv_sec % (60*60*24)) / (60*60));
	if (peer_offset.tv_sec % (60*60) > 60) printf("%ldm ", (long)(peer_offset.tv_sec % (60*60)) / 60);
	printf("%ld.%06lds%s\n", (peer_offset.tv_sec <0)? (long)peer_offset.tv_sec : (long)(peer_offset.tv_sec % 60),
		peer_offset.tv_usec, (peer_offset.tv_sec > 60*60*24)? " (Scrambler Active)" : "");
	int adjusted_delay= 0, send_ping= 1;

#ifdef DEBUG
	printf("[Packet | Initial] Ball pos == ( %1.2f, %1.2f ), dir == ( %1.2f, %1.2f ), paddles == { %1.2f, %1.2f }, team == %1d\n", 
		game->ball.pos[0], game->ball.pos[1],
		game->ball.dir[0], game->ball.dir[1],
		game->pad[0].mean, game->pad[1].mean, team);
#endif

	if (fcntl(sock, F_SETFL, O_NONBLOCK) == (-1))
		fprintf(stderr, "Socket I/O is blocking.\n");

	// network delay
	struct timeval reference, peer_delay;
	memset(&peer_delay, 0, sizeof(peer_delay));
	uint16_t rttmpos= UINT16_MAX;
//	gettimeofday(&reference, NULL);

	// indisputable state
	unsigned server_score[2]= { game->pad_attr[0].score, game->pad_attr[1].score };

	struct timeval caca_timer;
	memset(&caca_timer, 0, sizeof(caca_timer));
	caca_event_t ev;
	while(!signal_exiting)
	{
		// measure network lag periodically
		if (send_ping == 0) {
			gettimeofday(&relative, NULL);
			timeval_combine(1, &relative, (-1), &reference);
			if (relative.tv_sec> RTTM_QUERY_INTERVAL / 1000L)
				send_ping= 1;
			if ((relative.tv_sec == RTTM_QUERY_INTERVAL / 1000L) && (relative.tv_usec / 1000L >= RTTM_QUERY_INTERVAL % 1000L))
				send_ping= 1;
		}
		if (send_ping == 1) {
			if (netmessage_send(sock, NETMSG_POS, &rttmpos, sizeof(rttmpos), sendbuf) != NETMSG_SUCCESS)
				send_ping= (-1);
			gettimeofday(&reference, NULL);
			send_ping= 2;
		}

		// handle incoming messages
		while ( (netcode = netmessage_recv(sock, &msg, sizeof(msg), recvbuf)) == NETMSG_SUCCESS) {
#ifdef DEBUG
			printf("Receiving.\n");
#endif
			switch (msg.hdr.id) {

				case NETMSG_STAT:
				case NETMSG_UPDT:
					gameplay_apply_state(&msg, game, &team);
					// system clock offset (game state update ---> send delay is implicit)
					timeval_combine(1, &game->stamp, 1, &peer_offset);
					if (adjusted_delay) 	// network lag
						timeval_combine(1, &game->stamp, -1, &peer_delay);
					memcpy(&game->lasthit, &game->stamp, sizeof(game->lasthit));
					// save score
					for (int idx= 0; idx< 2; idx++)
						server_score[idx]= game->pad_attr[idx].score;
					break;

				case NETMSG_POS:
					// near-synchronous ping reply
					if ((send_ping != 2) || (msg.payload.position != rttmpos)) {
						fprintf(stderr, "Received irregular message (position %d).\n", msg.payload.position);
						break;
					}
					gettimeofday(&relative, NULL);
					timeval_combine(-1, &reference, 1, &relative);
					reference.tv_sec>>= 1; 	// two ways ---> one way
					reference.tv_usec>>= 1;
					memcpy(&peer_delay, &reference, sizeof(peer_delay));
					memcpy(&reference, &relative, sizeof(reference));
					printf("Measured network delay: %ld.%06lds\n", (long)peer_delay.tv_sec, (long)peer_delay.tv_usec);
					send_ping= 0;
					if (!adjusted_delay) { 	// determine the net clock offset
						peer_offset.tv_sec-= peer_delay.tv_sec;
						peer_offset.tv_usec-= peer_delay.tv_usec;
						if (peer_offset.tv_usec <0) {
							peer_offset.tv_usec+= 1000L * 1000L;
							peer_offset.tv_sec--;
						}
						adjusted_delay= 1;
					}
					break;

				case NETMSG_KICK:
					printf("Got kicked! Do you see what happens, Larry: %s\n", msg.payload.data);
					exit(PONG_KICKED);
					signal_exiting= 1;
					break;

				default:
					fprintf(stderr, "Received unknown message, id == %d ('%c').\n",
						msg.hdr.id, (msg.hdr.id > ' ')? msg.hdr.id : ' ');
					break;
			}
		}

		if ((netcode != NETMSG_SUCCESS) && (netcode != NETMSG_FAIL_DELIVER) && (netcode != NETMSG_PARTIAL)) {
			fprintf(stderr, "Socket failure while reading (%d).\n", netcode);
			exit(PONG_FAILURE);
		}

		if (game->status == gamestatus_running) {
			// adjust game
			gettimeofday(&relative, NULL);
			gameplay_update(game, &relative);
			// restore score
			for (int idx= 0; idx< 2; idx++)
				game->pad_attr[idx].score= server_score[idx];
		}
		if ((caca_timer.tv_sec == 0) && (caca_timer.tv_usec < 1000)) {
			// render game
#ifdef DEBUG
			printf("Drawing.\n");
#endif
			draw_everything(cv, game, pos, team);
			caca_refresh_display(dp);
			caca_timer.tv_usec= 1000L * DRAW_REFRESH_INTERVAL;
		}
		// try flushing send buffers
		if (sendbuf->len)
			netmessage_buffer_flush(sock, sendbuf);

		// wait for either a signal on the line or the refresh timer to expire
		fd_set listen;
		FD_ZERO(&listen);
		FD_SET(sock, &listen);
		int ret= select(sock +1, &listen, NULL, NULL, &caca_timer);
		if ((ret == (-1)) && (errno != EINTR)) {
			perror("select()");
			if (caca_timer.tv_usec > 1000) continue;
		}
		if ((ret == 1) && (send_ping == 2))
			continue; 	// hot phase of delay measurement

		// handle user events
		int newpos=0;
		while (caca_get_event(dp, CACA_EVENT_ANY, &ev, 0)) {
			switch (caca_get_event_type(&ev)) {
			case CACA_EVENT_RESIZE:
				draw_everything(cv, game, pos, team);
				break;
			case CACA_EVENT_KEY_PRESS:
				if (caca_get_event_key_ch(&ev) == 'q') signal_exiting=1;
				if (caca_get_event_key_ch(&ev) == CACA_KEY_UP) {
					pos += (game->pad[team].size / 2)? game->pad[team].size / 2 : .1f;
					newpos = 1;
				}
				if (caca_get_event_key_ch(&ev) == CACA_KEY_DOWN) {
					pos -= (game->pad[team].size / 2)? game->pad[team].size / 2 : .1f;
					newpos = 1;
				}
				break;
			case CACA_EVENT_MOUSE_MOTION:
				pos = 1 - ((float)caca_get_mouse_y(dp)) / caca_get_canvas_height(cv);
				newpos = 1;
				break;
			default:
				break;
			}  
		}

		// notify the server
		if (newpos) {
#ifdef DEBUG
			printf("Sending.\n");
#endif
			if (pos + game->pad[team].size / 2 > 1.f)
				pos = 1.f - game->pad[team].size / 2;
			if (pos - game->pad[team].size / 2 < 0.f)
				pos = game->pad[team].size / 2;
			uint16_t sendpos = pos * PONG_RANGE_SPREAD;
			int sendcode = netmessage_send(sock, NETMSG_POS, &sendpos, sizeof(sendpos), sendbuf);
			if ( (sendcode!= NETMSG_SUCCESS) && (sendcode != NETMSG_FAIL_DELIVER) && (sendcode != NETMSG_PARTIAL)) {
				printf("Socket failure while sending (%d).\n", sendcode);
				break;
			}
		}
	}

	caca_free_display(dp);
	caca_free_canvas(cv);

	netmessage_send(sock, NETMSG_EXIT, NULL, 0, sendbuf);
	freeaddrinfo(peers);
	close(sock);

#ifdef WIN32
	WSACLeanup();
#endif

	if (recvbuf) free(recvbuf);
	if (sendbuf) free(sendbuf);
  
	return 0;
}

