# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import wx
import numpy

# load modules
from ids import *
import config
import images
import doc
import mwx
import mspy


# FLOATING PANEL WITH MASSCALC TOOLS
# ----------------------------------

class panelMasscalc(wx.MiniFrame):
    """Masscalc tools."""
    
    def __init__(self, parent, tool='pattern'):
        wx.MiniFrame.__init__(self, parent, -1, 'Mass Calculator', size=(400, 300), style=wx.DEFAULT_FRAME_STYLE & ~ (wx.RESIZE_BOX | wx.MAXIMIZE_BOX))
        
        self.parent = parent
        
        self.currentTool = tool
        self.currentCompound = None
        self.currentIons = None
        self.currentIon = None
        self.currentProfile = None
        
        # make gui items
        self.makeGUI()
        wx.EVT_CLOSE(self, self.onClose)
        
        # select default tool
        self.onToolSelected(tool=self.currentTool)
    # ----
    
    
    def makeGUI(self):
        """Make gui notebook."""
        
        # make toolbar
        toolbar = self.makeToolbar()
        
        # make panels
        summary = self.makeSummaryPanel()
        ionseries = self.makeIonseriesPanel()
        pattern = self.makePatternPanel()
        
        # pack element
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(summary, 1, wx.EXPAND, 0)
        self.mainSizer.Add(ionseries, 1, wx.EXPAND, 0)
        self.mainSizer.Add(pattern, 1, wx.EXPAND, 0)
        
        self.mainSizer.Hide(1)
        self.mainSizer.Hide(2)
        self.mainSizer.Hide(3)
        
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
    # ----
    
    
    def makeToolbar(self):
        """Make toolbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrToolbar'], size=(-1, mwx.TOOLBAR_HEIGHT))
        
        # make buttons
        self.summary_butt = wx.BitmapButton(panel, ID_masscalcSummary, images.lib['masscalcSummaryOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.summary_butt.SetToolTip(wx.ToolTip("Compound summary"))
        self.summary_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.ionseries_butt = wx.BitmapButton(panel, ID_masscalcIonSeries, images.lib['masscalcIonSeriesOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.ionseries_butt.SetToolTip(wx.ToolTip("Ion series"))
        self.ionseries_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.pattern_butt = wx.BitmapButton(panel, ID_masscalcPattern, images.lib['masscalcPatternOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.pattern_butt.SetToolTip(wx.ToolTip("Isotopic pattern"))
        self.pattern_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        # make compound fields
        compound_label = wx.StaticText(panel, -1, "Formula:")
        self.compound_value = wx.TextCtrl(panel, -1, "", size=(250, -1))
        compound_label.SetFont(wx.SMALL_FONT)
        self.compound_value.Bind(wx.EVT_TEXT, self.onCompoundChanged)
        
        # make save button
        self.save_butt = wx.Button(panel, -1, "Save", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.save_butt.SetFont(wx.SMALL_FONT)
        self.save_butt.Bind(wx.EVT_BUTTON, self.onSave)
        self.save_butt.Enable(False)
        
        # pack elements
        self.toolbar = wx.BoxSizer(wx.HORIZONTAL)
        self.toolbar.AddSpacer(mwx.TOOLBAR_LSPACE)
        self.toolbar.Add(self.summary_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.Add(self.ionseries_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.pattern_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.AddSpacer(20)
        self.toolbar.Add(compound_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        self.toolbar.Add(self.compound_value, 1, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.AddSpacer(20)
        self.toolbar.Add(self.save_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.AddSpacer(mwx.TOOLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(self.toolbar, 1, wx.EXPAND)
        
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makeSummaryPanel(self):
        """Make compound summary panel."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        summaryFormula_label = wx.StaticText(panel, -1, "Composition:")
        self.summaryFormula_value = wx.TextCtrl(panel, -1, "", size=(200, -1), style=wx.TE_READONLY)
        
        summaryMono_label = wx.StaticText(panel, -1, "Monoisotopic mass:")
        self.summaryMono_value = wx.TextCtrl(panel, -1, "", size=(200, -1), style=wx.TE_READONLY)
        
        summaryAverage = wx.StaticText(panel, -1, "Average mass:")
        self.summaryAverage_value = wx.TextCtrl(panel, -1, "", size=(200, -1), style=wx.TE_READONLY)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(summaryFormula_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.summaryFormula_value, (0,1))
        grid.Add(summaryMono_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.summaryMono_value, (1,1))
        grid.Add(summaryAverage, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.summaryAverage_value, (2,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeIonseriesPanel(self):
        """Make controls for ionseries."""
        
        # init panel
        ctrlPanel = mwx.bgrPanel(self, -1, images.lib['bgrControlbar'], size=(-1, mwx.CONTROLBAR_HEIGHT))
        
        # make controls
        ionseriesAgent_label = wx.StaticText(ctrlPanel, -1, "Agent:")
        ionseriesAgent_label.SetFont(wx.SMALL_FONT)
        self.ionseriesAgentFormula_value = wx.TextCtrl(ctrlPanel, -1, config.masscalc['ionseriesAgent'], size=(60, mwx.SMALL_TEXTCTRL_HEIGHT))
        self.ionseriesAgentFormula_value.SetFont(wx.SMALL_FONT)
        self.ionseriesAgentFormula_value.Bind(wx.EVT_TEXT, self.onIonseriesChanged)
        
        ionseriesAgentCharge_label = wx.StaticText(ctrlPanel, -1, "Agent charge:")
        ionseriesAgentCharge_label.SetFont(wx.SMALL_FONT)
        self.ionseriesAgentCharge_value = wx.TextCtrl(ctrlPanel, -1, str(config.masscalc['ionseriesAgentCharge']), size=(40, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('int'))
        self.ionseriesAgentCharge_value.SetFont(wx.SMALL_FONT)
        self.ionseriesAgentCharge_value.Bind(wx.EVT_TEXT, self.onIonseriesChanged)
        
        ionseriesPolarity_label = wx.StaticText(ctrlPanel, -1, "Polarity:")
        ionseriesPolarity_label.SetFont(wx.SMALL_FONT)
        
        self.ionseriesPositive_radio = wx.RadioButton(ctrlPanel, -1, "Positive", style=wx.RB_GROUP)
        self.ionseriesPositive_radio.SetFont(wx.SMALL_FONT)
        self.ionseriesPositive_radio.Bind(wx.EVT_RADIOBUTTON, self.onIonseriesChanged)
        
        self.ionseriesNegative_radio = wx.RadioButton(ctrlPanel, -1, "Negative")
        self.ionseriesNegative_radio.SetFont(wx.SMALL_FONT)
        self.ionseriesNegative_radio.Bind(wx.EVT_RADIOBUTTON, self.onIonseriesChanged)
        
        if config.masscalc['ionseriesPolarity'] == -1:
            self.ionseriesNegative_radio.SetValue(True)
        else:
            self.ionseriesPositive_radio.SetValue(True)
        
        # pack controls
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_LSPACE)
        sizer.Add(ionseriesAgent_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.ionseriesAgentFormula_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(ionseriesAgentCharge_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.ionseriesAgentCharge_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(ionseriesPolarity_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.ionseriesPositive_radio, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.ionseriesNegative_radio, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(sizer, 1, wx.EXPAND)
        
        controls.Fit(ctrlPanel)
        ctrlPanel.SetSizer(controls)
        
        # make ions list
        self.makeIonsList()
        
        # pack main
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(ctrlPanel, 0, wx.EXPAND)
        mainSizer.Add(self.ionsList, 1, wx.EXPAND|wx.ALL, mwx.LISTCTRL_NO_SPACE)
        
        return mainSizer
    # ----
    
    
    def makePatternPanel(self):
        """Make controls for pattern simulation."""
        
        # init panel
        ctrlPanel = mwx.bgrPanel(self, -1, images.lib['bgrControlbar'], size=(-1, mwx.CONTROLBAR_HEIGHT))
        
        # make controls
        self.patternCollapse_butt = wx.BitmapButton(ctrlPanel, ID_masscalcCollapse, images.lib['arrowDown'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.patternCollapse_butt.Bind(wx.EVT_BUTTON, self.onCollapse)
        
        patternFwhm_label = wx.StaticText(ctrlPanel, -1, "FWHM:")
        patternFwhm_label.SetFont(wx.SMALL_FONT)
        self.patternFwhm_value = mwx.scrollTextCtrl(ctrlPanel, -1, str(config.masscalc['patternFwhm']), multiplier=0.1, limits=(0.001,10), digits=3, size=(55, mwx.SMALL_TEXTCTRL_HEIGHT))
        self.patternFwhm_value.SetFont(wx.SMALL_FONT)
        self.patternFwhm_value.Bind(wx.EVT_TEXT, self.onPatternChanged)
        
        patternIntensity_label = wx.StaticText(ctrlPanel, -1, "Intensity:")
        patternIntensity_label.SetFont(wx.SMALL_FONT)
        self.patternIntensity_value = mwx.scrollTextCtrl(ctrlPanel, -1, str(config.masscalc['patternIntensity']), multiplier=0.1, limits=(1,None), size=(70, mwx.SMALL_TEXTCTRL_HEIGHT))
        self.patternIntensity_value.SetFont(wx.SMALL_FONT)
        self.patternIntensity_value.Bind(wx.EVT_TEXT, self.onPatternChanged)
        
        patternBaseline_label = wx.StaticText(ctrlPanel, -1, "Baseline:")
        patternBaseline_label.SetFont(wx.SMALL_FONT)
        self.patternBaseline_value = mwx.scrollTextCtrl(ctrlPanel, -1, str(config.masscalc['patternBaseline']), multiplier=0.1, limits=(0,None), size=(70, mwx.SMALL_TEXTCTRL_HEIGHT))
        self.patternBaseline_value.SetFont(wx.SMALL_FONT)
        self.patternBaseline_value.Bind(wx.EVT_TEXT, self.onPatternChanged)
        
        patternShift_label = wx.StaticText(ctrlPanel, -1, "Shift:")
        patternShift_label.SetFont(wx.SMALL_FONT)
        self.patternShift_value = mwx.scrollTextCtrl(ctrlPanel, -1, str(0), step=0.001, digits=3, limits=(-1.,1.), size=(55, mwx.SMALL_TEXTCTRL_HEIGHT))
        self.patternShift_value.SetFont(wx.SMALL_FONT)
        self.patternShift_value.Bind(wx.EVT_TEXT, self.onPatternChanged)
        
        # pack controls
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        sizer.Add(self.patternCollapse_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(10)
        sizer.Add(patternFwhm_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.patternFwhm_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(patternIntensity_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.patternIntensity_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(patternBaseline_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.patternBaseline_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(patternShift_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.patternShift_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(sizer, 1, wx.EXPAND)
        
        controls.Fit(ctrlPanel)
        ctrlPanel.SetSizer(controls)
        
        # make plot canvas
        self.makePatternCanvas()
        
        # pack main
        self.patternSizer = wx.BoxSizer(wx.VERTICAL)
        self.patternSizer.Add(ctrlPanel, 0, wx.EXPAND, 0)
        self.patternSizer.Add(self.patternCanvas, 1, wx.EXPAND)
        
        return self.patternSizer
    # ----
    
    
    def makeIonsList(self):
        """Make ions list."""
        
        # init list
        self.ionsList = mwx.sortListCtrl(self, -1, size=(531, 200), style=mwx.LISTCTRL_STYLE_SINGLE)
        self.ionsList.SetFont(wx.SMALL_FONT)
        self.ionsList.setAltColour(mwx.LISTCTRL_ALTCOLOUR)
        
        # set events
        self.ionsList.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onIonSelected)
        self.ionsList.Bind(wx.EVT_KEY_DOWN, self.onListKey)
        
        # make columns
        self.ionsList.InsertColumn(0, "ion", wx.LIST_FORMAT_LEFT)
        self.ionsList.InsertColumn(1, "monoisotopic mass", wx.LIST_FORMAT_RIGHT)
        self.ionsList.InsertColumn(2, "average mass", wx.LIST_FORMAT_RIGHT)
        
        # set column widths
        for col, width in enumerate((190,160,160)):
            self.ionsList.SetColumnWidth(col, width)
    # ----
    
    
    def makePatternCanvas(self):
        """Make plot canvas and set defalt parameters."""
        
        # init canvas
        self.patternCanvas = mspy.plot.canvas(self, size=(550, 270), style=mwx.PLOTCANVAS_STYLE_PANEL)
        self.patternCanvas.draw(mspy.plot.container([]))
        
        # set default params
        self.patternCanvas.setProperties(xLabel='m/z')
        self.patternCanvas.setProperties(yLabel='a.i.')
        self.patternCanvas.setProperties(showLegend=True)
        self.patternCanvas.setProperties(showPosBar=False)
        self.patternCanvas.setProperties(showIntBar=True)
        self.patternCanvas.setProperties(intBarHeight=6)
        self.patternCanvas.setProperties(showGel=False)
        self.patternCanvas.setProperties(showCurTracker=True)
        self.patternCanvas.setProperties(checkLimits=False)
        self.patternCanvas.setProperties(autoScaleY=True)
        self.patternCanvas.setProperties(overlapLabels=False)
        self.patternCanvas.setProperties(xPosDigits=5)
        self.patternCanvas.setProperties(yPosDigits=2)
        self.patternCanvas.setProperties(distanceDigits=5)
        self.patternCanvas.setProperties(reverseDrawing=True)
        self.patternCanvas.setLMBFunction('xDistance')
        
        axisFont = wx.Font(config.spectrum['axisFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        self.patternCanvas.setProperties(axisFont=axisFont)
        
        self.patternCanvas.draw(mspy.plot.container([]))
    # ----
    
    
    def onClose(self, evt):
        """Hide this frame."""
        
        self.parent.updateTmpSpectrum(None)
        self.Destroy()
    # ----
    
    
    def onToolSelected(self, evt=None, tool=None):
        """Selected tool."""
        
        # get the tool
        if evt != None:
            tool = 'summary'
            if evt and evt.GetId() == ID_masscalcSummary:
                tool = 'summary'
            elif evt and evt.GetId() == ID_masscalcIonSeries:
                tool = 'ionseries'
            elif evt and evt.GetId() == ID_masscalcPattern:
                tool = 'pattern'
        
        # set current tool
        self.currentTool = tool
        
        # hide panels
        self.mainSizer.Hide(1)
        self.mainSizer.Hide(2)
        self.mainSizer.Hide(3)
        
        # hide save button
        self.toolbar.Hide(7)
        self.toolbar.Hide(8)
        
        # set icons off
        self.summary_butt.SetBitmapLabel(images.lib['masscalcSummaryOff'])
        self.ionseries_butt.SetBitmapLabel(images.lib['masscalcIonSeriesOff'])
        self.pattern_butt.SetBitmapLabel(images.lib['masscalcPatternOff'])
        
        # set panel
        if tool == 'summary':
            self.SetTitle("Compound Summary")
            self.mainSizer.Show(1)
            self.summary_butt.SetBitmapLabel(images.lib['masscalcSummaryOn'])
            self.save_butt.Enable(False)
            
        elif tool == 'ionseries':
            self.SetTitle("Ion Series")
            self.mainSizer.Show(2)
            self.ionseries_butt.SetBitmapLabel(images.lib['masscalcIonSeriesOn'])
            self.save_butt.Enable(False)
            
        elif tool == 'pattern':
            self.SetTitle("Isotopic Pattern")
            self.mainSizer.Show(3)
            self.toolbar.Show(7)
            self.toolbar.Show(8)
            self.pattern_butt.SetBitmapLabel(images.lib['masscalcPatternOn'])
            self.patternCollapse_butt.SetBitmapLabel(images.lib['arrowDown'])
            self.save_butt.Enable(True)
        
        # fit layout
        self.SetMinSize((-1,-1))
        self.mainSizer.Fit(self)
        self.Layout()
        size = self.GetSize()
        self.SetSize((size[0]+1,size[1]))
        self.SetSize(size)
        self.SetMinSize(size)
    # ----
    
    
    def onCollapse(self, evt):
        """Show / hide isotopic pattern panel."""
        
        # Show / hide panel
        if self.patternSizer.IsShown(1):
            self.patternSizer.Hide(1)
            self.patternCollapse_butt.SetBitmapLabel(images.lib['arrowRight'])
        else:
            self.patternSizer.Show(1)
            self.patternCollapse_butt.SetBitmapLabel(images.lib['arrowDown'])
        
        # fit layout
        self.SetMinSize((-1,-1))
        self.mainSizer.Fit(self)
        self.Layout()
        size = self.GetSize()
        self.SetSize((size[0]+1,size[1]))
        self.SetSize(size)
        self.SetMinSize(size)
    # ----
    
    
    def onCompoundChanged(self, evt=None):
        """Recalc all if compound changed."""
        
        # get all params
        if not self.getParams():
            self.currentCompound = None
            self.currentIons = None
            self.currentIon = None
            self.currentProfile = None
            self.clearSummary()
            self.clearPatternCanvas()
            self.updateIonsList()
            return
        
        # calculate summary
        self.updateSummary()
        
        # calculate ion series
        self.updateIonSeries()
        
        # calculate pattern
        self.updatePattern()
    # ----
    
    
    def onIonseriesChanged(self, evt):
        """Recalc ion series and pattern if params changed."""
        
        # get all params
        if not self.getParams():
            self.currentIons = None
            self.currentIon = None
            self.updateIonsList()
            return
        
        # calculate ion series
        self.updateIonSeries()
        
        # calculate pattern
        self.updatePattern()
    # ----
    
    
    def onPatternChanged(self, evt):
        """Recalc pattern if params changed."""
        
        # get all params
        if not self.getParams():
            self.clearPatternCanvas()
            return
        
        # use same X range
        xAxis = self.patternCanvas.getXCurrentRange()
        if xAxis == (0,1):
            xAxis = None
        
        # calculate pattern
        self.updatePattern(xAxis)
    # ----
    
    
    def onIonSelected(self, evt):
        """Show selected ion in the spectrum."""
        
        # get selected ion
        self.currentIon = self.currentIons[evt.GetData()]
        
        # recalculate pattern
        self.updatePattern()
        
        # show current ion in the spectrum
        self.parent.updateMassPoints([self.currentIon[1], self.currentIon[2]])
    # ----
    
    
    def onListKey(self, evt):
        """Export list if Ctrl+C."""
        
        # get key
        key = evt.GetKeyCode()
        
        # copy
        if key == 67 and evt.CmdDown():
            self.ionsList.copyToClipboard()
            
        # other keys
        else:
            evt.Skip()
    # ----
    
    
    def onSave(self, evt):
        """Save current pattern as doument."""
        
        # check data
        if self.currentProfile == None or self.currentCompound == None:
            wx.Bell()
            return
        
        # get ion
        if self.currentIon != None:
            charge = self.currentIon[3]
            ion = self.currentIon[4]
        else:
            charge = 0
            ion = '[M]'
        
        # make document
        document = doc.document()
        document.dirty = True
        document.title = '%s %s' % (self.currentCompound.formula(), ion)
        document.spectrum = self.currentProfile
        
        # make annotations
        mass = self.currentCompound.mass()
        mz = self.currentCompound.mz(charge=charge, agentFormula=config.masscalc['ionseriesAgent'], agentCharge=config.masscalc['ionseriesAgentCharge'])
        document.annotations.append(doc.annotation(label='monoisotopic m/z', mz=mz[0], intensity=config.masscalc['patternIntensity'], baseline=config.masscalc['patternBaseline']))
        document.annotations.append(doc.annotation(label='average m/z', mz=mz[1], intensity=config.masscalc['patternIntensity'], baseline=config.masscalc['patternBaseline']))
        
        # make notes
        document.notes = 'Theoretical isotopic pattern.\n'
        document.notes += 'FWHM: %s\n\n' % (config.masscalc['patternFwhm'])
        document.notes += 'Compound: %s\n' % (self.currentCompound.formula())
        document.notes += 'Monoisotopic mass: %s\n' % (mass[0])
        document.notes += 'Average mass: %s\n\n' % (mass[1])
        document.notes += 'Ion: %s\n' % (ion)
        document.notes += 'Monoisotopic m/z: %s\n' % (mz[0])
        document.notes += 'Average m/z: %s\n' % (mz[1])
        
        # add document
        self.parent.onDocumentNew(document=document)
    # ----
    
    
    def setData(self, formula='', charge=None, agentFormula='H', agentCharge=1, fwhm=None, intensity=None, baseline=None):
        """Set formula and charge."""
        
        # clear current data
        self.currentCompound = None
        self.currentIons = None
        self.currentIon = None
        self.currentProfile = None
        
        # check
        if not formula:
            formula = ''
        
        # update values
        self.compound_value.ChangeValue(formula)
        if agentFormula:
            self.ionseriesAgentFormula_value.ChangeValue(agentFormula)
        if agentCharge:
            self.ionseriesAgentCharge_value.ChangeValue(str(agentCharge))
        self.patternShift_value.ChangeValue('0')
        
        if fwhm != None:
            fwhm = max(0.001, fwhm)
            fwhm = min(10, fwhm)
            self.patternFwhm_value.ChangeValue(str(round(fwhm,3)))
        
        if intensity != None:
            intensity = round(intensity)
            if intensity > 10000 or intensity < -10000:
                intensity = '%0.1e' % intensity
            self.patternIntensity_value.ChangeValue(str(intensity))
        
        if baseline != None:
            baseline = round(baseline)
            if baseline > 10000 or baseline < -10000:
                baseline = '%0.1e' % baseline
            self.patternBaseline_value.ChangeValue(str(baseline))
        
        if not charge or charge > 0:
            self.ionseriesPositive_radio.SetValue(True)
        else:
            self.ionseriesNegative_radio.SetValue(True)
        
        try: wx.Yield()
        except: pass
        
        # calculate ions
        self.onCompoundChanged()
        
        # select propper ion
        if self.currentIons and charge != None:
            for x, ion in enumerate(self.currentIons):
                if ion[3] == charge:
                    self.ionsList.SetItemState(x, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
                    break
    # ----
    
    
    def getParams(self):
        """Get all params from dialog."""
        
        # try to get values
        try:
            
            # compound
            compound = self.compound_value.GetValue()
            if not compound:
                return False
            
            # charging agent
            ionseriesAgent = self.ionseriesAgentFormula_value.GetValue()
            if not ionseriesAgent:
                return False
            
            # ionseries
            config.masscalc['ionseriesAgentCharge'] = int(self.ionseriesAgentCharge_value.GetValue())
            if self.ionseriesNegative_radio.GetValue():
                config.masscalc['ionseriesPolarity'] = -1
            else:
                config.masscalc['ionseriesPolarity'] = 1
            
            # pattern
            patternFwhm = float(self.patternFwhm_value.GetValue())
            patternIntensity = float(self.patternIntensity_value.GetValue())
            patternBaseline = float(self.patternBaseline_value.GetValue())
            patternShift = float(self.patternShift_value.GetValue())
        
        except:
            wx.Bell()
            return False
        
        # check compound
        try:
            self.currentCompound = mspy.compound(compound)
        except:
            return False
        
        if not self.currentCompound.validate():
            wx.Bell()
            return False
        
        # check charging agent
        try:
            if ionseriesAgent != 'e':
                agent = mspy.compound(ionseriesAgent)
            config.masscalc['ionseriesAgent'] = ionseriesAgent
        except:
            return False
        
        if ionseriesAgent == 'e':
            config.masscalc['ionseriesAgentCharge'] = -1
            self.ionseriesAgentCharge_value.ChangeValue('-1')
        
        # check pattern values
        if patternFwhm < 0.001 \
            or patternFwhm > 10 \
            or patternIntensity < 1 \
            or patternBaseline >= patternIntensity:
            wx.Bell()
            return False
        else:
            config.masscalc['patternFwhm'] = patternFwhm
            config.masscalc['patternIntensity'] = patternIntensity
            config.masscalc['patternBaseline'] = patternBaseline
            config.masscalc['patternShift'] = patternShift
            return True
    # ----
    
    
    def updateSummary(self):
        """Calculate summary for curent compound."""
        
        # get summary
        formula = self.currentCompound.formula()
        mass = self.currentCompound.mass()
        
        # update gui
        self.summaryFormula_value.SetValue(formula)
        self.summaryMono_value.SetValue(str(round(mass[0],config.main['mzDigits'])))
        self.summaryAverage_value.SetValue(str(round(mass[1],config.main['mzDigits'])))
    # ----
    
    
    def updateIonSeries(self):
        """Calculate ion series for current compound."""
        
        # get neutral mass
        mass = self.currentCompound.mass()
        self.currentIons = [(0, mass[0], mass[1], 0, '[M]')]
        
        # get ions
        i = 0
        while True:
            
            # get charge
            i += 1
            charge = i*config.masscalc['ionseriesPolarity']*abs(config.masscalc['ionseriesAgentCharge'])
            
            # check ion
            if not self.currentCompound.validate(charge=charge, agentFormula=config.masscalc['ionseriesAgent'], agentCharge=config.masscalc['ionseriesAgentCharge']):
                break
            
            # get ion type
            if config.masscalc['ionseriesPolarity'] == 1 and config.masscalc['ionseriesAgentCharge'] > 0:
                iontype = '[M+%d%s] %d+' % (i, config.masscalc['ionseriesAgent'], abs(charge))
            elif config.masscalc['ionseriesPolarity'] == 1:
                iontype = '[M-%d%s] %d+' % (i, config.masscalc['ionseriesAgent'], abs(charge))
            elif config.masscalc['ionseriesAgentCharge'] < 0:
                iontype = '[M+%d%s] %d-' % (i, config.masscalc['ionseriesAgent'], abs(charge))
            else:
                iontype = '[M-%d%s] %d-' % (i, config.masscalc['ionseriesAgent'], abs(charge))
            
            # get mz
            mz = mspy.mz(mass, charge=charge, agentFormula=config.masscalc['ionseriesAgent'], agentCharge=config.masscalc['ionseriesAgentCharge'])
            
            # add to list
            self.currentIons.append((abs(charge), mz[0], mz[1], charge, iontype))
            if config.masscalc['ionseriesAgent'] == 'e':
                break
            
            # check limits
            if mz[0] < 100 or len(self.currentIons) >= 100:
                break
        
        # update list
        self.updateIonsList()
    # ----
    
    
    def updatePattern(self, xAxis=None):
        """Calculate isotopic pattern for current formula."""
        
        self.currentProfile = None
        
        # try to get selected charge
        if self.currentIon != None:
            charge = self.currentIon[3]
            legend = self.currentIon[4]
        else:
            charge = 0
            legend = '[M]'
        
        # make spectra container
        container = mspy.plot.container([])
        
        # check fwhm
        fwhm = min(config.masscalc['patternFwhm'], 0.9)
        
        # get pattern
        try:
            pattern = self.currentCompound.pattern(\
                fwhm=fwhm,\
                relIntThreshold=config.masscalc['patternThreshold'],\
                charge=charge,\
                agentFormula=config.masscalc['ionseriesAgent'],\
                agentCharge=config.masscalc['ionseriesAgentCharge']\
                )
        except:
            pattern = False
        
        # check pattern
        if not pattern:
            self.clearPatternCanvas()
            wx.Bell()
            return
        
        # correct baseline in pattern
        for x in range(len(pattern)):
            pattern[x][1] *= (config.masscalc['patternIntensity'] - config.masscalc['patternBaseline'])
        
        # make peaklist
        peaklist = []
        for isotope in pattern:
            peak = mspy.peak(mz=isotope[0], intensity=isotope[1]+config.masscalc['patternBaseline'], baseline=config.masscalc['patternBaseline'], charge=charge)
            peaklist.append(peak)
        peaklist = mspy.peaklist(peaklist)
        
        # make spectrum
        profile = mspy.profile(pattern, config.masscalc['patternFwhm'])
        profile += numpy.array([0.0, config.masscalc['patternBaseline']])
        
        # make spectrum object
        spectrum = mspy.scan(points=profile, peaks=peaklist)
        labelFont = wx.Font(config.spectrum['labelFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        spectrum = mspy.plot.spectrum(spectrum, \
            legend=legend, \
            spectrumColour=(16,71,185), \
            tickColour=(255,0,0), \
            showPoints=False, \
            showLabels=True, \
            showTicks=True, \
            labelDigits=config.main['mzDigits'], \
            labelBgr=True, \
            labelAngle=90,
            labelFont=labelFont)
        container.append(spectrum)
        
        # make isotops
        for isotope in pattern:
            peak = mspy.gaussian(isotope[0], isotope[1], config.masscalc['patternFwhm'])
            peak += numpy.array([0.0, config.masscalc['patternBaseline']])
            spectrum = mspy.plot.points(peak, \
                lineColour=(50,140,0), \
                showLines=True, \
                showPoints=False, \
                exactFit=True)
            container.append(spectrum)
        
        # update plot canvas
        self.patternCanvas.draw(container, xAxis=xAxis)
        
        # remeber profile
        self.currentProfile = mspy.scan(points=profile, peaks=peaklist)
        
        # update main spectrum canvas
        if self.currentIon != None:
            profile += numpy.array([config.masscalc['patternShift'], 0.0])
            self.parent.updateTmpSpectrum(profile)
        else:
            self.parent.updateTmpSpectrum(None)
    # ----
    
    
    def updateIonsList(self):
        """Update current ions list."""
        
        self.currentIon = None
        
        # clear previous data and set new
        self.ionsList.DeleteAllItems()
        self.ionsList.setDataMap(self.currentIons)
        
        # check data
        if not self.currentIons:
            return
        
        # add new data
        format = '%0.' + `config.main['mzDigits']` + 'f'
        for row, ion in enumerate(self.currentIons):
            
            # format data
            title = ion[4]
            mono = format % (ion[1])
            average = format % (ion[2])
            
            # add data
            self.ionsList.InsertStringItem(row, title)
            self.ionsList.SetStringItem(row, 1, mono)
            self.ionsList.SetStringItem(row, 2, average)
            self.ionsList.SetItemData(row, row)
        
        # sort data
        self.ionsList.sort()
        
        # scroll top
        self.ionsList.EnsureVisible(0)
    # ----
    
    
    def clearSummary(self):
        """Clear summary."""
        
        self.summaryFormula_value.SetValue('')
        self.summaryMono_value.SetValue('')
        self.summaryAverage_value.SetValue('')
    # ----
    
    
    def clearPatternCanvas(self):
        """Clear pattern canvas."""
        
        self.patternCanvas.draw(mspy.plot.container([]))
        self.parent.updateTmpSpectrum(None)
    # ----
    
    

