# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog to show and edit document information.

# load libs
import wx

# load modules
from nucleus import mwx


class mDocInfo(wx.Dialog):
    """Make information dialog."""

    # ----
    def __init__(self, parent, docData):
        wx.Dialog.__init__(self, parent, -1, "Document Information", style=wx.DEFAULT_DIALOG_STYLE)

        self.docData = docData

        # make gui items
        notebook = self.makeNotebook()
        buttons = self.makeButtons()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(notebook, 1, wx.EXPAND|wx.ALL, 20)
            mainSizer.Add(buttons, 0, wx.BOTTOM|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(notebook, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 5)

        self.SetSizer(mainSizer)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.Centre()
    # ----


    # ----
    def makeNotebook(self):
        """ Make information notebook. """

        # make notebook
        notebook = wx.Notebook(self, -1, size=(210, 210))

        # description
        self.general = generalPage(notebook, self.docData)
        notebook.AddPage(self.general, "Document Info")

        # notes
        self.notes = notesPage(notebook, self.docData)
        notebook.AddPage(self.notes, "Notes")

        return notebook
    # ----


    # ----
    def makeButtons(self):
        """ Make main buttons. """

        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        buttonBox.Add(OK_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        return buttonBox
    # ----


    # ----
    def getDialogData(self):
        """ Get data from each notebook page. """

        data = self.general.getData()
        data['notes'] = self.notes.getData()
        return data
    # ----


class generalPage(wx.Panel):
    """General description"""

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        # get user info
        date = document.getDescription('date')
        operator = document.getDescription('operator')
        contact = document.getDescription('contact')
        institution = document.getDescription('institution')
        instrument = document.getDescription('instrument')

        # get spectrum info
        specRange = document.getSpectrumXRange()
        specLength = document.getSpectrumLength()
        if specLength:
            startPoint = int(round(specRange[0]))
            endPoint = int(round(specRange[1]))
        else:
            startPoint = ' --- '
            endPoint = ' --- '
            specLength = ' --- '
        peaklist = document.getPeaklistLength()

        # get sequence info
        seqLength = document.getSequenceLength()
        modifications = document.getModificationsCount()
        if not seqLength:
            seqLength = ' --- '
            modifications = ' --- '

        # make items
        description = self.makeDescriptionBox(date, operator, contact, institution, instrument)
        spectrum = self.makeSpectrumBox(specLength,startPoint, endPoint, peaklist)
        sequence = self.makeSequenceBox(seqLength, modifications)

        # pack main frame
        if wx.Platform == '__WXMAC__':
            margin = 10
        else:
            margin = 5

        row2Sizer = wx.BoxSizer(wx.HORIZONTAL)
        row2Sizer.Add(spectrum, 1, wx.EXPAND|wx.LEFT|wx.RIGHT, margin)
        row2Sizer.Add(sequence, 1, wx.EXPAND|wx.LEFT|wx.RIGHT, margin)

        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(description, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, margin)
        mainSizer.Add(row2Sizer, 0, wx.EXPAND|wx.TOP|wx.BOTTOM, margin)

        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeDescriptionBox(self, date, operator, contact, institution, instrument):
        """ Box for document description. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Description"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        date_label = wx.StaticText(self, -1, "Date: ")
        self.date_value = wx.TextCtrl(self, -1, date, size=(270, -1))

        operator_label = wx.StaticText(self, -1, "Operator: ")
        self.operator_value = wx.TextCtrl(self, -1, operator, size=(270, -1))

        contact_label = wx.StaticText(self, -1, "Contact: ")
        self.contact_value = wx.TextCtrl(self, -1, contact, size=(270, -1))

        institution_label = wx.StaticText(self, -1, "Institution: ")
        self.institution_value = wx.TextCtrl(self, -1, institution, size=(270, -1))

        instrument_label = wx.StaticText(self, -1, "Instrument: ")
        self.instrument_value = wx.TextCtrl(self, -1, instrument, size=(270, -1))

        # pack items
        grid.Add(date_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.date_value, (0, 1))
        grid.Add(operator_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.operator_value, (1, 1))
        grid.Add(contact_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.contact_value, (2, 1))
        grid.Add(institution_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.institution_value, (3, 1))
        grid.Add(instrument_label, (4, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.instrument_value, (4, 1))
        mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeSpectrumBox(self, length, startPoint, endPoint, peaklist):
        """ Box for spectrum info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Spectrum Info"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        startPoint_label = wx.StaticText(self, -1, "Start point: ")
        startPoint_value = wx.TextCtrl(self, -1, str(startPoint), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        endPoint_label = wx.StaticText(self, -1, "End point: ")
        endPoint_value = wx.TextCtrl(self, -1, str(endPoint), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        length_label = wx.StaticText(self, -1, "Points: ")
        length_value = wx.TextCtrl(self, -1, str(length), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        peaklist_label = wx.StaticText(self, -1, "Peaks: ")
        peaklist_value = wx.TextCtrl(self, -1, str(peaklist), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        # pack items
        grid.Add(startPoint_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(startPoint_value, (0, 1))
        grid.Add(endPoint_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(endPoint_value, (1, 1))
        grid.Add(length_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(length_value, (2, 1))
        grid.Add(peaklist_label, (3, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(peaklist_value, (3, 1))
        mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeSequenceBox(self, length, modifications):
        """ Box for sequence info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Sequence Info"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        length_label = wx.StaticText(self, -1, "Amino acids: ")
        length_value = wx.TextCtrl(self, -1, str(length), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        modifications_label = wx.StaticText(self, -1, "Modifications: ")
        modifications_value = wx.TextCtrl(self, -1, str(modifications), size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)

        # pack items
        grid.Add(length_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(length_value, (0, 1))
        grid.Add(modifications_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(modifications_value, (1, 1))
        mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def getData(self):
        """ Get data from dialog. """

        data = {}
        data['date'] = self.date_value.GetValue()
        data['operator'] = self.operator_value.GetValue()
        data['contact'] = self.contact_value.GetValue()
        data['institution'] = self.institution_value.GetValue()
        data['instrument'] = self.instrument_value.GetValue()

        return data
    # ----


class notesPage(wx.Panel):
    """Document notes"""

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        # get notes
        notes = document.getDescription('notes')

        # make items
        self.notes_value = wx.TextCtrl(self, -1, notes, size=(300, 200), style=wx.TE_MULTILINE)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.notes_value, 1, wx.EXPAND|wx.ALL, 10)
        else:
            mainSizer.Add(self.notes_value, 1, wx.EXPAND|wx.ALL, 5)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def getData(self):
        """ Get data from dialog. """

        return self.notes_value.GetValue()
    # ----
