/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "xine_engine.h"
#include "application.h"
#include "exception.h"
#include "log.h"
#include "exception_handler.h"
#include "gdk_lock.h"
#include "mutex_lock.h"

#include <libgnome/libgnome.h>
#include <xine/xineutils.h>
#include <math.h>
#include <string.h>

class XineException : public Exception
{
private:
	String create_message(xine_stream_t* stream, const String& message)
	{
		String result;
		
		int error_code = xine_get_error(stream);
		switch (error_code)
		{
		case XINE_ERROR_NONE:				result = _("No error"); break;
		case XINE_ERROR_NO_INPUT_PLUGIN:	result = _("No input plugin"); break;
		case XINE_ERROR_NO_DEMUX_PLUGIN:	result = _("No demux plugin"); break;
		case XINE_ERROR_DEMUX_FAILED:		result = _("Demux failed"); break;
		case XINE_ERROR_MALFORMED_MRL:		result = _("Malformed URL"); break;
		case XINE_ERROR_INPUT_FAILED:		result = _("Input failed"); break;
		default:							result = _("Unknown xine error"); break;
		}
		
		String m = message;
		m += ": ";
		m += result;

		Log::write(m);
		return m;
	}
	
public:
	XineException(xine_stream_t* stream, const String& message) :
		Exception(create_message(stream, message)) {}
};

void XineEngine::dest_size_cb ( void *data,
	int video_width, int video_height, double video_pixel_aspect,
	int *dest_width, int *dest_height, double *dest_pixel_aspect )
{
	XineEngine* engine = (XineEngine*)data;
	*dest_pixel_aspect = engine->pixel_aspect;

	MutexLock lock(&engine->size_mutex);
	*dest_width        = engine->width;
	*dest_height       = engine->height;
}

void XineEngine::frame_output_cb ( void *data,
	int video_width, int video_height,
	double video_pixel_aspect, int *dest_x, int *dest_y,
	int *dest_width, int *dest_height,
	double *dest_pixel_aspect, int *win_x, int *win_y )
{
	XineEngine* engine = (XineEngine*)data;
	*dest_pixel_aspect = engine->pixel_aspect;

	MutexLock lock(&engine->size_mutex);
	*dest_x            = 0;
	*dest_y            = 0;
	*win_x             = 0;
	*win_y             = 0;
	*dest_width        = engine->width;
	*dest_height       = engine->height;
}

XineEngine::XineEngine ()
{		
	x11_visual_t	vis;
	int				screen;
	double			res_h, res_v;

	xine						= NULL;
	stream						= NULL;
	video_port					= NULL;
	audio_port					= NULL;
	mute_state					= false;
	fifo_output_stream			= NULL;
	current_dual_language_state	= ENGINE_DUAL_LANGUAGE_DISABLE;
	
	g_static_rec_mutex_init(&size_mutex);
	
	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();
	
	fifo_path = configuration.get_string_value("xine.fifo_path");
	if (!IO::FIFO::exists(fifo_path))
	{
		IO::FIFO::create(fifo_path);
	}
	
	String video_driver = configuration.get_string_value("xine.video_driver");
	String audio_driver = configuration.get_string_value("xine.audio_driver");
		
	xine = xine_new();
	
	if (xine == NULL)
	{
		throw Exception(_("Failed to initialise Xine"));
	}
	
	xine_init ( xine );

	display = XOpenDisplay(NULL);
	screen = XDefaultScreen(display);
	
	res_h = (DisplayWidth(display, screen) * 1000 / DisplayWidthMM(display, screen));
	res_v = (DisplayHeight(display, screen) * 1000 / DisplayHeightMM(display, screen));

	pixel_aspect = res_v / res_h;
	if(fabs(pixel_aspect - 1.0) < 0.01)
	{
		pixel_aspect = 1.0;
	}
	
	GtkWidget* event_box_video = application.get_glade().get_widget("event_box_video");
	widget = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(event_box_video), widget);
	gtk_widget_set_double_buffered(widget, false);
	
	gtk_widget_realize(widget);
	gtk_widget_show_all(widget);
	
	window_id = GDK_WINDOW_XID(widget->window);
	
	gtk_widget_add_events(widget, GDK_EXPOSE);
	
	g_signal_connect( G_OBJECT ( widget ), "expose_event", G_CALLBACK ( on_expose ), this );
	g_signal_connect( G_OBJECT ( widget ), "configure-event", G_CALLBACK ( on_size ), this );

	width	= 320;
	height	= 200;
		
	vis.display           = display;
	vis.screen            = screen;
	vis.d                 = window_id;
	vis.dest_size_cb      = dest_size_cb;
	vis.frame_output_cb   = frame_output_cb;
	vis.user_data         = this;

	video_port = xine_open_video_driver ( xine, video_driver.c_str(),
									  XINE_VISUAL_TYPE_X11, ( void * ) &vis );
	if ( video_port == NULL )
	{
		throw Exception(_("Failed to initialise video driver"));
	}

	audio_port = xine_open_audio_driver ( xine , audio_driver.c_str(), NULL );
	if ( audio_port == NULL)
	{
		throw Exception(_("Failed to initialise audio driver"));
	}

	stream	= xine_stream_new ( xine, audio_port, video_port );
	
	if (stream == NULL)
	{
		throw XineException(stream, "Failed to create stream");
	}

	xine_gui_send_vo_data ( stream, XINE_GUI_SEND_DRAWABLE_CHANGED, ( void * ) window_id );
	xine_gui_send_vo_data ( stream, XINE_GUI_SEND_VIDEOWIN_VISIBLE, ( void * ) 1 );
	mute(mute_state);
	
	// Set up deinterlacing
	String deinterlace_type = configuration.get_string_value("xine.deinterlace_type");
	if (deinterlace_type == "default")
	{
		xine_set_param( stream, XINE_PARAM_VO_DEINTERLACE, true );
	}
	else if (deinterlace_type == "tvtime")
	{
		xine_post_wire_video_port (xine_get_video_source (stream), video_port);
		
		xine_post_t* plugin = xine_post_init (xine, "tvtime", 0, &audio_port, &video_port);	
		if (plugin == NULL)
		{
			throw Exception("Failed to create tvtime plugin");
		}
		
		xine_post_out_t* plugin_output = xine_post_output (plugin, "video out")
			? : xine_post_output (plugin, "video")
			? : xine_post_output (plugin, xine_post_list_outputs (plugin)[0]);
		if (plugin_output == NULL)
		{
			throw Exception("Failed to get xine plugin output for deinterlacing");
		}
		
		xine_post_in_t* plugin_input = xine_post_input (plugin, "video")
			? : xine_post_input (plugin, "video in")
			? : xine_post_input (plugin, xine_post_list_inputs (plugin)[0]);
		
		if (plugin_input == NULL)
		{
			throw Exception("Failed to get xine plugin input for deinterlacing");
		}

		xine_post_wire (xine_get_video_source (stream), plugin_input);
		xine_post_wire_video_port (plugin_output, video_port);

		xine_set_param( stream, XINE_PARAM_VO_DEINTERLACE, true );
	}
	else if (deinterlace_type == "none")
	{
		// Ignore
	}
	else
	{
		throw Exception("Unknown deinterlace_type: '%s'", deinterlace_type.c_str());
	}
}

XineEngine::~XineEngine()
{
	close();
	
	if (stream != NULL)
	{
		xine_close(stream);
		xine_dispose(stream);
		stream = NULL;
	}

	if (xine != NULL)
	{
		if ( audio_port != NULL )
		{
			xine_close_audio_driver ( xine, audio_port );
		}
		audio_port = NULL;
		
		if ( video_port != NULL )
		{
			xine_close_video_driver ( xine, video_port );
		}
		video_port = NULL;

		xine_exit ( xine );
	
		xine = NULL;
	}
}

void XineEngine::mute(gboolean state)
{
	mute_state = state;
	if (stream != NULL)
	{
		xine_set_param(stream, XINE_PARAM_AUDIO_AMP_LEVEL, state ? 0 : 100);
	}
}

void XineEngine::write(const gchar* buffer, gsize length)
{	
	if (fifo_output_stream != NULL)
	{
		fifo_output_stream->write(buffer, length);
	}
}

void XineEngine::open()
{	
	video_thread = g_thread_create((GThreadFunc)video_thread_function, this, TRUE, NULL);

	if (fifo_output_stream == NULL)
	{
		fifo_output_stream = new IO::Channel(fifo_path, O_WRONLY);
		fifo_output_stream->set_encoding(NULL);
	}
}

gpointer XineEngine::video_thread_function(XineEngine* engine)
{
	TRY;
	
	Log::write(_("Video thread created"));

	String path = "fifo:/" + engine->fifo_path;

	if (engine->stream == NULL)
	{
		throw Exception("Stream is NULL");
	}
	
	Log::write(_("About to open FIFO '%s' ..."), path.c_str());
	if ( !xine_open ( engine->stream, path.c_str() ) )
	{
		if (engine->fifo_output_stream != NULL)
		{
			throw XineException (engine->stream, _("Failed to open video stream"));
		}
	}
	
	if (engine->fifo_output_stream != NULL)
	{
		Log::write(_("About to play from FIFO ..."));
		if ( !xine_play ( engine->stream, 0, 0 ) )
		{
			throw XineException (engine->stream, _("Failed to play video stream."));
		}
		Log::write(_("Playing from FIFO"));
	}
	
	THREAD_CATCH;
	
	return NULL;
}

void XineEngine::close()
{
	if (fifo_output_stream != NULL)
	{
		delete fifo_output_stream;
		fifo_output_stream = NULL;
	}

	if (stream != NULL)
	{
		xine_stop(stream);
	}
}

void XineEngine::set_subtitle_channel(gint channel)
{
	xine_set_param(stream, XINE_PARAM_SPU_CHANNEL, channel);
}

void XineEngine::set_audio_channel(gint channel)
{
	xine_set_param(stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, channel);
}

void XineEngine::expose()
{
//	gint width, height;
	gdk_drawable_get_size(GDK_DRAWABLE(widget->window), &width, &height);
	expose(0, 0, width, height);
}

void XineEngine::expose(gint x, gint y, gint width, gint height)
{
    XExposeEvent expose_event;

    memset(&expose_event, 0, sizeof(XExposeEvent));
    expose_event.x = x;
    expose_event.y = y;
    expose_event.width = width;
    expose_event.height = height;
    expose_event.display = display;
    expose_event.window = window_id;
    xine_port_send_gui_data(video_port, XINE_GUI_SEND_EXPOSE_EVENT, &expose_event);
}

gboolean XineEngine::on_expose(GtkWidget* widget, GdkEventExpose* event, XineEngine* engine)
{
	engine->expose(event->area.x, event->area.y,
				   event->area.width, event->area.height);
	return FALSE;
}

void XineEngine::on_size(GtkWidget* widget, GdkEventConfigure *event, XineEngine* engine)
{
	MutexLock lock(&engine->size_mutex);
	gint width = 0, height = 0;
	gdk_drawable_get_size(GDK_DRAWABLE(widget->window), &width, &height);
	engine->width = width;
	engine->height = height;
}

void XineEngine::set_dual_language_state(gint state)
{
	static xine_post_t* plugin = NULL;
	
	if (current_dual_language_state != state)
	{
		if (state == ENGINE_DUAL_LANGUAGE_DISABLE)
		{
			if (plugin != NULL)
			{
				Log::write(_("Disabling dual language"));
				xine_post_wire_audio_port (xine_get_audio_source (stream), audio_port);
			}
		}
		else
		{
			switch (state)
			{
			case ENGINE_DUAL_LANGUAGE_LEFT: Log::write(_("Enabling dual language for left channel"));  break;
			case ENGINE_DUAL_LANGUAGE_RIGHT: Log::write(_("Enabling dual language for right channel"));  break;
			default:
				throw Exception(_("Unknown dual language state"));
			}
			
			if (plugin == NULL)
			{
				Log::write(_("Creating upmix_mono plugin"));
				xine_post_wire_audio_port(xine_get_audio_source (stream), audio_port);

				plugin = xine_post_init (xine, "upmix_mono", 0, &audio_port, &video_port);
				if (plugin == NULL)
				{
					throw Exception(_("Failed to create upmix_mono plugin"));
				}
				
				xine_post_out_t* plugin_output = xine_post_output (plugin, "audio out")
					? : xine_post_output (plugin, "audio")
					? : xine_post_output (plugin, xine_post_list_outputs (plugin)[0]);
				if (plugin_output == NULL)
				{
					throw Exception(_("Failed to get xine plugin output for upmix_mono"));
				}
				
				xine_post_in_t* plugin_input = xine_post_input (plugin, "audio")
					? : xine_post_input (plugin, "audio in")
					? : xine_post_input (plugin, xine_post_list_inputs (plugin)[0]);
				
				if (plugin_input == NULL)
				{
					throw Exception(_("Failed to get xine plugin input for upmix_mono"));
				}
				
				xine_post_wire (xine_get_audio_source (stream), plugin_input);
				xine_post_wire_audio_port (plugin_output, audio_port);
				
				Log::write(_("upmix_mono plugin created and wired"));
			}
			else
			{
				Log::write(_("upmix_mono plugin already created, using existing"));
			}
			
			int parameter = -1;
			switch (state)
			{
			case ENGINE_DUAL_LANGUAGE_LEFT: parameter = 0; break;
			case ENGINE_DUAL_LANGUAGE_RIGHT: parameter = 1; break;
			}

			Log::write(_("Setting channel on upmix_mono plugin to %d"), parameter);

			const xine_post_in_t *in = xine_post_input (plugin, "parameters");
			const xine_post_api_t* api = (const xine_post_api_t*)in->data;
			const xine_post_api_descr_t* param_desc = api->get_param_descr();
			
			if (param_desc->struct_size != 4)
			{
				throw Exception("ASSERT: parameter size != 4");
			}

			api->set_parameters (plugin, (void*)&parameter);
		}
		
		current_dual_language_state = state;
	}
}
