/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "scheduled_recordings_dialog.h"
#include "application.h"
#include "exception_handler.h"
#include "recording_manager.h"
#include "xml.h"

#define EVENT_DESCRIPTION_LINE_LENGTH 60

#define COLUMN_CHANNEL_NAME		0
#define COLUMN_TYPE				1
#define COLUMN_START_TIME		2
#define COLUMN_DURATION			3
#define COLUMN_DESCRIPTION		4

ScheduledRecordingsTreeViewRowCollection::ScheduledRecordingsTreeViewRowCollection(GtkTreeModel* model, GtkTreeIter* parent_iter)
{
	this->parent_iter = parent_iter;
	this->model = model;
}

void ScheduledRecordingsTreeViewRowCollection::add_row(ScheduledRecordingsTreeViewRow& row)
{
	gtk_tree_store_append(GTK_TREE_STORE(model), &iter, parent_iter);
	row.attach(model, &iter);
	row.update();
}

ScheduledRecordingsDialog::ScheduledRecordingsDialog()
{
	Glade& glade = Application::get_current().get_glade();
	dialog_scheduled_recordings = glade.get_widget("dialog_scheduled_recordings");
	tree_view = new ScheduledRecordingsTreeView(GTK_TREE_VIEW(glade.get_widget("treeview_scheduled_recordings")));

	g_signal_connect( G_OBJECT ( glade.get_widget("button_scheduled_recordings_close") ), "clicked", G_CALLBACK ( on_button_scheduled_recordings_close_clicked ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("button_scheduled_recordings_add") ), "clicked", G_CALLBACK ( on_button_scheduled_recordings_add_clicked ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("button_scheduled_recordings_delete") ), "clicked", G_CALLBACK ( on_button_scheduled_recordings_delete_clicked ), this );
}

void ScheduledRecordingsDialog::on_button_scheduled_recordings_close_clicked(GtkButton* button, ScheduledRecordingsDialog* scheduled_recordings_dialog)
{
	TRY;
	scheduled_recordings_dialog->hide();
	CATCH;
}

void ScheduledRecordingsDialog::on_button_scheduled_recordings_add_clicked(GtkButton* button, ScheduledRecordingsDialog* scheduled_recordings_dialog)
{
	TRY;
	Application::get_current().show_scheduled_recording_dialog();
	scheduled_recordings_dialog->tree_view->load();
	CATCH;
}

void ScheduledRecordingsDialog::on_button_scheduled_recordings_delete_clicked(GtkButton* button, ScheduledRecordingsDialog* scheduled_recordings_dialog)
{
	TRY;
	String description = scheduled_recordings_dialog->tree_view->get_selected_description();
	Application::get_current().get_recording_manager().remove_scheduled_recording(description);
	scheduled_recordings_dialog->tree_view->load();
	CATCH;
}

String ScheduledRecordingsTreeView::get_selected_description()
{
	gchar* description = NULL;
	GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
	GtkTreeIter iter;
	GtkTreeModel* model = NULL;
	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		throw Exception(_("A scheduled recording has not been selected"));
	}
	
	gtk_tree_model_get(model, &iter, COLUMN_DESCRIPTION, &description, -1);
	return description;
}

void ScheduledRecordingsDialog::show()
{
	Application& application = Application::get_current();
	tree_view->load();
	gtk_dialog_run(GTK_DIALOG(dialog_scheduled_recordings));
	hide();
	application.check_scheduled_recordings();
}

void ScheduledRecordingsDialog::hide()
{
	gtk_widget_hide(dialog_scheduled_recordings);
}

ScheduledRecordingsTreeViewRow::ScheduledRecordingsTreeViewRow()
{
	model		= NULL;
	iter.stamp	= 0;
}

String ScheduledRecordingsTreeViewRow::get_tree_view_column_text(gchar* s)
{
	String result = (s == NULL) ? "" : s;
	g_free(s);
	return result;
}

void ScheduledRecordingsTreeViewRow::attach(GtkTreeModel* model, GtkTreeIter* iter)
{
	this->model = model;
	this->iter = *iter;
}

void ScheduledRecordingsTreeViewRow::update()
{
	if (model == NULL)
	{
		throw Exception("Row was not attached to a table");
	}
	
	String type = scheduled_recording.get_type_text();
	String start_time = scheduled_recording.get_start_time_text();
	String duration = scheduled_recording.get_duration_text();

	gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
		COLUMN_CHANNEL_NAME,	scheduled_recording.channel_name.c_str(),
		COLUMN_TYPE,			type.c_str(),
		COLUMN_START_TIME,		start_time.c_str(),
		COLUMN_DURATION,		duration.c_str(),
		COLUMN_DESCRIPTION,		scheduled_recording.description.c_str(),
		-1);
}

ScheduledRecordingsTreeView::ScheduledRecordingsTreeView(GtkTreeView* tree_view_epg) : tree_view(tree_view_epg)
{
	GtkTreeStore* tree_store = gtk_tree_store_new(5,
		G_TYPE_STRING,	// COLUMN_CHANNEL_NAME
		G_TYPE_STRING,	// COLUMN_TYPE
		G_TYPE_STRING,	// COLUMN_START_TIME
		G_TYPE_STRING,	// COLUMN_DURATION
		G_TYPE_STRING	// COLUMN_DESCRIPTION
		);
	
	gtk_tree_view_set_model(tree_view_epg, GTK_TREE_MODEL(tree_store));
	GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view_epg);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

	GtkCellRenderer* text_renderer			= gtk_cell_renderer_text_new();
	GtkTreeViewColumn* column_channel		= gtk_tree_view_column_new_with_attributes (_("Channel"),		text_renderer, "text", COLUMN_CHANNEL_NAME,	NULL);
	GtkTreeViewColumn* column_type			= gtk_tree_view_column_new_with_attributes (_("Type"),			text_renderer, "text", COLUMN_TYPE,			NULL);
	GtkTreeViewColumn* column_start_time	= gtk_tree_view_column_new_with_attributes (_("Start Time"),	text_renderer, "text", COLUMN_START_TIME,	NULL);
	GtkTreeViewColumn* column_duration		= gtk_tree_view_column_new_with_attributes (_("Duration"),		text_renderer, "text", COLUMN_DURATION,		NULL);
	GtkTreeViewColumn* column_description	= gtk_tree_view_column_new_with_attributes (_("Description"),	text_renderer, "text", COLUMN_DESCRIPTION,	NULL);

	gtk_tree_view_column_set_resizable(column_description, true);
	gtk_tree_view_column_set_max_width(column_description, 1000);
	
	gtk_tree_view_append_column(tree_view_epg, column_channel);
	gtk_tree_view_append_column(tree_view_epg, column_type);
	gtk_tree_view_append_column(tree_view_epg, column_start_time);
	gtk_tree_view_append_column(tree_view_epg, column_duration);
	gtk_tree_view_append_column(tree_view_epg, column_description);

	g_signal_connect ( G_OBJECT ( tree_view ), "row-activated", G_CALLBACK ( on_tree_view_row_activated ), this );
	
	gtk_tree_view_set_search_column(tree_view, COLUMN_DESCRIPTION);
}

void ScheduledRecordingsTreeView::clear()
{
	GtkTreeModel* tree_model = gtk_tree_view_get_model(tree_view);
	gtk_tree_store_clear(GTK_TREE_STORE(tree_model));
}

void ScheduledRecordingsTreeView::load()
{
	RecordingManager& recording_manager = Application::get_current().get_recording_manager();
	XPathResult scheduled_recordings(recording_manager.get_scheduled_recordings());
	clear();	
	int count = scheduled_recordings.get_count();
	for (int i = 0; i < count; i++)
	{
		ScheduledRecordingsTreeViewRow row;
		row.scheduled_recording.load(scheduled_recordings.get_result(i));
		add_row(row);
	}
}

void ScheduledRecordingsTreeView::add_row(ScheduledRecordingsTreeViewRow& row)
{
	ScheduledRecordingsTreeViewRowCollection rows = get_rows();
	return rows.add_row(row);
}

void ScheduledRecordingsTreeView::on_tree_view_row_activated(
		GtkTreeView*					tree_view,
		GtkTreePath*					path,
		GtkTreeViewColumn*				column,
		ScheduledRecordingsTreeView*	scheduled_recordings_tree_view)
{
	TRY;
	GtkTreeIter iter;
	GtkTreeModel* model = gtk_tree_view_get_model(scheduled_recordings_tree_view->tree_view);
	gtk_tree_model_get_iter(model, &iter, path);

	const gchar* description = NULL;

	gtk_tree_model_get(model, &iter, COLUMN_DESCRIPTION, &description, -1);

	Application::get_current().show_scheduled_recording_dialog(description);
	scheduled_recordings_tree_view->load();
	CATCH;
}

ScheduledRecordingsTreeViewRowCollection ScheduledRecordingsTreeView::get_rows()
{
	GtkTreeModel* model = gtk_tree_view_get_model(tree_view);
	return ScheduledRecordingsTreeViewRowCollection(model);
}
