/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "dvb_demuxer.h"
#include "exception.h"
#include <libgnome/libgnome.h>
#include <string.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>

DvbDemuxer::DvbDemuxer(const String& device_path)
{
	fd = -1;
	
	if((fd = open(device_path.c_str(),O_RDWR|O_NONBLOCK)) < 0)
	{
		throw SystemException(_("Failed to open demux device"));
	}

	pfd[0].fd = fd;
	pfd[0].events = POLLPRI;
}

DvbDemuxer::~DvbDemuxer()
{
	if (fd != -1)
	{
		stop();
		close(fd);
	}
}

void DvbDemuxer::set_pes_filter(uint16_t pid, dmx_pes_type_t pestype)
{
	struct dmx_pes_filter_params parameters;
	
	parameters.pid     = pid;
	parameters.input   = DMX_IN_FRONTEND;
	parameters.output  = DMX_OUT_TS_TAP;
	parameters.pes_type = pestype;
	parameters.flags   = DMX_IMMEDIATE_START;
	
	if (ioctl(fd, DMX_SET_PES_FILTER, &parameters) < 0)
	{
		throw SystemException(_("Failed to set PES filter"));
	}
}

void DvbDemuxer::set_filter(ushort pid, ushort table_id, ushort mask)
{
	struct dmx_sct_filter_params parameters;

	memset( &parameters, 0, sizeof( dmx_sct_filter_params ) );
	
	parameters.pid = pid;
	parameters.timeout = 0;
	parameters.filter.filter[0] = table_id;
	parameters.filter.mask[0] = mask;
	parameters.flags = DMX_IMMEDIATE_START;

	if (ioctl(fd, DMX_SET_FILTER, &parameters) < 0)
	{
		throw SystemException(_("Failed to set section filter for demuxer"));
	}
}

void DvbDemuxer::set_buffer_size(unsigned int buffer_size)
{
	if (ioctl(fd, DMX_SET_BUFFER_SIZE, buffer_size) < 0)
	{
		throw SystemException(_("Failed to set demuxer buffer size"));
	}
}

int DvbDemuxer::read_data(unsigned char* buffer, size_t length)
{
	int bytes_read = 0;
	
	int poll_result = poll(pfd,1,5000);
	if (poll_result<0)
	{
		throw SystemException(_("Failed to poll for demuxer"));
	}
	
	if (poll_result == 0)
	{
		throw Exception(_("Timeout while polling for demuxer"));
	}
		
	bytes_read = read(fd, buffer, length);
	if (bytes_read == -1)
	{
		throw SystemException(_("Failed to read data from demuxer"));
	}
	
	return bytes_read;
}

void DvbDemuxer::stop()
{
	if (ioctl(fd, DMX_STOP) < 0)
	{
		throw SystemException(_("Failed to stop demuxer"));
	}
}

int DvbDemuxer::get_fd() const
{
	return fd;
}
