/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include <libgnomeui/libgnomeui.h>

#include "application.h"
#include "dvb_dvr.h"
#include "dvb_si.h"
#include "integer.h"
#include "exception_handler.h"
#include "network.h"
#include "mutex_lock.h"
#include "gdk_lock.h"
#include "config.h"
#include "ts2pes.h"
#include "scan_dialog.h"

#define AUTO_SURF_INTERVAL			5
#define POKE_INTERVAL				60
#define RECORDING_CHECK_INTERVAL	30
#define EPG_CHECK_INTERVAL			5
#define EPG_UPDATE_INTERVAL			60
#define MAX_EPG_EVENT_LIST_SIZE		500
#define BUFFER_SIZE					PACKET_SIZE*100 // Don't exceed 64K for UDP

Application* current_application = NULL;
char* configuration_file_argument = NULL;

GOptionEntry option_entries[] = {
	{ "configuration-file", 'c', G_OPTION_FLAG_FILENAME, G_OPTION_ARG_FILENAME,
		&configuration_file_argument, "Configuration file to use", "<configuration file>" },
	{ NULL }
};

Application::Application (int argc, char** argv)
{
	TRY
	
	current_application			= this;
	log_file					= NULL;
	application_quit			= false;
	terminate_threads			= true;
	stream_thread				= NULL;
	epg_thread					= NULL;
	tuner						= NULL;
	main_window					= NULL;
	recording_manager			= NULL;
	recording_file_stream		= NULL;
	network_stream				= NULL;
	error_messages				= NULL;
	auto_surf_iterator			= NULL;
	epg_events					= NULL;
	epg_event_list_size			= 0;
	scheduled_recording_node	= NULL;
	channel_change_complete		= false;

	g_static_rec_mutex_init (&stream_mutex);
	g_static_rec_mutex_init (&configuration_mutex);
	g_static_rec_mutex_init (&error_message_mutex);
	g_static_rec_mutex_init (&epg_mutex);
	
	if (!g_thread_supported ())
	{
		g_thread_init (NULL);
	}
	gdk_threads_init();

	gnet_init();

	GDK_LOCK;
	
	GOptionContext* option_context = g_option_context_new (PACKAGE);
	g_option_context_add_main_entries (option_context, option_entries, NULL);
	gnome_program_init(PACKAGE, VERSION,
		LIBGNOMEUI_MODULE, argc, argv,
		GNOME_PARAM_GOPTION_CONTEXT, option_context,
		GNOME_PARAM_HUMAN_READABLE_NAME, "Me TV",
		GNOME_PROGRAM_STANDARD_PROPERTIES, NULL,
		GNOME_PARAM_NONE);
		
	main_thread = g_thread_self();
	exe_path	= argv[0];
		
	gtk_init_add(on_init, this);
	guint check_error_messages_timeout_id = gdk_threads_add_timeout (1000, on_check_error_messages_timer, this);
	gtk_main();
	g_source_remove(check_error_messages_timeout_id);
	
	CATCH;
}

Application::~Application()
{
	stop();

	if (main_window != NULL)
	{
		delete main_window;
		main_window = NULL;
	}
	
	if (recording_manager != NULL)
	{
		delete recording_manager;
		recording_manager = NULL;
	}
	
	if (tuner != NULL)
	{
		tuner = NULL;
		delete tuner;
	}
}

Application& Application::get_current()
{
	if (current_application == NULL)
	{
		throw Exception(_("Application has not been initialised"));
	}
	
	return *current_application;
}

IO::Channel& Application::get_log_file()
{
	if (log_file == NULL)
	{
		throw Exception(_("Log file has not been initialised"));
	}
	
	return *log_file;
}

gboolean Application::on_init(gpointer data)
{
	((Application*)data)->init();	
	return FALSE;
}

void Application::init()
{
	TRY;
	
	check_scheduled_recordings();

	String application_directory = Application::get_directory();
	if (!IO::Directory::exists(application_directory))
	{
		IO::Directory::create(application_directory);
	}
	
	gchar* exe_dir = g_path_get_dirname (exe_path.c_str());
	exe_directory = exe_dir;
	g_free(exe_dir);
	
	log_file = new IO::Channel::Channel(application_directory + "/me-tv.log", O_CREAT | O_TRUNC | O_WRONLY);
	log_file->set_encoding(NULL);
	
	Log::write("Me TV Version: %s", PACKAGE_VERSION);

	String configuration_file;
	if (configuration_file_argument != NULL)
	{
		configuration_file = configuration_file_argument;
	}
	else
	{
		configuration_file = application_directory + "/me-tv.config";
	}
	configuration.load(configuration_file);
	
	String recording_directory(g_get_home_dir());
	
	configuration.set_default_string_value("dvr_path", "/dev/dvb/adapter0/dvr0");
	configuration.set_default_string_value("demux_path", "/dev/dvb/adapter0/demux0");
	configuration.set_default_string_value("frontend_path", "/dev/dvb/adapter0/frontend0");
	configuration.set_default_boolean_value("mute", false);
	configuration.set_default_string_value("engine_type", "xine");
	configuration.set_default_string_value("recording_directory", recording_directory);
	configuration.set_default_string_value("recording_type", "ts");
	configuration.set_default_int_value("video_position", -1);
	configuration.set_default_string_value("epg_path", get_directory() + "/epg.xml");
	configuration.set_default_string_value("channel_file_path", get_directory() + "/channels.conf");
	configuration.set_default_int_value("x", -1);
	configuration.set_default_int_value("y", -1);
	configuration.set_default_int_value("width", -1);
	configuration.set_default_int_value("height", -1);
	configuration.set_default_int_value("record_extra_before", 5);
	configuration.set_default_int_value("record_extra_after", 10);
	configuration.set_default_int_value("broadcast_port", 2005);
	configuration.set_default_boolean_value("always_on_top", true);
	configuration.set_default_boolean_value("fullscreen", false);
	configuration.set_default_boolean_value("show_controls", true);
	configuration.set_default_boolean_value("show_epg", true);
	configuration.set_default_boolean_value("quit_on_close", true);
	configuration.set_default_boolean_value("start_minimised_in_tray", false);	
	configuration.set_default_string_value("epg_encoding", "auto");
	configuration.set_default_boolean_value("fullscreen_workaround", false);
	configuration.set_default_boolean_value("pid_selection_workaround", false);
	configuration.set_default_int_value("epg_span_hours", 3);
	configuration.set_default_boolean_value("sort_channels_alphabetically", false);
	configuration.set_default_string_value("xine.video_driver", "auto");
	configuration.set_default_string_value("xine.audio_driver", "auto");
	configuration.set_default_string_value("xine.deinterlace_type", "default");
	configuration.set_default_string_value("xine.fifo_path", get_directory() + "/video.fifo");
		
	demux_path	= configuration.get_string_value("demux_path");
	dvr_path	= configuration.get_string_value("dvr_path");

	glade.load();

	String epg_path = configuration.get_string_value("epg_path");
	if (IO::File::exists(epg_path))
	{
		Log::write(_("Loading EPG from '%s'"), epg_path.c_str());
	}
	else		
	{
		Log::write(_("Creating EPG file at '%s'"), epg_path.c_str());
	}
	epg.load(epg_path);
	Log::write(_("EPG file loaded"));
	
	int frontend_type = FE_OFDM;
	String channel_file_path = configuration.get_string_value("channel_file_path");
	String frontend_path = configuration.get_string_value("frontend_path");
	
	if (IO::File::exists(frontend_path))
	{
		tuner = new DvbTuner(frontend_path);
		frontend_type = tuner->get_frontend_type();
		
		if (!IO::File::exists(channel_file_path))
		{
			String message = String::format(
				_("You don't have a channel file at '%s'.  Would you like Me TV to create one for you?"),
				channel_file_path.c_str());
			gint response = show_message_dialog(message.c_str(), GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO);
			if (response == GTK_RESPONSE_YES)
			{
				ScanDialog dialog(frontend_type);
				dialog.show();
			}
		}
	}
	else
	{
		String test_path = get_test_path();
		if (!IO::File::exists(test_path))
		{
			throw Exception(_("No tuner device"));
		}
	}
	
	if (!IO::File::exists(channel_file_path))
	{
		throw Exception(_("There's no channels.conf file at '%s'. Me TV cannot continue."), channel_file_path.c_str());
	}
	
	channel_manager.load(frontend_type, channel_file_path);

	Log::write(_("Channel file loaded"));
	
	if (channel_manager.get_channel_count() == 0)
	{
		throw Exception(_("There are no usable channels in the channels.conf file."));
	}

	recording_manager = new RecordingManager();
	
	main_window = new MainWindow();

	mute(configuration.get_boolean_value("mute"));

	if (configuration.has_value("last_channel"))
	{
		String last_channel_name = configuration.get_string_value("last_channel");
		Channel* last_channel = channel_manager.get_channel(last_channel_name);
		if (last_channel != NULL)
		{
			video_channel_name = last_channel->name;
		}
	}

	if (video_channel_name.is_empty())
	{
		GSList* channels = channel_manager.get_channels();
		Channel* channel = (Channel*)channels->data;
		video_channel_name = channel->name;		
	}
	
	dialog_about = glade.get_widget("dialog_about");
	g_signal_connect( G_OBJECT ( dialog_about ), "delete-event", G_CALLBACK ( gtk_widget_hide ), this );
	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog_about), PACKAGE_VERSION);

	status_icon = new StatusIcon();
	
	if (!configuration.get_boolean_value("start_minimised_in_tray"))
	{
		main_window->show();
	}
	else
	{
		mute(true);
	}

	timeout_id = gdk_threads_add_timeout (1000, on_timer, this);
	
	while (!application_quit)
	{
		TRY;
		gtk_main();
		CATCH;
	}
	
	CATCH;
	
	gtk_main_quit();
}

RecordingManager& Application::get_recording_manager()
{
	if (recording_manager == NULL)
	{
		throw Exception(_("The recording manager has not been initialised"));
	}

	return *recording_manager;
}

void Application::quit()
{
	gboolean want_close = true;
	
	if (recording_manager != NULL &&(recording_manager->has_scheduled_recordings()))
	{
		gint result = show_message_dialog(_("You currently have programs scheduled for recording. "\
			"If Me TV is closed then it will not be able record your scheduled programs. "\
			"Are you sure that you want to exit Me TV?"), GTK_MESSAGE_INFO, GTK_BUTTONS_YES_NO);
		
		want_close = (result == GTK_RESPONSE_YES);
	}
	
	if (want_close)
	{
		application_quit = true;
		g_source_remove(timeout_id);
		stop();
		gtk_main_quit();
	}
}

void Application::check_error_messages()
{
	MutexLock lock(&error_message_mutex);
	
	GSList* iterator = error_messages;
	while (iterator != NULL)
	{
		char* message = (char*)iterator->data;
		show_error_message_dialog(message);
		g_free(message);
		iterator = g_slist_next(iterator);
		error_messages = g_slist_delete_link(error_messages, error_messages);
	}
}

void Application::push_epg_event(const EpgEvent& event)
{
	MutexLock lock(&epg_mutex);

	static gboolean message_recorded = false;
	
	if (epg_event_list_size < MAX_EPG_EVENT_LIST_SIZE)
	{
		epg_events = g_slist_prepend(epg_events, new EpgEvent(event));
		epg_event_list_size++;
	}
	else
	{		
		if (!message_recorded)
		{
			message_recorded = true;
			Log::write(_("Too many events in EPG queue"));
		}
	}
}

void Application::check_epg_events()
{
	MutexLock lock(&epg_mutex);
	gboolean update = false;
	
	while (epg_events != NULL)
	{		
		EpgEvent* event = (EpgEvent*)epg_events->data;
			
		if (epg.add_event(*event) > 0)
		{
			update = true;
		}

		delete event;
		epg_events = g_slist_remove_link(epg_events, epg_events);
		epg_event_list_size--;
	}
	
	if (epg.prune() > 0)
	{
		update = true;
	}
	
	epg_event_list_size = 0;

	if (update == true)
	{
		Log::write(_("Saving EPG data"));
		epg.save();
				
		Log::write(_("Updating EPG UI"));
		update_epg();
	}
}

gboolean Application::on_check_error_messages_timer(gpointer data)
{
	TRY
	Application& application = *(Application*)data;
	application.check_error_messages();
	CATCH
		
	return TRUE;
}

gboolean Application::on_timer(gpointer data)
{
	static time_t last_poke_time = 0;
	static time_t last_auto_surf_time = 0;
	static time_t last_epg_check_time = 0;
	static time_t last_epg_update_time = 0;
	static gboolean initialise = true;
	
	Application& application = *(Application*)data;
	MutexLock lock(&application.stream_mutex);
	
	TRY
		
	if (initialise == true)
	{
		initialise = false;
		application.update_epg();
		application.change_channel(application.get_video_channel());
	}
	
	if (application.channel_change_complete)
	{
		application.main_window->channel_change_complete();
		application.channel_change_complete = false;
	}
	
	time_t now = DateTime::now(); 

	application.main_window->on_timer();
	
	if (now - last_poke_time > POKE_INTERVAL)
	{
		if (application.main_window->is_visible() && !application.main_window->is_minimised()) 		
		{ 		
			application.main_window->send_fake_key();
		}
		last_poke_time = now;
	}

	if (now - application.last_recording_check_time > RECORDING_CHECK_INTERVAL)
	{
		xmlNodePtr node = application.get_recording_manager().check();
		
		if (node != NULL)
		{
			if (node != application.scheduled_recording_node)
			{
				application.scheduled_recording_node = node;
				ScheduledRecording scheduled_recording(node);
				Log::write(_("Scheduled recording for '%s' found"), scheduled_recording.description.c_str());
				Channel& channel = application.get_channel(scheduled_recording.channel_name);
				application.start_record_force(channel, scheduled_recording.description);
			}
		}
		else
		{
			if (application.is_recording() && application.scheduled_recording_node != NULL)
			{
				application.scheduled_recording_node = NULL;
				application.record(false);
			}
		}
		
		application.last_recording_check_time = now;
	}
	
	if ((application.auto_surf_iterator != NULL) && (now - last_auto_surf_time > AUTO_SURF_INTERVAL))
	{
		Channel& channel = *(Channel*)application.auto_surf_iterator->data;
		application.change_channel(channel, true);
		application.auto_surf_iterator = g_slist_next(application.auto_surf_iterator);
		
		if (application.auto_surf_iterator == NULL)
		{
			application.main_window->auto_surf(false);
			application.change_channel(application.auto_surf_return_channel);
		}
		
		last_auto_surf_time = now;
	}
	
	if (application.tuner != NULL)
	{
		if (now - last_epg_check_time > EPG_CHECK_INTERVAL)
		{
			application.check_epg_events();
			last_epg_check_time = now;
		}
	}
	
	if (now - last_epg_update_time > EPG_UPDATE_INTERVAL)
	{
		application.update_epg();
		last_epg_update_time = now;
	}

	CATCH;

	return TRUE;
}

void Application:: start_epg_thread()
{
	MutexLock lock(&stream_mutex);

	if (tuner != NULL)
	{
		if (epg_thread == NULL)
		{
			epg_thread = g_thread_create((GThreadFunc)epg_thread_function, this, TRUE, NULL);
			if (epg_thread == NULL)
			{			
				throw SystemException(_("Failed to create EPG thread"));
			}
			Log::write(_("EPG thread created"));
		}
	}
}

void Application::start()
{
	MutexLock lock(&stream_mutex);
	
	terminate_threads = false;

	if (main_window != NULL && main_window->get_engine() != NULL)
	{
		main_window->get_engine()->open();
	}
	
	stream_thread = g_thread_create((GThreadFunc)stream_thread_function, this, TRUE, NULL);
	if (stream_thread == NULL)
	{
		throw SystemException(_("Failed to create stream thread"));
	}
	Log::write(_("Stream thread created"));
	
	start_epg_thread();
}

void Application::stop()
{
	terminate_threads = true;
	
	MutexLock lock(&stream_mutex);
	record(false);
	
	if (epg_thread != NULL)
	{
		g_thread_join(epg_thread);
		epg_thread = NULL;
	}
	
	if (tuner != NULL && !video_channel_name.is_empty())
	{
		Log::write(_("Purging remaining EPG events"));
		check_epg_events();
		Log::write(_("EPG events purged"));
	}

	if (stream_thread != NULL)
	{
		SCOPE_LOG(_("Stopping stream thread"));
		g_thread_join(stream_thread);
		stream_thread = NULL;
	}
	
	if (main_window != NULL && main_window->get_engine() != NULL)
	{
		main_window->get_engine()->close();
	}
}

String Application::get_video_event_title()
{
	String result;
	const Channel& channel = get_video_channel();
	xmlNodePtr node = epg.get_current_event(channel);
	if (node != NULL)
	{
		EpgEvent epg_event(node);
		result = epg_event.get_title();
	}
	
	return result;
}

const String& Application::get_video_channel_name() const
{
	return video_channel_name;
}

Channel& Application::get_channel(const String& channel_name)
{
	Channel* channel = channel_manager.get_channel(channel_name);
	if (channel == NULL)
	{
		throw Exception(_("Channel '%s' does not exist"), channel_name.c_str());
	}
	return *channel;
}

Channel& Application::get_video_channel()
{
	if (video_channel_name.is_empty())
	{
		throw Exception(_("No current channel is selected for viewing"));
	}

	return get_channel(video_channel_name);
}

void Application::change_channel (const String& channel_name, gboolean keep_auto_surf)
{
	change_channel(get_channel(channel_name), keep_auto_surf);
}

gboolean Application::check_main_thread(const String& caller)
{
	gboolean is_main = (main_thread == g_thread_self());
	if (!is_main)
	{
		String message = String::format(_("The call to '%s' is not in the main thread."), caller.c_str());
		push_error_message(message);
	}
	
	return is_main;
}

void Application::change_channel (Channel& channel, gboolean keep_auto_surf)
{
	Log::write(_("Request to change channel to '%s'"), channel.name.c_str());

	if (keep_auto_surf == false)
	{
		auto_surf_iterator = NULL;
	}
	
	MutexLock lock(&stream_mutex);
	
	if (video_channel_name == channel.name && terminate_threads == false)
	{
		Log::write(_("Already tuned to '%s'"), channel.name.c_str());
	}
	else
	{
		bool want_change = true;
		bool same_transponder = false;
		
		if (recording_file_stream != NULL && !same_transponder)
		{
			gint response = show_message_dialog(
				_("You have to stop recording to change to this channel. Do you still want to change channels?"),
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_YES_NO);
			
			if (response != GTK_RESPONSE_YES)
			{
				want_change = false;
				Log::write(_("User cancelled channel change"));
			}
		}

		if (want_change)
		{
			Log::write(_("Changing channel to '%s'"), channel.name.c_str());
			
			stop();
			
			if (tuner != NULL)
			{
				Transponder& transponder = channel.get_transponder();
				Log::write(_("Tuning to transponder at %d ..."), transponder.get_frequency());
				tuner->tune_to(transponder);
				Log::write(_("Tuned to transponder at %d"), transponder.get_frequency());
			}
			video_channel_name = channel.name;
			configuration.set_string_value("last_channel", channel.name);

			start();

			main_window->change_channel_start();
			
			Log::write(_("Channel changed to '%s'"), channel.name.c_str());
		}
	}
}

void Application::toggle_mute()
{
	mute(!configuration.get_boolean_value("mute"));
}

void Application::broadcast(gboolean state)
{
	if (!state && network_stream == NULL)
	{
		return;
	}

	if (state && network_stream != NULL)
	{
		return;
	}

	MutexLock lock(&stream_mutex);
	Log::write(state ? _("Broadcast on") : _("Broadcast off"));
	main_window->broadcast(state);

	if (state)
	{
		network_stream = new MulticastSocket("224.0.1.2",
			configuration.get_int_value("broadcast_port"));
	}
	else
	{
		MulticastSocket* filestream = network_stream;
		network_stream = NULL;
		delete filestream;
	}
}

void Application::toggle_auto_surf()
{
	auto_surf(auto_surf_iterator == NULL);
}

void Application::auto_surf(gboolean state)
{
	MutexLock lock(&stream_mutex);
	gboolean in_auto_surf = auto_surf_iterator != NULL;
	
	if (state && in_auto_surf)
	{
		return;
	}
	
	if (!state && !in_auto_surf)
	{
		return;
	}

	Log::write(state ? _("Starting auto surf") : _("Stopping auto surf"));

	if (state && !in_auto_surf)
	{
		if (recording_file_stream != NULL)
		{
			throw Exception(_("Can't auto surf while recording"));
		}
		else
		{
			auto_surf_return_channel = get_video_channel().name;
			auto_surf_iterator = get_channel_manager().get_channels();
			main_window->auto_surf(true);
		}
	}
	
	if (!state && in_auto_surf)
	{
		auto_surf_iterator = NULL;
		main_window->auto_surf(false);
	}
}

void Application::mute(gboolean state)
{
	MutexLock lock(&stream_mutex);
	main_window->mute(state);
	configuration.set_boolean_value("mute", state);
}

void Application::toggle_broadcast()
{
	broadcast(network_stream == NULL);
}

void Application::hide_controls()
{
	main_window->hide_controls();
}

void Application::audio_channel(int channel)
{
	main_window->audio_channel(channel);
}

void Application::subtitle_channel(int channel)
{
	main_window->subtitle_channel(channel);
}

void Application::toggle_record()
{
	record(!is_recording());
}

void Application::start_record(const String& title)
{
	start_record(get_video_channel(), title);
}

String Application::fix_path(const String& path)
{
	String result = path;
	result.canon(G_CSET_A_2_Z G_CSET_a_2_z G_CSET_DIGITS "-_", ' ');	
	return result;
}

void Application::stop_record()
{
	MutexLock lock(&stream_mutex);
	
	gboolean was_scheduled = scheduled_recording_node != NULL;
	
	main_window->record(false);

	if (recording_file_stream != NULL)
	{
		scheduled_recording_node = NULL;
		IO::Channel* filestream = recording_file_stream;
		recording_file_stream = NULL;
		delete filestream;
	}
	
	if (was_scheduled && !application_quit) 		
	{ 		
		gint result = show_message_dialog( 		
			_("You have stopped a scheduled recording.  "\
			"If you do not delete the scheduled recording then it will start again automatically.  "\
			"Would you like to delete this scheduled recording?"), 		
			GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO); 		
		if (result == GTK_RESPONSE_YES) 		
		{ 		
			remove_current_scheduled_recording(); 		
		} 		
	}   	
}

void Application::start_record_force(Channel& channel, const String& title)
{
	if (get_video_channel().name != channel.name)
	{
		if (recording_file_stream != NULL)
		{
			stop_record();
		}
		change_channel(channel);
	}
	start_record(channel, title);
}

void Application::start_record(Channel& channel, const String& title)
{
	MutexLock lock(&stream_mutex);

	if (get_video_channel().name != channel.name)
	{
		throw Exception(_("Failed to record: you are not tuned to the channel that you want to record"));
	}

	DateTime now;
	String date_time = String::format("%d-%02d-%02d %.2d%.2d",
		now.get_year(),
		now.get_month(),
		now.get_day_of_month(),
		now.get_hour(),
		now.get_minute()
		);
	
	String clean_channel_name = fix_path(channel.name);
	String clean_title = fix_path(title);
	
	String filename = configuration.get_string_value("recording_directory");
	filename += "/" + clean_channel_name;
	filename += " " + clean_title;
	filename += " " + date_time;
	filename += ".mpeg";

	Log::write(_("Recording filename is '%s'"), filename.c_str());
	recording_file_name = filename;

	recording_file_stream = new IO::Channel(recording_file_name, O_CREAT | O_WRONLY);
	recording_file_stream->set_encoding(NULL);
	Log::write(_("Created recording file stream"));
	
	main_window->record(true);
	Log::write(_("Recording started"));
}

void Application::record(gboolean state)
{
	if (state && recording_file_stream != NULL)
	{
		return;
	}

	if (!state && recording_file_stream == NULL)
	{
		return;
	}
	
	if (state)
	{
		Log::write(_("Request to start recording"));
	}
	else
	{
		Log::write(_("Request to stop recording"));
	}

	if (recording_file_stream != NULL)
	{
		stop_record();
	}
	
	if (state)
	{		
		Channel& channel = get_video_channel();
		Log::write(_("Current channel being viewed is '%s'"), channel.name.c_str());
		String title;
		{
			xmlNodePtr node = epg.get_current_event(channel);
			if (node == NULL)
			{
				title = _("No current event");
			}
			else
			{
				EpgEvent event(node);
				title = event.get_title();
			}
		}
		start_record(channel, title);
	}
}

const Transponder& Application::get_current_transponder()
{
	return get_video_channel().get_transponder();
}
		
gpointer Application::epg_thread_function(Application* application)
{
	TRY;

	gboolean is_atsc = application->tuner != NULL && application->tuner->get_frontend_type() == FE_ATSC;
	DvbDemuxer demuxer_eit(application->demux_path);
	const Transponder& transponder = application->get_current_transponder();
	demuxer_eit.set_buffer_size(DVB_SECTION_BUFFER_SIZE);
	demuxer_eit.set_filter(EIT_PID, EIT_ID);
	DvbSectionParser parser;
		
	while (!application->terminate_threads)
	{
		try
		{
			DvbEventInformationSection section;

			parser.parse_eis(demuxer_eit, section);
	
			guint service_id = section.service_id;
			
			// For ATSC there's no service ID
			if (is_atsc)
			{
				service_id = 0;
			}
			
			Channel* channel = application->channel_manager.get_channel(transponder.get_frequency(), service_id);
			
			if (channel != NULL)
			{
				for( unsigned int k = 0; section.short_events.size() > k; k++ )
				{
					EpgEvent event(channel, section.short_events[k]);
					application->push_epg_event(event);
				}
			}
		}
		catch(const Exception& ex)
		{
			Log::write(_("Exception in EPG thread: %s"), ex.get_message().c_str());
		}
	}

	THREAD_CATCH;
	
	return NULL;
}

void Application::write(uint8_t* buffer, int size)
{
	if (recording_file_stream != NULL)
	{
		recording_file_stream->write((gchar*)buffer, size);
	}
}

void Application::on_write(uint8_t *buf,  int size, void *priv)
{
	Application::get_current().write(buf, size);
}

void Application::stream_loop(IO::Channel& input, guint audio_pid, guint video_pid)
{
	gsize length = 0;
	gchar buffer[BUFFER_SIZE];
	Engine* engine = main_window->get_engine();
	gboolean record_as_ps = false;
	
	String recording_type = configuration.get_string_value("recording_type");

	Ts2Pes ts2pes(&on_write, audio_pid, video_pid, recording_type == "ps");

	if (recording_type == "ps")
	{
		record_as_ps = true;
		ts2pes.go();
	}
	
	while (!terminate_threads)
	{
		try
		{
			length = input.read(buffer, BUFFER_SIZE);
			
			if (length > 0)
			{
				if (engine != NULL)
				{
					engine->write(buffer, length);
				}

				if (recording_file_stream != NULL)
				{
					if (record_as_ps)
					{
						ts2pes.run((uint8_t*)buffer, length);
					}
					else
					{
						recording_file_stream->write(buffer, length);
					}
				}

				if (network_stream != NULL)
				{
					network_stream->write(buffer, length);
				}
			}
			else
			{
				usleep(100000);
			}
		}
		catch(const Exception& ex)
		{
			Log::write("Exception in %s: %s", __PRETTY_FUNCTION__, ex.get_message().c_str());
		}
	}
}

gpointer Application::stream_thread_function(Application* application)		
{
	TRY;
		
	Channel& channel = application->get_video_channel();

	SCOPE_LOG("Stream thread function");
	if (application->tuner != NULL)
	{
		DvbDvr dvr(application->dvr_path);
		DvbDemuxer demuxer_pat(application->demux_path);
		DvbDemuxer demuxer_pmt(application->demux_path);
		DvbDemuxer demuxer_video(application->demux_path);
		DvbDemuxer demuxer_audio(application->demux_path);
		DvbDemuxer demux_subtitle(application->demux_path);
		DvbDemuxer demux_teletext(application->demux_path);
		DvbDemuxer demux_ac3(application->demux_path);
		DvbSectionParser parser;
		
		gint video_pid = channel.default_video_pid;
		gint audio_pid = channel.default_audio_pid;
		gint ac3_pid = 0;
		gint subtitle_pid = channel.default_subtitle_pid;
		guint subtitle_channel = -2;

		gboolean pid_selection_workaround = application->get_configuration().get_boolean_value("pid_selection_workaround");
		if (!pid_selection_workaround)
		{
			try
			{
				Log::write(_("Trying to auto detect PIDs"));
				
				ProgramAssociationSection pas;
				demuxer_pat.set_filter(PAT_PID, PAT_ID);
				parser.parse_pas(demuxer_pat, pas);
				guint length = pas.program_associations.size();
				gint program_map_pid = -1;
				
				Log::write(_("Found %d associations"), length);
				for (guint i = 0; i < length; i++)
				{
					ProgramAssociation program_association = pas.program_associations[i];
					
					if (program_association.program_number == channel.service_id)
					{
						program_map_pid = program_association.program_map_pid;
					}
				}
				
				if (program_map_pid == -1)
				{
					throw Exception("Failed to find PMT ID for service");
				}
				
				demuxer_pmt.set_filter(program_map_pid, PMT_ID);
				
				ProgramMapSection pms;
				parser.parse_pms(demuxer_pmt, pms);

				demuxer_pmt.set_pes_filter(program_map_pid, DMX_PES_OTHER);
				demuxer_pat.set_pes_filter(PAT_ID, DMX_PES_OTHER);

				if (pms.video_pids.size() > 0)
				{
					video_pid = pms.video_pids[0];
				}
				
				gsize audio_pids_size = pms.audio_pids.size();
				if (audio_pids_size > 0)
				{
					gboolean default_audio_is_available = false;
					
					channel.audio_pids.clear();
					for (guint i = 0; i < audio_pids_size; i++)
					{
						gint pid = pms.audio_pids[i];
						default_audio_is_available |= (pid == audio_pid);
						channel.audio_pids.push_back(pid);
					}

					// If the default is not availablen then pick the first one that is.
					if (!default_audio_is_available)
					{
						audio_pid = pms.audio_pids[0];
					}
				}
				
				gsize subtitle_pids_size = pms.subtitle_pids.size();
				if (subtitle_pids_size > 0)
				{
					gboolean default_subtitle_is_available = false;
					
					channel.subtitle_pids.clear();
					for (guint i = 0; i < subtitle_pids_size; i++)
					{
						gint pid = pms.subtitle_pids[i];
						default_subtitle_is_available |= (pid == subtitle_pid);
						channel.subtitle_pids.push_back(pid);
					}
					
					// If the default is not available then disable subtitles.
					if (!default_subtitle_is_available)
					{
						channel.default_subtitle_pid = 0;
						subtitle_pid = 0;
					}
				}
									
				if (pms.teletext_pids.size() > 0)
				{
					gint teletext_pid = pms.teletext_pids[0];
					demux_teletext.set_pes_filter(teletext_pid, DMX_PES_OTHER);
					Log::write(_("Stream Thread: Set teletext PID filter to %d (0x%X)"), teletext_pid, teletext_pid);
				}
				
				if (pms.ac3_pids.size() > 0)
				{
					ac3_pid = pms.ac3_pids[0];
					demux_subtitle.set_pes_filter(pms.ac3_pids[0], DMX_PES_OTHER);
					Log::write(_("Stream Thread: Set AC3 PID filter to %d (0x%X)"), ac3_pid, ac3_pid);
				}
				
				Log::write(_("PIDs read successfully"));
			}
			catch(const Exception& ex)
			{
				Log::write(_("Failed to set PIDs automatically: %s"), ex.get_message().c_str());
			}
		}

		demuxer_video.set_pes_filter(video_pid, DMX_PES_VIDEO);
		Log::write(_("Stream Thread: Set video PID filter to %d (0x%X)"), video_pid, video_pid);
	
		if (audio_pid != 0)
		{
			demuxer_audio.set_pes_filter(audio_pid, DMX_PES_AUDIO);
			Log::write(_("Stream Thread: Set audio PID filter to %d (0x%X)"), audio_pid, audio_pid);
		}
		else
		{
			audio_pid = ac3_pid;
		}
		
		if (subtitle_pid != 0)
		{
			subtitle_channel = 0;
			demux_subtitle.set_pes_filter(subtitle_pid, DMX_PES_OTHER);
			Log::write(_("Stream Thread: Set subtitle PID filter to %d (0x%X)"), subtitle_pid, subtitle_pid);
		}
		
		application->channel_change_complete = true;
		dvr.set_encoding(NULL);
		application->stream_loop(dvr, audio_pid, video_pid);
	}
	else
	{
		String test_path = application->get_test_path();
		if (IO::File::exists(test_path))
		{
			IO::Channel input(test_path, O_RDONLY);
			input.set_encoding(NULL);
			application->stream_loop(input, channel.default_audio_pid, channel.default_video_pid);
		}
	}

	THREAD_CATCH;
	
	return NULL;
}

String Application::get_directory()
{
	const gchar* home_dir = g_get_home_dir();
	return String::format("%s/.me-tv", home_dir);
}

Channel& Application::get_channel(int frequency, int service_id)
{
	Channel* channel = channel_manager.get_channel(frequency, service_id);
	if (channel == NULL)
	{
		throw Exception(_("Channel (frequency=%d, service_id=%d) does not exist"), frequency, service_id);
	}
	return *channel;
}

void Application::toggle_fullscreen()
{
	main_window->toggle_fullscreen();
}

void Application::toggle_controls()
{
	main_window->toggle_controls();
}

void Application::push_error_message(const String& message)
{
	Log::write(_("Pushing error message onto queue: %s"), message.c_str());

	MutexLock lock(&error_message_mutex);
	char* message_copy = g_strdup(message.c_str());
	error_messages = g_slist_append(error_messages, message_copy);
}

void Application::show_program_details_dialog (const String& channel_name, int program_id)
{
	EpgEvent event(epg.get_event(channel_name, program_id));	
	main_window->show_program_details_dialog(event);
}

void Application::update_epg()
{
	String status_message = get_status_message();
	status_message = "Me TV - " + status_message;
	status_icon->set_title(status_message);
	
	main_window->update_epg();
}

String Application::get_test_path() const
{
	gchar* test_path = g_build_filename(get_directory().c_str(), "test.ts", NULL);
	String result = test_path;
	g_free(test_path);

	return result;
}

void Application::show_scheduled_recording_dialog()
{
	main_window->show_scheduled_recording_dialog();
}

void Application::show_scheduled_recording_dialog(const String& description)
{
	xmlNodePtr node = recording_manager->get_scheduled_recording(description);
	if (node == NULL)
	{
		throw Exception(_("Failed to find scheduled recording with description '%s'.  It may have been deleted."), description.c_str());
	}

	ScheduledRecording scheduled_recording(node);
	main_window->show_scheduled_recording_dialog(scheduled_recording);
}

String Application::get_status_message()
{
	String message = video_channel_name;
	String title = get_video_event_title();
	if (!title.is_empty())
	{
		message += " - ";
		message += title;
	}
	
	return message;
}

void Application::toggle_visibility()
{
	if (main_window->is_visible())
	{
		main_window->hide();
	}
	else
	{
		mute(false);
		main_window->show();
	}
}

void Application::show_help()
{
	GError* error = NULL;
	
	gnome_help_display("me-tv", NULL, &error);
	
	if (error != NULL)
	{
		throw Exception(_("Failed to display help: %s"), error->message);
	}
		
 	if (main_window->get_is_fullscreen())
 	{
 		main_window->show_fullscreen(false);
	}
}

void Application::show_about()
{
	gboolean fullscreen_workaround = configuration.get_boolean_value("fullscreen_workaround");
	gboolean is_fullscreen = main_window->get_is_fullscreen();

	if (fullscreen_workaround && is_fullscreen)
	{
		main_window->maximise();
		main_window->show_fullscreen(false);
	}
	
	gtk_dialog_run(GTK_DIALOG(dialog_about));
	gtk_widget_hide(dialog_about);
	
	if (fullscreen_workaround && is_fullscreen)
	{
		main_window->show_fullscreen(true);
	}
}

gint Application::show_message_dialog(const String& message, GtkMessageType type, GtkButtonsType buttons)
{
	gboolean fullscreen_workaround = false;
	gboolean is_fullscreen = false;
	GtkWindow* parent = NULL;

	if (current_application != NULL && current_application->main_window != NULL)
	{
		fullscreen_workaround = current_application->configuration.get_boolean_value("fullscreen_workaround");
		is_fullscreen = current_application->main_window->get_is_fullscreen();
		parent = GTK_WINDOW(current_application->main_window->get_widget());
	}

	if (fullscreen_workaround && is_fullscreen)
	{
		current_application->main_window->maximise();
		current_application->main_window->show_fullscreen(false);
	}

	GtkWidget* dialog = gtk_message_dialog_new(parent, GTK_DIALOG_MODAL, type, buttons, message.c_str());
	gtk_window_set_title (GTK_WINDOW(dialog), APPLICATION_NAME);
	gint result = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);

	if (fullscreen_workaround && is_fullscreen)
	{
		current_application->main_window->show_fullscreen(true);
	}
	
	return result;
}

gint Application::show_error_message_dialog(const String& message)
{	
	return show_message_dialog(message, GTK_MESSAGE_ERROR);
}

void Application::show_scheduled_recordings_dialog()
{
	main_window->show_scheduled_recordings_dialog();
}

void Application::set_selected_event(xmlNodePtr event)
{
	main_window->set_selected_event(event);
}

void Application::restart_stream()
{
	stop();
	start();
}

MainWindow& Application::get_main_window()
{
	if (main_window == NULL)
	{
		throw Exception(_("Main window has not been initialised"));
	}
	
	return *main_window;
}

gboolean Application::is_scheduled_recording()
{
	MutexLock lock(&stream_mutex);
	return scheduled_recording_node != NULL;
}

void Application::remove_current_scheduled_recording()
{
	MutexLock lock(&stream_mutex);
	ScheduledRecording scheduled_recording(scheduled_recording_node);
	recording_manager->remove_scheduled_recording(scheduled_recording.description);
}

DvbTuner& Application::get_tuner()
{
	if (tuner == NULL)
	{
		throw Exception(_("Tuner has not been initialised"));
	}
	
	return *tuner;
}

void Application::set_dual_language_state(gint state)
{
	Engine* engine = main_window->get_engine();
	if (engine != NULL)
	{
		engine->set_dual_language_state(state);
	}
}
