""" The entry point for an Envisage Workbench application. """


# Tell everything that we're using the workbench plugin!
from enthought.envisage import _set_using_workbench
_set_using_workbench(True)


# Standard library imports.
from logging import INFO
import logging
import os
from ConfigParser import SafeConfigParser

# Enthought library imports.
from enthought.envisage.api import Application
from enthought.pyface.api import GUI, ImageResource, SplashScreen
from enthought.traits.api import Property, Instance, Str

# Local imports.
from services import IWORKBENCH, IWORKBENCH_UI
from workbench import Workbench
from workbench_ui import WorkbenchUI


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class WorkbenchApplication(Application):
    """ The entry point for an Envisage Workbench application.

    i.e. a GUI application whose user interface is provided by the workbench
    plugin.

    """

    # The name of the configuration file section for the splash screen.
    SPLASH_SCREEN = 'enthought.envisage.workbench.splash_screen'

    # Default values for the configuration file.
    CONFIG_DEFAULTS = {
        SPLASH_SCREEN + '.image'         : 'splash',
        SPLASH_SCREEN + '.log_level'     : INFO,
        SPLASH_SCREEN + '.text_location' : (5, 5),
        SPLASH_SCREEN + '.text_color'    : 'black'
    }

    #### 'Application' interface ##############################################

    # Does this application require a GUI?
    #
    # This is a hangover from the first implementation of 'Application' it is
    # deprecated and will be removed in future versions. For now we have to set
    # it to False so that we can create the GUI ourselves.
    requires_gui = False

    #### 'WorkbenchApplication' interface #####################################

    # The name of the application configuration file.
    config = Str('application.ini')

    # The workbench service.
    workbench = Property(Instance(Workbench))

    # The workbench UI service.
    workbench_ui = Property(Instance(WorkbenchUI))

    ###########################################################################
    # 'WorkbenchApplication' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_workbench(self):
        """ Property getter. """

        return self.get_service(IWORKBENCH)

    def _get_workbench_ui(self):
        """ Property getter. """

        return self.get_service(IWORKBENCH_UI)

    #### Methods ##############################################################

    def run(self):
        """ Runs the application.

        This does the following (so you don't have to ;^):-

        1) Starts the application
        2) Starts the GUI event loop
        3) When the event loop terminates, stops the application

        """

        logger.debug('---------- workbench application ----------')

        # Create the GUI (this puts up the splash screen if required).
        if self.gui is None:
            self.gui = self._create_gui()

        # Start the application.
        self.start()

        # Start the GUI event loop.
        #
        # THIS CALL DOES NOT RETURN UNTIL THE GUI IS CLOSED.
        self.gui.start_event_loop()

        # Stop the application.
        self.stop()

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_gui(self):
        """ Creates a GUI """

        # Get the configuration information.
        config = self._load_config(self.config, self.CONFIG_DEFAULTS)

        # Create a GUI and put up the splash screen.
        return GUI(self._create_splash_screen(config))

    def _create_splash_screen(self, config):
        """ Creates a splash screen. """

        image         = config[self.SPLASH_SCREEN + '.image']
        log_level     = config[self.SPLASH_SCREEN + '.log_level']
        text_color    = config[self.SPLASH_SCREEN + '.text_color']
        text_location = config[self.SPLASH_SCREEN + '.text_location']

        splash_screen = SplashScreen(
            image         = ImageResource(image, search_path=[os.getcwd()]),
            log_level     = log_level,
            text_color    = text_color,
            text_location = text_location
        )

        return splash_screen

    def _load_config(self, path, defaults={}):
        """ Loads a config file. """

        config = defaults.copy()

        parser = SafeConfigParser()
        parser.read(path)

        for section in parser.sections():
            for name, value in parser.items(section):
                try:
                    actual = eval(value)

                except:
                    actual = value

                config[section + "." + name] = actual

        return config

#### EOF ######################################################################
