""" A factory for workbench actions. """


# Enthought library imports.
from enthought.pyface.api import ImageResource
from enthought.traits.api import HasTraits

# Local imports.
from workbench_action_proxy import WorkbenchActionProxy


class WorkbenchActionFactory(HasTraits):
    """ A factory for workbench actions.

    The factory creates action implementations from action extensions.

    """

    ###########################################################################
    # 'WorkbenchActionFactory' interface.
    ###########################################################################

    def create_action(self, window, extension):
        """ Creates an action implementation from an action extension.

        Unless the action extension specifies 'lazy_load' as False, a proxy
        action is created that loads the actual action implementation
        on-demand.

        Wherever possible, 'lazy_load' should be True!

        """

        if extension.lazy_load:
            action = self._create_proxy(window, extension)

        else:
            action = self._create_action(window, extension)

        return action

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_action(self, window, extension):
        """ Creates an action implementation. """

        application = window.application

        # The plugin definition that the action was defined in.
        plugin_definition = extension._action_set_._definition_

        # We are about to actually import the action class, so make sure that
        # the plugin that contributed it has been started.
        application.start_plugin(plugin_definition.id)

        # Import the implementation class.
        klass = application.import_symbol(extension.class_name)

        # Create an instance of it.
        action = klass(
            accelerator = extension.accelerator,
            description = extension.description,
            id          = extension.id,
            name        = extension.name,
            style       = extension.style,
            tooltip     = extension.tooltip,
            window      = window
        )

        # We don't set this is in the constructor in case the action uses a
        # static trait handler to listen for changes to the 'checked' trait
        # (which is a common idiom). If we did set it in the constructor then
        # it might fire before the name, style and window traits have been
        # initialized (since we can't guarantee the order of traits
        # initialization via a **kw argument).
        action.checked = extension.checked
        action.enabled = extension.enabled

        # Create the action's image resource relative to the specified path or,
        # if a path isn't specified, to the directory that the plugin
        # definition lives in.
        if extension.image_path is not None and len(extension.image_path) > 0:
            search_path = extension.image_path
        else:
            search_path = [plugin_definition.location]
        action.image = ImageResource(name=extension.image,
            search_path=search_path
        )

        return action

    def _create_proxy(self, window, extension):
        """ Creates an action proxy. """

        # The plugin definition that the action was defined in.
        plugin_definition = extension._action_set_._definition_

        proxy = WorkbenchActionProxy(
            bundle = extension,

            application = window.application,
            accelerator = extension.accelerator,
            checked = extension.checked,
            class_name = extension.class_name,
            defined_in = plugin_definition.id,
            description = extension.description,
            enabled = extension.enabled,
            id = extension.id,
            image_path = extension.image,
            image_search_path = extension.image_path,
            lazy_load = extension.lazy_load,
            name = extension.name,
            style = extension.style,
            tooltip = extension.tooltip,
            window = window,

            # fixme: Hacks!
            enabled_when = extension.enabled_when,
            disabled_when = extension.disabled_when
        )

        return proxy

#### EOF ######################################################################
