""" The base class for all views. """


# Major library imports.
import wx

# Enthought library imports.
from enthought.traits.api import Any, Bool, Enum, Event, HasTraits, List, Str


class View(HasTraits):
    """ The base class for all views. """

    #### 'View' interface #####################################################

    # The toolkit-specific control that represents the view.
    control = Any

    # Does the view currently have the focus?
    has_focus = Bool(False)

    # The view's icon.
    icon = Str

    # The view's unique identifier.
    id = Str

    # The view's name.
    name = Str

    # The default position of the view relative to the workarea.
    position = Enum('left', 'right', 'top', 'bottom',
                    'left top', 'left bottom',
                    'right top', 'right bottom',
                    'top left', 'top right',
                    'bottom left', 'bottom right')

    # The current selection within the view.
    selection = List

    # The workbench window that the view is in.
    window = Any

    # The part pane that the view is in.
    part = Any

    # Whether the view is open
    opened = Bool(False)

    # Event fired when the UI is shutting down. This is NOT fired each time
    # the view is hidden, but only when the UI Plugin (i.e., the application
    # is shutting down. Use this, for example, to save view preferences.
    closing = Event

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    def open(self):
        """ Opens the view. """

        # If we haven't been explicitly attached to a window, we open up in
        # the active workbench window.
        if self.window is None:
            from ui_plugin import UIPlugin
            self.window = UIPlugin.instance.active_window

        if self.part is None:
            self._create()

        self.part.show()

        self.opened = True

        return

    def close(self):
        """ Closes the view. """

        if self.opened:
            self.part.hide()
            self.opened = False

        return

    def _create(self):
        """ Creates the view. """

        # Create a view part for this view.
        self.part = self._create_part()

        # Create the contents of the window.
        self._content = self._create_contents(self.part.control)

        # Set the content of the view part.
        self.part.set_content(self._content)

        # fixme: Quick hack to set the window selection when the view gets
        # focus.
        wx.EVT_SET_FOCUS(self._content, self._on_set_focus)
        wx.EVT_KILL_FOCUS(self._content, self._on_kill_focus)

        return

    def _create_part(self):
        """ Get the control from the containing workbench window. """

        return self.window.create_view(self.name, self.position)

    def _create_contents(self, parent):
        """ Create the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        """
        print '--------------------------------', self
        raise NotImplementedError

    def set_focus():

        return

    ###########################################################################
    # Trait handlers
    ###########################################################################

    def _position_changed(self, new, old):
        # If we're currently visible, remove our content from the current
        # part (at the time of this writing, this actually *removes* a tab
        # for us from the tab control at the old position.)
        if self.opened:
            self.part.hide()

        # If we've ever been opened, then we have a content window that we want
        # to try and preserve to maintain state of this view.  All we want to
        # do is move this content to a new part reflecting our new position.
        if self.part is not None:
            self.part = self._create_part()
            self._content.Reparent(self.part.control)
            self.part.set_content(self._content)

        # If we're supposed to be currently visible, make sure our content is
        # still showing within the new part.
        if self.opened:
            self.part.show()

    def _on_set_focus(self, event):
        """ Called when the view gets focus. """

        self.has_focus = True
        self.window.selection = self.selection

        event.Skip()

        return

    def _on_kill_focus(self, event):
        """ Called when the view loses focus. """

        self.has_focus = False

        event.Skip()
        
        return
    
#### EOF ######################################################################
