""" The Envisage UI plugin. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage import Application, Plugin
from enthought.pyface.api import ImageResource, confirm, YES
from enthought.traits.api import Any, Dict, List, Instance, Property, Str, Tuple

# Local imports.
from workbench_window import WorkbenchWindow
from ui_plugin_definition import LibraryVersions, UIBranding


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class UIPlugin(Plugin):
    """ The Envisage UI plugin. """

    #### Class interface ######################################################

    # The shared plugin instance.
    instance = None

    #### 'UIPlugin' interface #################################################

    # The image displayed in the about box.
    about_image = Instance(ImageResource, ImageResource('about'))

    # The active workbench window (ie. the last one that gained focus).
    active_window = Instance(WorkbenchWindow)

    # The application icon.
    application_icon = Instance(ImageResource, ImageResource('application'))

    # The application name (used for window titles etc).
    application_name = Str('Envisage')

    # Versions of libraries.
    library_versions = List(Str)

    # All open workbench windows.
    windows = Property(List(WorkbenchWindow))

    # The size of toolbar images
    tool_bar_image_size = Tuple

    #### Private interface ####################################################

    # The application that the plugin is part of.
    _application = Instance(Application)

    # Shadow trait for the 'windows' property.
    _windows = Dict(Instance(WorkbenchWindow), Any)

    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **kw):
        """ Creates a new plugin. """

        # Base-class constructor.
        super(UIPlugin, self).__init__(**kw)

        # Set the shared instance.
        UIPlugin.instance = self

        return

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin.

        Can be called manually, but is usually called exactly once when the
        plugin is first required.

        """

        self._application = application

        # Initialize any application branding (application name, icon etc).
        branding = application.extension_registry.get_extensions(UIBranding)
        if len(branding) > 0:
            if len(branding) > 1:
                logger.warn("found more than one branding contribution")

            # If more than one plugin has made a contribution to the branding
            # extension point,  then we just take the first one.
            self._initialize_branding(branding[0])

        library_versions = \
            application.extension_registry.get_extensions(LibraryVersions)
        if len(library_versions) > 0:
            self._initialize_versions(library_versions)

        # Create and open the first workbench window!
        window = self.new_workbench_window()
##         window.open()

        # FIXME: When the application first starts up the active
        # window is not set to the default window.  This breaks
        # scripting from Python.
        self.active_window = window

        # We don't open the window until the application has started all
        # plugins.
        application.on_trait_change(self._on_application_started, 'started')

        return

    def stop(self, application):
        """ Stops the plugin.

        Can be called manually, but is usually called exactly once when the
        application exits.

        """

        # Save the active window's size as the default size.
        self.preferences.set('window_size', self.active_window.size)
        self.preferences.set('window_position', self.active_window.position)

        # Get the visibility of each of the views in the active window and save
        # that as a preference.
        visibility = {}
        for view in self.active_window.views:
            visibility[view.id] = view.opened
        self.preferences.set('view_visibility', visibility)

        # Get the page_layout preferences.
        page_layout = self.active_window.page_layout
        page_layout_preferences = page_layout.get_preferences()
        self.preferences.set('page_layout', page_layout_preferences)

        self.save_preferences()

        return


    ###########################################################################
    # 'UIPlugin' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_windows(self):
        """ Returns all open workbench windows. """

        return self._windows.keys()


    #### Methods ##############################################################

    def new_workbench_window(self):
        """ Creates a new workbench window.

        When all of the workbench windows are closed, the application will
        exit.

        """

        size = self.preferences.get('window_size', (800,600))
        position = self.preferences.get('window_position', (-1, -1))
        page_layout_preferences = self.preferences.get('page_layout', None)
        if page_layout_preferences is None:
            page_layout_preferences = {}

        image_size = (16, 16)
        if self.tool_bar_image_size != ():
            image_size = self.tool_bar_image_size
        window = WorkbenchWindow(
            self, title=self.application_name, icon=self.application_icon,
            position=position, size=size, tool_bar_image_size = image_size
            )

        window.page_layout_preferences = page_layout_preferences

        # Listen for the window being activated/closed.
        window.on_trait_change(self._on_window_activated, 'activated')
        window.on_trait_change(self._on_window_closed, 'closed')

        return window


    def window_can_close(self, window):
        """ Returns True iff the window can close, otherwise False. """

        # If this is the only window left then we may need to prompt the user
        # to make sure that they really want to exit.
        if len(self._windows) == 1:
            can_close = self._confirm_exit(window)
            if can_close:
                return self._application.stop()
        else:
            can_close = True

        return can_close


    ###########################################################################
    # Private interface.
    ###########################################################################

    def _confirm_exit(self, window):
        """ Returns True iff it is Ok to exit the application. """

        if self.preferences.get('prompt_on_exit'):
            answer = confirm(
                window.control,
                "Exit %s?" % self.application_name, "Confirm Exit"
            )
            can_close = answer == YES
        else:
            can_close = True

        return can_close

    def _initialize_branding(self, branding):
        """ Initialize the application branding. """

        # The search path for images starts relative to the plugin definition
        # that made the contribution.
        path = [branding._definition_.location]
        self.about_image      = ImageResource(branding.about_image, path)
        self.application_icon = ImageResource(branding.application_icon, path)
        self.application_name = branding.application_name
        if branding.tool_bar_image_size != ():
            self.tool_bar_image_size = branding.tool_bar_image_size

        return

    def _initialize_versions(self, versions):
        """ Initialize the application version strings. """

        for version in versions:
            self.library_versions.extend(version.versions)

        return

    #### Trait change handlers ################################################

    def _on_application_started(self):
        """ Called when the application has started all plugins. """
        # open the workbench window after all plugins have started.
        self.active_window.open()
        return

    def _on_window_activated(self, window, trait_name, old, new):
        """ Called when a window is activated. """

        if new:
            self.active_window = window

            # We keep track of all of the workbench windows so that we can
            # detect when the user attempts to close the last one open.
            self._windows[window] = None

        return

    def _on_window_closed(self, window, trait_name, old, new):
        """ Called when a window is closed. """

        del self._windows[window]

        return

#### EOF ######################################################################
