""" A view containing an interactive Python shell. """


# Enthought library imports.
from enthought.envisage.ui import View
from enthought.pyface.api import PythonShell

# Plugin definition imports.
from enthought.envisage.ui.python_shell.python_shell_plugin_definition import Namespace


class PythonShellView(View):
    """ A view containing an interactive Python shell. """

    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **kw):
        """ Creates a new plugin. """

        # Base-class constructor.
        super(View, self).__init__(**kw)

        # Make sure that the plugin that we are part of has been started.
        from enthought.envisage.ui.python_shell import PythonShellPlugin
        plugin = PythonShellPlugin.instance

        # Add ourselves to the plugin.
        plugin.views.append(self)
        
        return

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        """

        self.shell = PythonShell(parent)
        self.shell.on_trait_change(self._on_key_pressed, 'key_pressed')

        # fixme: Demo hack.
        self.shell.on_trait_change(
            self._on_command_executed, 'command_executed'
        )
        
        # Namespace contributions.
        application = self.window.plugin.application
        extensions = application.extension_registry.get_extensions(Namespace)
        for extension in extensions:
            # Bindings.
            for name, value in extension.bindings.items():
                self.bind(name, value)

            # Commands.
            for command in extension.commands:
                self.execute_command(command)
            
        self.control = self.shell.control

        from sets import Set
        self.shell._LOCALS = self.control.interp.locals.keys()
        
        return self.control

    ###########################################################################
    # 'PythonShellView' interface.
    ###########################################################################

    def bind(self, name, value):
        """ Binds a name to a value in the interpreter's namespace. """
        
        self.shell.bind(name, value)
        
        return

    def execute_command(self, command):
        """ Execute a command in the interpreter. """
        
        return self.shell.execute_command(command)


    def _on_command_executed(self, shell):
        from enthought.envisage.ui.python_shell import PythonShellPlugin
        plugin = PythonShellPlugin.instance
        plugin.command_executed = shell

        return

    #### trait event handlers #################################################

    def _on_key_pressed(self, event):
        """ Called when a key is pressed. """

        if event.alt_down and event.key_code == 317:
            zoom = self.shell.control.GetZoom()
            if zoom != 20:
                self.shell.control.SetZoom(zoom+1)

        elif event.alt_down and event.key_code == 319:
            zoom = self.shell.control.GetZoom()
            if zoom != -10:
                self.shell.control.SetZoom(zoom-1)

        return
    

#### EOF ######################################################################
