#-------------------------------------------------------------------------------
#  
#  Handle the user exporting the currently selected object to a repository.  
#  
#  Written by: David C. Morrill
#  
#  Date: 04/03/2006
#  
#  (c) Copyright 2006 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

""" Handle the user exporting the currently selected object to a repository.
"""

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from enthought.traits.api \
    import Property
    
from enthought.envisage.workbench.action \
    import SelectionListenerAction
    
from enthought.envisage.repository \
    import IREPOSITORY
    
from enthought.envisage.repository.repository_extensions \
    import ExportableObject
    
from enthought.naming.api \
    import Binding

#-------------------------------------------------------------------------------
#  'ExportSelection' class:  
#-------------------------------------------------------------------------------

class ExportSelection ( SelectionListenerAction ):
    """ Export the currently selection to a repository.
    """

    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # The tuple of exportable object classes supported:
    classes = Property
    
    # The dictionary mapping classes to resource ids':
    mapping = Property
        
    #---------------------------------------------------------------------------
    # 'Action' interface:
    #---------------------------------------------------------------------------

    #--- Public interface: -----------------------------------------------------

    def perform ( self, event ):
        """ Performs the export action.
        """
        object = self._get_selection()
        resource_id, label = self.mapping[ object.__class__ ]
        self.window.application.get_service( IREPOSITORY ).export_object( 
                            object, title = 'Export ' + label + ' to Repository')

    #---------------------------------------------------------------------------
    # 'SelectionListenerAction' interface:
    #---------------------------------------------------------------------------

    #--- public interface ------------------------------------------------------

    def refresh ( self ):
        """ Refresh the enabled/disabled state of the action.
        """
        window = self.window
        if ((window is not None) and
            (len( window.selection ) == 1) and
             isinstance( self._get_selection(), self.classes )):
            self.enabled = True
        else:
            self.enabled = False

    #---------------------------------------------------------------------------
    #  Property implementations:  
    #---------------------------------------------------------------------------
                
    def _get_classes ( self ):
        if not hasattr( self, '_classes' ):
            self._get_extensions()
        return self._classes
        
    def _get_mapping ( self ):
        if not hasattr( self, '_mapping' ):
            self._get_extensions()
        return self._mapping
        
    #---------------------------------------------------------------------------
    #  Private methods:  
    #---------------------------------------------------------------------------
                
    def _get_extensions ( self ):
        app     = self.window.application
        classes = []
        self._mapping = mapping = {}
        for item in app.get_extensions( ExportableObject ):
            klass = app.import_symbol( item.class_name )
            classes.append( klass )
            mapping[ klass ] = ( item.id, item.label )
        self._classes = tuple( classes )
        
    def _get_selection ( self ):        
        object = self.window.selection[0]
        if isinstance( object, Binding ):
            object = object.obj
        return object

