// Copyright (C) 1994 The New York Group Theory Cooperative
// See magnus/doc/COPYRIGHT for the full notice.

// Contents: Definition of MalcevSet class.
//
// Principal Author: Eugene Paderin
//
// Status: Draft
//
// Description:
//
//   This is a helper structure for SGOfFreeNilpotentGroup class.
//   The subgroup is defined by giving set of words. This set is
//   further extended and brought to special form in order to 
//   find the Mal'cev basis of the subgroup. Class MalcevSet keeps
//   the generating words and maintains all the set transformations.
//
// Special Notes:
//
//
// Revision History:
//

#ifndef _MALCEV_SET_H_
#define _MALCEV_SET_H_

#include "NilpotentCollector.h"
#include "QuickAssociations.h"
#include "AbelianGroup.h"

//================================================================
//------------------- MalcevSet ----------------------------------
//================================================================

// The class is based on QuickAssociations. The key is theWord.leader()

class MalcevSet {

public:
  
  //-----------------------------------------------------
  //   Constructors  
  //-----------------------------------------------------

  MalcevSet();
  // just a placeholder

  MalcevSet(const VectorOf<Word>& v, const NGCollector& nc);
  // put the words to the set
  // ** Time-consuming

  // copy constructor and destructor provided by compiler


  //-----------------------------------------------------
  //   Accessors 
  //-----------------------------------------------------


  int cardinality() const { return theSet.cardinality(); }
  // returns number of elements currently in the basis

  bool isMalcevBasis() const { return isBasis; }
  // returns true if the set is proved to be Malcev basis
  //@ep some semantical imperfectness: here 'false' really means 'dontknow'.

  bool isNormalClosure() const;
  // returns true if the basis is normal closure
  // The set must be Malcev basis 
  // ** Time-consuming


  //-----------------------------------------------------
  //   Methods to form the set
  //-----------------------------------------------------

  // assignment operator provided by compiler

  void makeFull() const; // logical const !
  // makes the set to be the Malcev basis of the subgroup
  // Slow

  MalcevSet normalClosure() const;
  // makes the normal closure of the set
  // Slow


  AbelianGroup mapToQuotient(int weight) const;
  // maps the subgroup generated by the set to the quotient 
  // G_weight / G_{weight+1}
  // Slow, the set must be full
  // The result is initialized (cyclic decomposition found)


  //-----------------------------------------------------
  //   Word containment problem
  //-----------------------------------------------------


  bool contains(const Word& w) const;
  // returns true iff the word is a member of the set
  // ** Time-consuming

  bool decomposeWord(const PolyWord& w, PolyWord& decomp) const;
  // if w is an element of the subgroup, computes its decomposition;
  // if not, decomp is empty word
  // returns true iff the word can be decomposed
  // ** Time-consuming


  //-----------------------------------------------------
  //   Conversions to other presentations
  //-----------------------------------------------------


  VectorOf<Word> getWords() const;
  // returns the basis as a vector of words in terms of
  // the group generators

  VectorOf<Word> getCommutatorWords() const;
  // returns the basis as a vector of words in terms of
  // the basic commutators

  VectorOf<PolyWord> getPolyWords() const;
  // returns the basis as a vector of PolyWords


  //-----------------------------------------------------
  //   I/O
  //-----------------------------------------------------


  void printOn(ostream& s) const;

  friend ostream& operator < (ostream& s, const MalcevSet& b);
  // IPC output

  friend istream& operator > (istream& s, MalcevSet& b);
  // IPC intput


  //-----------------------------------------------------
  //   Helper methods
  //-----------------------------------------------------

private:

  bool reduceWords(PolyWord& mw1, PolyWord& mw2) const;
  // Reduces two words
  // After reduction the leading term of mw1 is in minimal power,
  // the leading term of mw2 is greater.
  // Returns true iff mw1 was changed
  // ** Time-consuming

  static PolyWord makeCommutator( PolyWord& mw1, PolyWord& mw2 );
  // produces commutator of two words

  static int power(const PolyWord& pw) { 
    if( pw.isEmpty() ) return 0;
    return pw.firstLetter().power;
  }
  // power

  static int absPower(const PolyWord& pw) { 
    if( pw.isEmpty() ) return 0;
    return abs( pw.firstLetter().power );
  }
  // absolute power :)

  static Generator leader(const PolyWord& pw) {
    if( pw.isEmpty() ) return 1;
    return pw.firstLetter().gen;
  }
  //the leading letter of the collected form

  static int sign(const PolyWord& pw) {
    if( pw.isEmpty() ) return 0;
    return pw.firstLetter().power > 0 ? 1 : 0;
  }
  // The sign of the leader power

  PolyWord collect(const PolyWord& pw) const {
    return theCollector.collect(pw);
  }

  void checkMembership(PolyWord& w) const;
  // checks whether w can be presented as an admissible set of exponents.
  // returns the "remainder"
  // The argument must be collected
  // ** Time-consuming

  bool addWord(const Word& w);
  // adds w to the set
  // returns true iff the set was changed
  // ** Time-consuming

  bool addWord(const PolyWord& w);
  // returns true iff the set was changed
  // ** Time-consuming

  void makeNormalClosure();
  // makes the set to be normally closed
  // ** Time-consuming

  //-----------------------------------------------------
  //   Data
  //-----------------------------------------------------

private:

  QuickAssociationsOf<Generator, PolyWord> theSet;
  bool isBasis;
  Trichotomy isNormal;
  NGCollector theCollector;

  friend class FPNilpotentGroupRep;
  friend class SGOfFreeNilpotentGroupRep;

};

#endif
