/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLDriver.h
    \brief Definition of OpenGLDriver.
    \author Graphics Section, ECMWF

    Started: May 2004
*/

#ifndef _OpenGLDriver_H
#define _OpenGLDriver_H

#include <BaseDriver.h>
#include <OpenGLDriverAttributes.h>
#include <XmlNode.h>
#include <SelectionMode.h>
#include <Symbol.h>
#include "AnimationRules.h"

#define GL_GLEXT_PROTOTYPES

#include <typeinfo>
#include <GL/gl.h>
#include <GL/glu.h>

#include <OpenGLNode.h>

class MtInputEvent;

namespace magics
{
class DotShadingProperties;
class FillShadingProperties; 
class HatchShadingProperties;

class OpenGLLayerNode;
class OpenGLLayoutNode;
class OpenGLPreviewLayoutNode;
class OpenGLDrawableNode;
class OpenGLPageNode;
class OpenGLFontHandler;
class OpenGLDplItem;
class OpenGLBoxRenderObject;
class OpenGLTextureItem;
class OpenGLAnimationControl;
class OpenGLFrameNode;
class OpenGLPainter;
class OpenGLSymbolItem;
class OpenGLSymbolManager;


/*! \class glHole
    \brief class to hold point data for shading
*/
class glHole {
	GLdouble* 	w;
public:
	glHole(int n) {w = new GLdouble [3*n];}
	glHole() {w = new GLdouble [1]; };
	~glHole() { delete [] w;};
	GLdouble* GetData() { return w; };					
};

typedef	list<glHole*> glHoleList;


class MagicsEvent;


/*! \class OpenGLDriver
    \brief This driver produces output for OpenGL
    \ingroup drivers

    This driver produces as through OpenGL which is intended for desktop visualisation 
    such as in Metview. The OpenGL API used can be hardware accelerated or not by using Mesa.

    The driver requires a GUI toolkit to provide a Widget for the render area. Currently supported 
    are the Motif OpenGL widget (GLw).
*/

class OpenGLDriver: public BaseDriver, public OpenGLDriverAttributes , public OpenGLNodeVisitor
{

friend class OpenGLMagnifier;
friend class OpenGLZoomStackGui;
friend class OpenGLManagerWidget;
friend class OpenGLPickObject;
friend class OpenGLPickObjectCreator;
friend class OpenGLPickObjectManager;
friend class OpenGLBaseWidget;
friend class OpenGLGui;
friend class OpenGLLabelWidget;
friend class OpenGLDriverManager;
friend class OpenGLCellBarWidget;

public:
	OpenGLDriver();
	~OpenGLDriver();
	void open();
	void close();

	/*!
	  \brief sets a new XML node
	*/
	void set(const XmlNode& node)
	{
		if ( magCompare(node.name(), "opengl") )
		{
			XmlNode basic = node;
			basic.name("driver");
			BaseDriver::set(basic);
			basic.name("opengl");
			OpenGLDriverAttributes::set(basic);
		}
        if ( observer_ ) registerObserver(observer_);
	}

	/*!
	  \brief sets a new map
	*/
	void set(const map<string, string>& map)
	{
		BaseDriver::set(map);
		OpenGLDriverAttributes::set(map);
         if ( observer_ ) registerObserver(observer_);
	}
	
	/* OpenGLDriver specific methods */

	void redisplay() const;

	float dimensionX() {return dimensionX_;};
	float dimensionY() {return dimensionY_;};
	
	OpenGLNode* glTree() {return glTree_;};
	

	//! Selection of Interactive mode: tooltip : user callback called when mouse moves. 
	void tooltip()  {}	
	void pickSelection(const SelectionMode&);
	void pointSelection(const SelectionMode&);  	
	void areaSelection(const SelectionMode&);	
	void lineSelection(const SelectionMode&);  	 
	void polylineSelection(const SelectionMode&);
	void polygonSelection(const SelectionMode&); 
	void noMode();
	void setInteractiveMode(const InteractiveMode);	

	//! Restore the framebuffer contents from the background texture 
	void restoreFb();

	void inputEvent(MtInputEvent*);
		
	void notify(MagicsZoomEvent&);
	void notify(MagicsMagnifierEvent&);
	void notify(MagicsAntialiasingEvent&);
	void notify(MagicsRestoreFbEvent&);
	void notify(MagicsAnimationCurrentStepEvent&);
	void notify(MagicsLayerUpdateEvent&);

	//! OpenGL callbacks
	static void TessCombineCB(GLdouble coords[3],GLdouble**,GLfloat*,GLdouble **dataOut);
	static void TessErrorCB(GLenum errorCode);
	
	//! Observers	
	void registerObserver(OpenGLDriverObserver*);
	void notifyObservers(MagicsEvent&) const; 

	void unregisterObserver(OpenGLDriverObserver*);
	void notifyObservers(OpenGLDriverObserver::Function) const;
	void notifyObservers(OpenGLDriverObserver::SelectionFunction, SelectionObject*);
	void notifyObservers(OpenGLDriverObserver::InteractiveModeFunction,InteractiveMode);
	void notifyObservers(OpenGLDriverObserver::BoolFunction,bool);
	void notifyObservers(OpenGLDriverObserver::IntFunction,int);
	void notifyObservers(OpenGLDriverObserver::StringFunction,const string&);
	void notifyObservers(OpenGLDriverObserver::InputEventFunction,MtInputEvent*);

	void notifyResize(float, float);
	
	//! OpenGL control
	void swapFb() const; 
	void swapFbWithCurrentContent() const; 
	void resetFb() const;
	OpenGLAnimationControl *animationControl();
	AnimationStep* stepObjectToRender() {return stepObjectToRender_;}
	
private:
	void startPage() const;
	void endPage() const;

	void newLayer(const Layer&) const;
	void closeLayer(const Layer&) const;

	void project(const Layout&) const;
	void project(const PreviewLayout&) const;
	void project(const MagnifierLayout&) const;

	void unproject() const;
	
	void setNewLineWidth(const float) const;	
	void setNewColour(const Colour &col) const;
	int  setLineParameters(const LineStyle style, const float w) const;

	void renderPolyline(const int, float *, float *) const;
	void renderPolyline2x(const int, int*, const int) const;
	void renderPolyline2(const int n, float *x, float *y) const;
	void renderSimplePolygon(const int, float *, float *) const;
 	void renderText(const Text& text) const;
			
	void circle(const float x, const float y, const float r, const int) const;
	bool renderPixmap(float,float,float,float,int,int,unsigned char*,int,bool hasAlpha=false) const;
	bool renderCellArray(const Image& obj) const;	
	void renderSymbols(const Symbol& symbol) const;

	//! Method to print string about this class on to a stream of type ostream (virtual).
	void print(ostream&) const;
	void debugOutput(const string &s) const;

	/* OpenGLDriver specific methods */     
	
	OpenGLNode *getNewLayoutParentInTree() const;
	void project(OpenGLLayoutNode *, OpenGLNode*,const Layout&) const;

	//Symbols
	void renderTextSymbols(const TextSymbol&) const;
	void generateSymbolDpl(OpenGLSymbolItem*,svgSymbol) const;

	//! Method to redisplay a Layout (virtual).

	MAGICS_NO_EXPORT void redisplay(const PreviewLayout&) const;
	MAGICS_NO_EXPORT void redisplay(const MagnifierLayout&) const;

	MAGICS_NO_EXPORT void redisplay(const StepRenderer&) const;

	void renderFilledPolygon(int,float*,float*) const;
	void Primitive_simple(int n, float* cx, float* cy, GLenum style) const;
	void Primitive_cluster(int i, int n, float* cx, float* cy, GLenum style) const;
	void checkError(string s) const;
	
	void renderTextForMagnifier(OpenGLLayoutNode *,
			            const Text*,float,float);	
	void renderTextItem(const Text* text,float mfactor=1.,float tfactor=1.,
		            float xoff=0.,float yoff=0.,float xmin=0.,float ymin=0.) const;
	
	void endPolygon() const {};	

	void addGlNode(OpenGLNode*) const;

	void setDimension(OpenGLLayoutNode*) const;
	void setDimension(float,float) const;
	void resetDimension() const;

	void setCoordRatio(OpenGLLayoutNode*) const;
	void setCoordRatio(float,float) const;
	void resetCoordRatio() const;

	float calcMagnifierMaxPower();
	void clearMagnifierLayout();
	void magnifierLayoutUpdate(float,float,float,float,float,float,float);

	//Interactive mode	
		
	
	
	void setNewPolygonMode(const GLint) const;
	void mapBgImageToFb(const int) const;	
	void mapBgImageToFb(const int,const int, const int, const int, const int) const;
	void mapBgImageToFb(const int,const int, const int, const int, const int,
			              const int, const int, const int, const int) const;
	void saveFbToBgImage(const int) const;
	
	bool visitEnter(OpenGLNode&);
	bool visitLeave(OpenGLNode&);
	void visit(OpenGLNode&);


	bool visitEnter_renderTree(OpenGLNode&);
	bool visitLeave_renderTree(OpenGLNode&);
	void visit_renderTree(OpenGLNode&);
	void renderTree(int=0);

	bool visitEnter_renderPreview(OpenGLNode&);
	bool visitLeave_renderPreview(OpenGLNode&);
	void visit_renderPreview(OpenGLNode&);
	void renderPreview(OpenGLPreviewLayoutNode*,int,int,int,int);

	bool visitEnter_getLayerTree(OpenGLNode&);
	bool visitLeave_getLayerTree(OpenGLNode&);
	void visit_getLayerTree(OpenGLNode&);
	void getLayerTree();
	void layerUpdate();

	// MOTIF related
	GLushort		linePatterns_[5];
	
	mutable GLUtesselator	*tobj_;
	mutable string	currentFont_;
	mutable int		currentFontSize_;
	mutable bool		ready_;
	mutable bool		clipping_;
	mutable GLint		currentPolygonMode_;
	float			lineWidthFactor_;
        
	mutable bool		antialiasing_;
	mutable bool initialized_;
	mutable GLuint tex_bg_id_[2];	
	
	mutable SelectionMode pickObjectAttr_;
	
	// Display lists
	mutable map<OpenGLLayoutNode*,OpenGLAnimationControl*> animationStepMap_;
	//mutable list<OpenGLFrameNode*> framesToRender_;

	mutable OpenGLNode*			layerTree_;
	mutable OpenGLNode*       		glTree_;
	//mutable list<OpenGLDrawableNode*>     	drawableList_;
	//mutable list<OpenGLPageNode*>         	pageList_;
	mutable OpenGLDplItem*		     	actDpl_;
	mutable int				stepToRender_;
	mutable AnimationStep*                  stepObjectToRender_;

	mutable list<OpenGLTextureItem*>	texList_;

	//mutable list<OpenGLLayoutNode*>      	glAnimationLayoutList_;
	mutable stack<OpenGLLayoutNode*>      	glLayoutStack_;
	mutable stack<OpenGLLayerNode*>      	glLayerStack_;
	mutable stack<OpenGLNode*>      	glNodeStack_;
	mutable OpenGLNode*			currentGlNode_;

	mutable OpenGLSymbolManager*		symbols_;


	mutable float 				magnifierZoom_;


	mutable bool				emptyNode_;

	static 	bool 	fboUsed_;	

	
	OpenGLFontHandler* fontHandler_;	

	OpenGLPainter*		painter_;

	typedef void (OpenGLDriver::*VisitFunctionVoidPtr)(OpenGLNode&);
	typedef bool (OpenGLDriver::*VisitFunctionBoolPtr)(OpenGLNode&);
	
	VisitFunctionVoidPtr visitFunc_;
	VisitFunctionBoolPtr visitEnterFunc_;
	VisitFunctionBoolPtr visitLeaveFunc_;

	vector<OpenGLDriverObserver*> observers_;	
	
	//! Copy constructor - No copy allowed
	OpenGLDriver(const OpenGLDriver&);
	//! Overloaded << operator to copy - No copy allowed
	OpenGLDriver& operator=(const OpenGLDriver&);

	// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const OpenGLDriver& p)
		{ p.print(s); return s; }
};

} // namespace magics
#endif
