/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLTextureItem.cc
    \brief Implementation of the OpenGLTextureItem class.
    \author Graphics Section, ECMWF

    Started: August 2008
*/
#include <OpenGLTextureItem.h>
#include <Log.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#define PNG_DEBUG 3
#include <png.h>

using namespace magics;

OpenGLTextureItem::OpenGLTextureItem()
{
 	transparent_=true; 
	blendingSrcFactor_=GL_SRC_ALPHA;
	blendingDestFactor_=GL_ONE_MINUS_SRC_ALPHA;
	blendingAlpha_=0.;
	texId_=0;
	
	//We have to store the bound status in this variable becuse 
	//glIsTexture cannot be used in display lists
	filled_=false;

	fading_=false;
	fadingFactor_=0.5;

	dplMode_=false;
}

OpenGLTextureItem::~OpenGLTextureItem()
{
	clear();
}	

void OpenGLTextureItem::generate()
{	
	if(isEmpty())
	{
		glGenTextures(1,&texId_);
		filled_=false;
	}
}

void  OpenGLTextureItem::clear()
{	
 	if(!isEmpty()) 
      	{	 
		glDeleteTextures(1,&texId_);
	  	texId_=0;
		filled_=false;
	}
}

bool OpenGLTextureItem::isEmpty()
{
	return (texId_==0)?true:false;
}

bool OpenGLTextureItem::isFilled()
{
	//We have to store the bound status in this variable because 
	//glIsTexture cannot be used in display lists
	return filled_;

	/*if(glIsTexture(texId_) == GL_TRUE) 
        {	 
		return true;
	}
	return false;*/
}


void OpenGLTextureItem::setBlending(GLenum s,GLenum d,float a) 
{
	blendingSrcFactor_=s;
	blendingDestFactor_=d;
	setBlendingAlpha(a);
}

void OpenGLTextureItem::setBlendingAlpha(float a)
{
	blendingAlpha_=a;
	glBlendColor(a,a,a,a);
}

void OpenGLTextureItem::setDplMode(bool m)
{
	dplMode_=m;
}

void OpenGLTextureItem::bind()
{
	glBindTexture(GL_TEXTURE_2D,texId_);
}

void OpenGLTextureItem::unBind()
{
	glBindTexture(GL_TEXTURE_2D,0);
}

void OpenGLTextureItem::mapTexture(const float spx1, const float spy1, const float spx2, const float spy2,
		                   const int tpx1, const int tpy1, const int tpx2, const int tpy2)
{
	//Texture must be filled
	assert(isFilled() == true);

	vector<PaperPoint> sp(4), tp(4);
	
	//Source
	sp[0].x(spx1); sp[0].y(spy1);
	sp[1].x(spx2); sp[1].y(spy1);
	sp[2].x(spx2); sp[2].y(spy2);
 	sp[3].x(spx1); sp[3].y(spy2);

	//Target
	tp[0].x(tpx1); tp[0].y(tpy1);
	tp[1].x(tpx2); tp[1].y(tpy1);
	tp[2].x(tpx2); tp[2].y(tpy2);
 	tp[3].x(tpx1); tp[3].y(tpy2);

	mapTexture(sp,tp);
}

void OpenGLTextureItem::mapTexture(const float tpx1, const float tpy1, const float tpx2, const float tpy2)
{
	//Texture must be filled
	assert(isFilled() == true);

	vector<PaperPoint> sp(4), tp(4);
	
	//Source
	sp[0].x(0.); sp[0].y(0.);
	sp[1].x(1.); sp[1].y(0.);
	sp[2].x(1.); sp[2].y(1.);
 	sp[3].x(0.); sp[3].y(1.);

	//Target
	tp[0].x(tpx1); tp[0].y(tpy1);
	tp[1].x(tpx2); tp[1].y(tpy1);
	tp[2].x(tpx2); tp[2].y(tpy2);
 	tp[3].x(tpx1); tp[3].y(tpy2);

	mapTexture(sp,tp);
}

void OpenGLTextureItem::mapTexture(const vector<PaperPoint> &sp,const vector<PaperPoint> &tp)
{	
	//Texture must be filled
	assert(isFilled() == true); 
           
	if(sp.size() != tp.size()) return;
	if(sp.size() < 4) return;
		
	//Polygons front-face is CCW
	glPushAttrib(GL_POLYGON_BIT | GL_CURRENT_BIT);
	//glFrontFace(GL_CCW);
	//glEnable(GL_CULL_FACE);	
	glDisable(GL_CULL_FACE);
	//glCullFace(GL_BACK);		
	
	if(dplMode_ == false)
	{
		//We allow transparency
		if(transparent_)
		{
			glEnable(GL_BLEND); 		
			glBlendFunc(blendingSrcFactor_,blendingDestFactor_);
		}

		bind();
	}

	if(fading_)
	{
		glColor4f(0.3,0.3,0.3,0.3);
		glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
	}	
	else
	{
		glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);	
	}

	glEnable(GL_TEXTURE_2D);
			
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL); 
    	//glPolygonMode(GL_FRONT,GL_FILL); 

	if(sp.size() == 4)
	{		
		glBegin( GL_QUADS );
	}
	else
	{
		glBegin( GL_POLYGON );
	}

	for(int i=0; i < sp.size(); i++)
	{
		glTexCoord2f(sp[i].x(),sp[i].y());
		glVertex2f(tp[i].x(),tp[i].y());
	}
	
    	glEnd();	   	

	if(dplMode_ == false)
	{
		if(transparent_)
		{
			glDisable(GL_BLEND);
		}
		unBind();
	}

	glDisable(GL_TEXTURE_2D);  	
	
	glPopAttrib();   
}

void OpenGLTextureItem::generateFromFb(const int fx,const int fy, const int fw, const int fh,GLenum img_type)
{     
	//Generate the texture!
	generate();

	if(img_type != GL_RGBA && img_type != GL_RGB)
	{
		return;
	}

     	glBindTexture(GL_TEXTURE_2D,texId_);
    	
	/*glPixelStorei(GL_UNPACK_ALIGNMENT,1);

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
   	//glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    	//glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );	
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );*/	

	glCopyTexImage2D(GL_TEXTURE_2D,0, img_type, 
      	      		 fx,fy,fw,fh,0);

	x_=fx;
	y_=fy;
	width_=fw,
	height_=fh;

	filled_=true;
 
}


void OpenGLTextureItem::modifyFromFb(const int fx,const int fy, const int fw, const int fh, const float tx, const float ty)
{     
	//Texture must be filled
	assert(isFilled() == true );

	glBindTexture(GL_TEXTURE_2D,texId_);
      	           
	glCopyTexSubImage2D(GL_TEXTURE_2D,0, 
      	      tx,ty,fx,fy,fw,fh);
                
}

void OpenGLTextureItem::generateFromImage(GLubyte *img_data, int width, int height,GLenum img_type)
{ 
	//Genarate the texture!
	generate();

	if(img_type != GL_RGBA && img_type != GL_RGB)
	{
		return;
	}

	//glGenTextures(1,&texId_);
	glBindTexture(GL_TEXTURE_2D,texId_);
		
	/*glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );		
	glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);*/

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );  	
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );	

	glTexImage2D(GL_TEXTURE_2D,0,img_type,width,height,0,img_type,GL_UNSIGNED_BYTE,img_data);

	width_=width;
	height_=height;

	filled_=true;

	glBindTexture(GL_TEXTURE_2D,0);
}



int OpenGLTextureItem::generateFromPng(const char* png_fname)
{
	//Genarate the texture!
	generate();

	png_byte header[8];	// 8 is the maximum size that can be checked
	int i, j, k, m;
	int width, height;
	png_byte color_type;
	png_byte bit_depth;
	png_structp png_ptr;
	png_infop info_ptr;
	int number_of_passes;
	png_bytep * row_ptr; 

	//Open file and test it
	FILE *fp = fopen(png_fname, "rb");
	if (!fp)
	{
		Log::debug() << "File " <<  png_fname << " is not readable!" << endl;
		return 1;	
	}
		
	fread(header, 1, 8, fp);
	if(png_sig_cmp(header, 0, 8))
	{
		Log::debug() << "File " <<  png_fname << " is not recognized as a PNG file!" << endl;
		return 1;
	}

	//Initialize 
	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	
	if (!png_ptr) return 1;
		
	info_ptr = png_create_info_struct(png_ptr);
	
	if (!info_ptr) return 1;
		
	if (setjmp(png_jmpbuf(png_ptr))) return 1;
		
	png_init_io(png_ptr,fp);
	png_set_sig_bytes(png_ptr, 8);

	png_read_info(png_ptr, info_ptr);

	width = info_ptr->width;
	height = info_ptr->height;
	color_type = info_ptr->color_type;
	bit_depth = info_ptr->bit_depth;

	number_of_passes = png_set_interlace_handling(png_ptr);
	png_read_update_info(png_ptr, info_ptr);

	//Read file
	if (setjmp(png_jmpbuf(png_ptr))) return 1;	

	row_ptr = (png_bytep*) malloc(sizeof(png_bytep) * height);
	for (j=0; j<height; j++)
		row_ptr[j] = (png_byte*) malloc(info_ptr->rowbytes);

	png_read_image(png_ptr,row_ptr);

        fclose(fp);

	if (info_ptr->color_type != PNG_COLOR_TYPE_RGBA) return 1;
		//abort_("[process_file] color_type of input file must be PNG_COLOR_TYPE_RGBA (is %d)", info_ptr->color_type);
	
	
	GLubyte *img_data = new GLubyte[width*height*4];	
	
	/*m=0;
	for (j=0; j<height; j++) 
	{
		png_byte* row = row_ptr[j];
		for (i=0; i<width; i++) 		
		{				
			png_byte* ptr = &(row[i*4]);
			for(k=0;k< 4;k++)
			{
				img_data[m]=ptr[k];
				m++;
			}
		}
	}*/				
	
	m=0;
	for (j=height-1; j>=0; j--) 
	{
		png_byte* row = row_ptr[j];
		for (i=0; i<width; i++) 		
		{				
			png_byte* ptr = &(row[i*4]);
			for(k=0;k< 4;k++)
			{
				img_data[m]=ptr[k];
				m++;
			}
		}
	}		


	//glGenTextures(1,&texId_);
	glBindTexture(GL_TEXTURE_2D,texId_);
		
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );		
	glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);
	
	glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA,width,height,0,GL_RGBA,GL_UNSIGNED_BYTE,img_data);

	delete [] img_data;

	width_=width;
	height_=height;
	
	filled_=true;

	return 0;
}


/*
void OpenGLDriver::mapBgImageToFb( const int id, const int xs0, const int ys0, const int xs1, const int ys1,
				   const int xt0, const int yt0, const int xt1, const int yt1) const
{
	xs0=static_cast<float>(xs0)/static_cast<float>(dimensionX_);
	ys0=static_cast<float>(ys0)/static_cast<float>(dimensionY_);
	xs1=static_cast<float>(xs1)/static_cast<float>(dimensionX_);
	ys1=static_cast<float>(ys1)/static_cast<float>(dimensionY_);

	vector<PaperPoint> sp(4), tp(4);

	sp[0].x(tx0); sp[0].y(ty0);
	sp[1].x(tx1); sp[1].y(ty0);
	sp[2].x(tx1); sp[2].y(ty1);
	sp[3].x(tx0); sp[3].y(ty1);
	
	tp[0].x(xt0); tp[0].y(yt0);
	tp[1].x(xt1); tp[1].y(yt0);
	tp[2].x(xt1); tp[2].y(yt1);
	tp[3].x(xt0); tp[3].y(yt1);

	bgTex_[id].mapTextureRect(sp,tp);
}

*/	