
/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLMagnifierScale.cc
    \brief Implementation of OpenGLMagnifierScale.
    \author Graphics Section, ECMWF

    Started: August 2008

*/
#include <Colour.h>
#include <OpenGLMagnifierScale.h>
#include <OpenGLMagnifier.h>
#include <OpenGLTextureItem.h>
#include <OpenGLPainter.h>

#include <MtInputEvent.h>

#include <math.h>

using namespace magics;

void OpenGLMagnifierScaleItem::set(float san, float ean,float r1,float r2)
{	
	startAngle_=san;
	endAngle_=ean;

	radIn_=r1;
	radOut_=r2;
	
}

OpenGLMagnifierScaleSlider::OpenGLMagnifierScaleSlider()
{
	tex_= new OpenGLTextureItem;
	//tex_->generateFromPng(getSharedPngFileName("round_button","16x16"));

	string p=getEnvVariable("MAGPLUS_HOME") + MAGPLUS_PATH_TO_SHARE_ + "ui/16x16/round_button.png";
	tex_->generateFromPng(p.c_str());
}

OpenGLMagnifierScaleSlider::~OpenGLMagnifierScaleSlider()
{
	delete tex_;
}

bool OpenGLMagnifierScaleSlider::checkPointInSlider(int x, int y)
{
	int xc,yc;
        getCentre(xc,yc);

	float d=(x_[0]-x_[2])*(x_[0]-x_[2])+(y_[0]-y_[2])*(y_[0]-y_[2]);
	
	if((x-xc)*(x-xc) + (y-yc)*(y-yc) < d/4.)
	{
		return true;
	}

	return false;		
}


void OpenGLMagnifierScaleSlider::getCentre(int &xc,int &yc)
{
        xc=(x_[0]+x_[2])/2.;
	yc=(y_[0]+y_[2])/2.;
}        
  
void OpenGLMagnifierScaleSlider::setCentre(int xc,int yc)
{
        int xc_act,yc_act;       
        getCentre(xc_act,yc_act);
        
        for(int i=0; i< 4; i++)
        {
                x_[i]+=xc-xc_act;
                y_[i]+=yc-yc_act;
        }        
}      
      
bool OpenGLMagnifierScaleSlider::shiftSlider(int dx, int dy,float maxAngle,float minAngle)
{
        bool ret=false;
	int xc, yc;      
        getCentre(xc,yc);
        
        float a=xc*xc+yc*yc;
        float b=(xc+dx)*(xc+dx)+(yc+dy)*(yc+dy);
        float c=dx*dx+dy*dy;
        
        float dfi=-(a+b-c)/(2.*sqrt(a*b));
        dfi=acos(dfi);

        float fi=atan2f(xc,yc);                
        if(fi< 0.) fi=2*PI+fi;

	if(fi+dfi > maxAngle)
        {
                dfi=maxAngle-fi;
		ret=true;
        }
        else if(fi+dfi < minAngle)
        {
                dfi=minAngle+fi;
		ret=true;
        }
        
        int xcn=cos(dfi)*xc-sin(dfi)*yc;
        int ycn=sin(dfi)*xc+cos(dfi)*yc;
 
        setCentre(xcn,ycn);

	return ret;       
}
       
void OpenGLMagnifierScaleSlider::render()
{
	vector<PaperPoint> sp, tp;

	sp.push_back(PaperPoint(0.,0.));
	tp.push_back(PaperPoint(x_[0],y_[0]));

	sp.push_back(PaperPoint(1.,0.));
	tp.push_back(PaperPoint(x_[1],y_[1]));

	sp.push_back(PaperPoint(1.,1.));
	tp.push_back(PaperPoint(x_[2],y_[2]));

	sp.push_back(PaperPoint(0.,1.));
	tp.push_back(PaperPoint(x_[3],y_[3]));

	tex_->mapTexture(sp,tp);
 
}
	
//=======================================================
//
// OpenGLMagnifierScale
//
//=======================================================

OpenGLMagnifierScale::OpenGLMagnifierScale(vector<float> &vals, int actIndex,OpenGLMagnifier *m,Colour c1, Colour c2)
{
	magnifier_=m;	
	minColour_=c1;
	maxColour_=c2;	

	setFactors(vals,actIndex);
		
	//Set factor selection cell-bar

	cellHeight_=13.;
	cellWidth_=6.;
	cellOutlineColour_=Colour(0.2,0.2,0.2);	
	cellSelectionLineWidth_=1.5;	
	areaOutlineColour_=Colour(0.,0.,0.);
	areaTopFillColour_=Colour(0.95,0.95,0.95);
	areaBottomFillColour_=Colour(0.7,0.7,0.7);
	areaBorderSize_=10.; 	
	areaWidth_=cellWidth_+10;
	areaResolution_=20;
	textColour_=Colour(0.4,0.4,0.4);
	textSelectionColour_=Colour(0.,0.,0.);

	drag_=false;

}

OpenGLMagnifierScale::~OpenGLMagnifierScale()
{
}

void OpenGLMagnifierScale::setFactors(vector<float>& vals,int actIndex)
{	
	//clear the current factors!
	clear();

	actIndex_=actIndex;

	Colour c1=minColour_;
	Colour c2=maxColour_;

	for(int i=0; i<vals.size();i++)
	{						
		float weight=static_cast<float>(i)/vals.size();
			
		if(weight < 0 || weight > 1) weight=0;

		float r=weight*c2.red()+(1-weight)*c1.red();
		float g=weight*c2.green()+(1-weight)*c1.green();
		float b=weight*c2.blue()+(1-weight)*c1.blue();
	
		Colour col(r,g,b);
			
		Hsl hsl=col.hsl();			
		hsl.light_*=0.4;
		Colour col_d(hsl);		
		
		hsl=col.hsl();
		hsl.light_*=1.25;
		if(hsl.light_> 1) hsl.light_=1.;	
		Colour col_b(1.,1.,1.);
		
		Colour col_s(0.25,0.25,0.25);

		push_back(OpenGLMagnifierScaleItem(vals[i],col,col_d,col_b,col_s)); 		
	}	
}


void OpenGLMagnifierScale::set(float r)
{	
	areaRadIn_=r;

	cellAngleSize_=cellHeight_/r;
	
	cellStartAngle_=PI+size()*cellAngleSize_/2.;
	cellEndAngle_=cellStartAngle_-size()*cellAngleSize_;

	float r1=r+(areaWidth_-cellWidth_)/2.;
	float r2=r+areaWidth_-(areaWidth_-cellWidth_)/2.;

	for(int i=0; i < size(); i++)
	{
		float an1=cellStartAngle_-i*cellAngleSize_;
		float an2=an1-cellAngleSize_;
		
		at(i).set(an1,an2,r1,r2);
	}	

        setSlider();
}

int OpenGLMagnifierScale::identifyPointInScale(float dx, float dy)
{
        float dist=sqrt(dx*dx+dy*dy);       
        return identifyPointInScale(dist,dx,dy);
}

int OpenGLMagnifierScale::identifyPointInScale(float dist, float dx, float dy)
{
	float rad1=areaRadIn_+1;
	float rad2=areaRadIn_+areaWidth_;
		
	if(dist >=  rad1 && dist <= rad2 )
	{
		float angle=atan2f(dy,dx);
		if(angle < 0.) angle=2*PI+angle;
		if(angle <= cellStartAngle_  && angle >= cellEndAngle_)
		{
			for(int i=0; i< size(); i++)
			{
				if(angle <= at(i).startAngle_ && angle >= at(i).endAngle_)
				{
					return i;
				}
			}
		}
	}	

	return -1;		
}


void OpenGLMagnifierScale::setSlider(int index)
{
        if(index >= 0 && index < size())
	{
		//float w=at(index).radOut_-at(index).radIn_-2;
                //float rlen=cellAngleSize_*(at(index).radIn_+w/2.);
                //float dfi=(rlen-w/2.)/(at(index).radIn_+w/2.);
 
		float fi=(at(index).startAngle_+at(index).endAngle_)/2.;
		float r=(at(index).radIn_+at(index).radOut_)/2.;
		float xc=cos(fi)*r;
		float yc=sin(fi)*r;
		float w=cellWidth_+6.;

		slider_.setX(0,xc-w/2.);
		slider_.setY(0,yc-w/2.);
		
		slider_.setX(1,xc+w/2.);
		slider_.setY(1,yc-w/2.);

		slider_.setX(2,xc+w/2.);
		slider_.setY(2,yc+w/2.);

                slider_.setX(3,xc-w/2.);
		slider_.setY(3,yc+w/2.);
		

		/*slider_.setX(0,cos(at(index).startAngle_-dfi)*at(index).radIn_+1);
		slider_.setY(0,sin(at(index).startAngle_-dfi)*at(index).radIn_+1);

		slider_.setX(1,cos(at(index).endAngle_+dfi)*at(index).radIn_+1);
		slider_.setY(1,sin(at(index).endAngle_+dfi)*at(index).radIn_+1);

		slider_.setX(2,cos(at(index).endAngle_+dfi)*at(index).radOut_-1);
		slider_.setY(2,sin(at(index).endAngle_+dfi)*at(index).radOut_-1);

		slider_.setX(3,cos(at(index).startAngle_-dfi)*at(index).radOut_-1);
		slider_.setY(3,sin(at(index).startAngle_-dfi)*at(index).radOut_-1);*/		
	}		
}


void OpenGLMagnifierScale::setSlider()
{
        setSlider(actIndex_);
}

bool OpenGLMagnifierScale::setActIndex(int index)
{
        if(actIndex_ !=index && index >= 0 && index < size())
        {
 		actIndex_=index;
               	magnifier_->remagnify();   
                return true;
        } 
        return false;
}      


void OpenGLMagnifierScale::render()
{
	glPushAttrib(GL_ENABLE_BIT | GL_LINE_BIT);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);	

	renderPanel();
	renderScale();

	glPopAttrib();
}


void OpenGLMagnifierScale::renderPanel()
{
	float rad1, rad2;
	float ap, ap_next, dal;
	float xx, yy;

	int i;

	//-------------------------------------
	// Render (fill) the factor cell area
	//-------------------------------------			
		
	rad1=areaRadIn_;
	rad2=rad1+areaWidth_;
									
	ap=cellStartAngle_;
	//ap_next=ap-cellAngleSize_-cellAngleGap_;
				
	glPolygonMode(GL_FRONT,GL_FILL); 
		
	glShadeModel(GL_SMOOTH);
		
	dal=cellAngleSize_*size()/areaResolution_;

	float dcol=(areaTopFillColour_.red()-areaBottomFillColour_.red())/areaResolution_;
	float acol=areaBottomFillColour_.red();

	for(i=0; i< areaResolution_; i++)
	{
		glBegin(GL_QUADS);
		
		glColor3f(acol+dcol,acol+dcol,acol+dcol);		
		xx=cos(ap-dal)*rad2;
		yy=sin(ap-dal)*rad2;
		glVertex2f(xx,yy);
		
		glColor3f(acol,acol,acol);
		xx=cos(ap)*rad2;
		yy=sin(ap)*rad2;
		glVertex2f(xx,yy);
			
		glColor3f(acol,acol,acol);
		xx=cos(ap)*rad1;
		yy=sin(ap)*rad1;
		glVertex2f(xx,yy);
			
		glColor3f(acol+dcol,acol+dcol,acol+dcol);
		xx=cos(ap-dal)*rad1;
		yy=sin(ap-dal)*rad1;
		glVertex2f(xx,yy);
			
		glEnd();
			
		ap-=dal;

		acol+=dcol;			
	}

	glShadeModel(GL_FLAT);		

	//Triangle at the bottom of the area

	ap=cellStartAngle_;
	Colour *col=&areaBottomFillColour_;
	glColor3f(col->red(),col->green(),col->blue());	
	
	glBegin(GL_TRIANGLES);
			
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
		
	xx=cos(ap+areaBorderSize_/rad1)*rad1;
	yy=sin(ap+areaBorderSize_/rad1)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	glEnd();

	//Triangle at the top of the area	

	ap=cellStartAngle_-cellAngleSize_*size();
	col=&areaTopFillColour_;
	glColor3f(col->red(),col->green(),col->blue());	

	glBegin(GL_TRIANGLES);
				
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap-areaBorderSize_/rad1)*rad1;
	yy=sin(ap-areaBorderSize_/rad1)*rad1;
	glVertex2f(xx,yy);
		
	glEnd();
		
	//---------------------------------------		
	// Render the factor cell area outline
	//---------------------------------------
		
	ap=cellStartAngle_;
		
	glPolygonMode(GL_FRONT,GL_LINE); 
	col=&areaOutlineColour_;
	glColor3f(col->red(),col->green(),col->blue());
						
	glBegin(GL_LINE_STRIP);				
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad2;
		yy=sin(ap)*rad2;
		glVertex2f(xx,yy);			
		ap-=dal;
	}		
	glEnd();
		
	glBegin(GL_LINE_STRIP);	
		
	ap=cellStartAngle_+areaBorderSize_/rad1;
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	ap=cellStartAngle_;
							
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad1;
		yy=sin(ap)*rad1;
		glVertex2f(xx,yy);			
		ap-=dal;
	}				
		
		
	ap+=dal-areaBorderSize_/rad1;		
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	glEnd();		
		
	glBegin(GL_LINES);		
				
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap+areaBorderSize_/rad1)*rad2;
	yy=sin(ap+areaBorderSize_/rad1)*rad2;
	glVertex2f(xx,yy);
					
	ap=cellStartAngle_;	
			
	xx=cos(ap+areaBorderSize_/rad1)*rad1;
	yy=sin(ap+areaBorderSize_/rad1)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);		
		
	glEnd();	

	glShadeModel(GL_FLAT);	

	renderTextPanel();	

}


void OpenGLMagnifierScale::renderTextPanel()
{
	float rad1, rad2;
	float ap, ap_next, dal;
	float xx, yy;

	int i;

	//-------------------------------------
	// Render (fill) the factor cell area
	//-------------------------------------			
		
	//rad1=areaRadIn_;
	//rad2=rad1+areaWidth_;

	rad1=areaRadIn_-2-18;
	rad2=areaRadIn_-2;
								
	ap=cellStartAngle_;
	//ap_next=ap-cellAngleSize_-cellAngleGap_;
				
	glPolygonMode(GL_FRONT,GL_FILL); 
		
	glShadeModel(GL_SMOOTH);
		
	dal=cellAngleSize_*size()/areaResolution_;

	float dcol=(areaTopFillColour_.red()-areaBottomFillColour_.red())/areaResolution_;
	float acol=areaBottomFillColour_.red();

	glShadeModel(GL_FLAT);		

	//Triangle at the bottom of the area

	ap=cellStartAngle_;
	Colour *col=&areaBottomFillColour_;
	glColor3f(col->red(),col->green(),col->blue());	
	
	glBegin(GL_TRIANGLES);

	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);		
		
	xx=cos(ap+areaBorderSize_/rad1)*rad2;
	yy=sin(ap+areaBorderSize_/rad1)*rad2;
	glVertex2f(xx,yy);
			
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);	

	glEnd();

	//Triangle at the top of the area	

	ap=cellStartAngle_-cellAngleSize_*size();
	col=&areaTopFillColour_;
	glColor3f(col->red(),col->green(),col->blue());	

	glBegin(GL_TRIANGLES);

	xx=cos(ap-areaBorderSize_/rad2)*rad2;
	yy=sin(ap-areaBorderSize_/rad2)*rad2;
	glVertex2f(xx,yy);	
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
			
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	glEnd();
		
	//---------------------------------------		
	// Render the factor cell area outline
	//---------------------------------------
		
	/*ap=cellStartAngle_;
		
	glPolygonMode(GL_FRONT,GL_LINE); 
	col=&areaOutlineColour_;
	glColor3f(col->red(),col->green(),col->blue());
						
	glBegin(GL_LINE_STRIP);				
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad2;
		yy=sin(ap)*rad2;
		glVertex2f(xx,yy);			
		ap-=dal;
	}		
	glEnd();
		
	glBegin(GL_LINE_STRIP);	
		
	ap=cellStartAngle_+areaBorderSize_/rad1;
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	ap=cellStartAngle_;
							
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad1;
		yy=sin(ap)*rad1;
		glVertex2f(xx,yy);			
		ap-=dal;
	}				
		
		
	ap+=dal-areaBorderSize_/rad1;		
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	glEnd();		
		
	glBegin(GL_LINES);		
				
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap+areaBorderSize_/rad1)*rad2;
	yy=sin(ap+areaBorderSize_/rad1)*rad2;
	glVertex2f(xx,yy);
					
	ap=cellStartAngle_;	
			
	xx=cos(ap+areaBorderSize_/rad1)*rad1;
	yy=sin(ap+areaBorderSize_/rad1)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);		
		
	glEnd();	
*/

	//-------------------------------------
	// Render (fill) the text area
	//-------------------------------------			
		
	//rad1=areaRadIn_-25;
	//rad2=areaRadIn_;
									
	ap=cellStartAngle_;
	//ap_next=ap-cellAngleSize_-cellAngleGap_;
				
	glPolygonMode(GL_FRONT,GL_FILL); 
		
	glShadeModel(GL_SMOOTH);
		
	dal=cellAngleSize_*size()/areaResolution_;

	dcol=(areaTopFillColour_.red()-areaBottomFillColour_.red())/areaResolution_;
	acol=areaBottomFillColour_.red();

	for(i=0; i< areaResolution_; i++)
	{
		glBegin(GL_QUADS);
		
		glColor3f(acol+dcol,acol+dcol,acol+dcol);		
		xx=cos(ap-dal)*rad2;
		yy=sin(ap-dal)*rad2;
		glVertex2f(xx,yy);
		
		glColor3f(acol,acol,acol);
		xx=cos(ap)*rad2;
		yy=sin(ap)*rad2;
		glVertex2f(xx,yy);
			
		glColor3f(acol,acol,acol);
		xx=cos(ap)*rad1;
		yy=sin(ap)*rad1;
		glVertex2f(xx,yy);
			
		glColor3f(acol+dcol,acol+dcol,acol+dcol);
		xx=cos(ap-dal)*rad1;
		yy=sin(ap-dal)*rad1;
		glVertex2f(xx,yy);
			
		glEnd();
			
		ap-=dal;

		acol+=dcol;			
	}

	glShadeModel(GL_FLAT);		

	//---------------------------------------		
	// Render the factor cell area outline
	//---------------------------------------
		
	ap=cellStartAngle_;
		
	glPolygonMode(GL_FRONT,GL_LINE); 
	col=&areaOutlineColour_;
	glColor3f(col->red(),col->green(),col->blue());
						
	glBegin(GL_LINE_STRIP);				
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad2;
		yy=sin(ap)*rad2;
		glVertex2f(xx,yy);			
		ap-=dal;
	}		
	glEnd();
		
	glBegin(GL_LINE_STRIP);	
		
	ap=cellStartAngle_+areaBorderSize_/rad2;
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
		
	ap=cellStartAngle_;
							
	for(i=0; i<= areaResolution_; i++)
	{							
		xx=cos(ap)*rad1;
		yy=sin(ap)*rad1;
		glVertex2f(xx,yy);			
		ap-=dal;
	}				
		
		
	ap+=dal-areaBorderSize_/rad2;		
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
		
	glEnd();		
		
	/*glBegin(GL_LINES);		
				
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap+areaBorderSize_/rad1)*rad2;
	yy=sin(ap+areaBorderSize_/rad1)*rad2;
	glVertex2f(xx,yy);
					
	ap=cellStartAngle_;	
			
	xx=cos(ap+areaBorderSize_/rad1)*rad1;
	yy=sin(ap+areaBorderSize_/rad1)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);		
		
	glEnd();*/	

	
}


void OpenGLMagnifierScale::renderScale()
{
	float rad1, rad2;
	float ap, ap_next, dal;
	float xx,yy;
	int i;

	//----------------------------------
	// Render the factor cells 
	//----------------------------------				
		
	for(i=0; i< size(); i++)
	{						
		float arad1, arad2;
		
		glPolygonMode(GL_FRONT,GL_FILL); 				
			
		Colour *col;
		Colour *col1;
		
		if(i<=actIndex_)
		{
			col=&at(i).colour_;
			col1=&at(i).colour_;
			arad1=at(i).radIn_; 
			arad2=at(i).radOut_; 
		}		
		else
		{
			col=&at(i).colour_;
			col1=&at(i).colour_;
			arad1=at(i).radIn_; 
			arad2=at(i).radOut_; 		
		}
				
		ap=at(i).startAngle_;
		ap_next=at(i).endAngle_;	

		glShadeModel(GL_SMOOTH);
		
		glBegin(GL_QUADS);
			
		glColor3f(col1->red(),col1->green(),col1->blue());						
		xx=cos(ap)*arad2;
		yy=sin(ap)*arad2;		
		glVertex2f(xx,yy);
		
		glColor3f(col->red(),col->green(),col->blue());										
		xx=cos(ap)*arad1;
		yy=sin(ap)*arad1;					
		glVertex2f(xx,yy);
						
		xx=cos(ap_next)*arad1;
		yy=sin(ap_next)*arad1;		
		glVertex2f(xx,yy);
						
		glColor3f(col1->red(),col1->green(),col1->blue());							
		
		xx=cos(ap_next)*arad2;
		yy=sin(ap_next)*arad2;		
		

		glVertex2f(xx,yy);		

		glEnd();	

		glShadeModel(GL_FLAT);				
		
		
		glPolygonMode(GL_FRONT,GL_LINE); 
			
		glBegin(GL_QUADS);
		
		col=&cellOutlineColour_;
		glColor3f(col->red(),col->green(),col->blue());
			
		xx=cos(ap)*arad2;
		yy=sin(ap)*arad2;		
		glVertex2f(xx,yy);
						
		xx=cos(ap)*arad1;
		yy=sin(ap)*arad1;					
		glVertex2f(xx,yy);
			
		xx=cos(ap_next)*arad1;
		yy=sin(ap_next)*arad1;		
		glVertex2f(xx,yy);
					
		xx=cos(ap_next)*arad2;
		yy=sin(ap_next)*arad2;		
			
		glVertex2f(xx,yy);		

		glEnd();	

	 	//text
		xx=cos((ap+ap_next)/2.)*(at(i).radIn_-10);
		yy=sin((ap+ap_next)/2.)*(at(i).radIn_-10);
		float angle=PI-ap;

		stringstream sst;
		sst << at(i).value_;
		
		Colour ctext=textColour_;
		
		if(i == actIndex_)
		{
			ctext=textSelectionColour_;
		}
				
		OpenGLPainter::instance()->renderText(xx,yy,ctext,7,sst.str(),angle);
								
	}

	setSlider();

	slider_.render();

	//-----------------------------------
	// Render the factor cell selector
	//-----------------------------------		
/*
	rad1=at(actIndex_).radIn_-1; 
	rad2=at(actIndex_).radOut_+1;
		
	ap=at(actIndex_).startAngle_;							
	ap_next=at(actIndex_).endAngle_;
		
	glLineWidth(cellSelectionLineWidth_);
		
	col=&at(actIndex_).selectorColour_;		
	
	glColor3f(col->red(),col->green(),col->blue());				

	glBegin(GL_LINE_LOOP);			
	
	xx=cos(ap)*rad2;
	yy=sin(ap)*rad2;
	glVertex2f(xx,yy);
		
	xx=cos(ap)*rad1;
	yy=sin(ap)*rad1;
	glVertex2f(xx,yy);
	
	xx=cos(ap_next)*rad1;
	yy=sin(ap_next)*rad1;
	glVertex2f(xx,yy);
		
	xx=cos(ap_next)*rad2;
	yy=sin(ap_next)*rad2;
	glVertex2f(xx,yy);
	
	glEnd();*/
	
}
//--------------------------------------------------
// Event handling
//--------------------------------------------------

void OpenGLMagnifierScale::event(MtInputEvent *event,float xcent, float ycent)
{	
	int x,y;
	MtMouseEvent* mev;	

	switch(event->type())
	{
	
	case Mt::MousePressEvent:
 		mev = (MtMouseEvent*) event;		
		x=mev->x()-xcent;
		y=mev->y()-ycent;
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
		{			
			input(x,y);									
		}
		
		break;
	
	
	case Mt::MouseMoveEvent:
		mev = (MtMouseEvent*) event; 
							
		if(mev->button() &  Mt::LeftButton)
		{
			x=mev->x()-xcent;
			y=mev->y()-ycent;
			if(drag_)
 			{	
				sliderDrag(x,y);	
			}
		}
					
		break;	
	
	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent*) event; 
		x=mev->x()-xcent;
		y=mev->y()-ycent;
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			 						
			if(drag_)
			{
				sliderDrag_end(x, y);			
			}
 		}
		break;
	
	default:
		break;
	
	}
	
}

//----------------------------------------
// Button1 press event
//----------------------------------------

void OpenGLMagnifierScale::input(int x,int y)
{
	//If slider selected then start slider drag
	if(slider_.checkPointInSlider(x,y) == true)
	{
		sliderDrag_start(x,y);
		return;
	}		
	
	//Scale is selected
	setActIndex(identifyPointInScale(x,y));
}

void OpenGLMagnifierScale::sliderDrag_start(int x,int y)
{			
	drag_=true;
	
	prevX_=x;
	prevY_=y;
}

void OpenGLMagnifierScale::sliderDrag(int x,int y)
{	
	if(slider_.shiftSlider(x-prevX_,y-prevY_,
                cellStartAngle_-cellAngleSize_/2., 
                cellEndAngle_+cellAngleSize_/2.))
        {        	
	        prevX_=x;
	        prevY_=y;
	}
        else
        {
                slider_.getCentre(prevX_,prevY_);
        }
        
        int xc,yc;
        slider_.getCentre(xc,yc);
          
	if(setActIndex(identifyPointInScale(x,y)) == false)
        {
                magnifier_->renderScale();
        }        
}

void OpenGLMagnifierScale::sliderDrag_end(int x,int y)
{				
	drag_=false;
	
	//setActValue();
	//executeCallback(MGLW_ValueChangedCallback,&actValueIndex_);	
}


bool OpenGLMagnifierScale::active()
{
	return drag_;
}

	