/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file BasePointsHandler.h
    \brief Definition of the Template base class AbstractPoints.
    
    Magics Team - ECMWF 2004
    
    Started: Fri 16-Jan-2004
    
    Changes:
    
*/

#ifndef AbstractPoints_H
#define AbstractPoints_H

#include "magics.h"


namespace magics {


template <class P> class AbstractPoints;

template <class P>
struct MinMaxHelper {
   MinMaxHelper(const AbstractPoints<P>& points) : 
            minX_(int_MAX), maxX_(int_MIN),
            minY_(int_MAX), maxY_(int_MIN),
            min_(int_MAX), max_(int_MIN)
    {
            points.for_each(*this);
            points.setToFirst();
            
    }
    void operator()(const P& point)
    {
        if (point.x() < minX_ ) minX_ = point.x();
        if (point.x() > maxX_ ) maxX_ = point.x();
        if (point.y() < minY_ ) minY_ = point.y();
        if (point.y() > maxY_ ) maxY_ = point.y();
        
        if (point.value() < min_ ) min_ = point.value();
        if (point.value() > max_ ) max_ = point.value();
     
    }
    double minX_;
    double maxX_;
    double minY_;
    double maxY_;
    double min_;
    double max_;
};

class BasePointsHandler 
{
public:
	BasePointsHandler() {}
	virtual ~BasePointsHandler() { };
	virtual double minX() const { return int_MIN; }
	virtual double minY() const { return int_MIN; }
	virtual double maxX() const { return int_MAX; }
	virtual double maxY() const { return int_MAX; }
	virtual double min() const  { return int_MIN; }
	virtual double max() const  { return int_MIN; }
};


template <class P>
class AbstractPoints : public BasePointsHandler {

public:
	AbstractPoints() : helper_(0) {};
	virtual ~AbstractPoints() { if (helper_) delete helper_; };
	//! Method to go through the collection. The Class 0 needs to define 
	// void operator()(const P&)
	template <class O>
	void for_each(O& object) const
	{
		setToFirst();
		while (more()) {
		  object(current());
		  advance();
		}  
	}
	//! Method to set the current position to the first point.(abstract)
	virtual void setToFirst() const {assert(false);};
	//! Method to test the end of collection.
	virtual bool more() const {assert(false);return false;};
	//! Method to return the current value
	virtual const P& current() const {assert(false);};
	//! Method to advance
	virtual void advance() const {assert(false);};

	virtual int size() const {
		points_ = 0;
		for_each(*this);
		return points_;
	}

	virtual double minX() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).minX_;
	}
	virtual double minY() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).minY_;
	}
	virtual double maxX() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).maxX_;
	}
	virtual double maxY() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).maxY_;
	}
	virtual double min() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).min_;
	}
	virtual double max() const { 
		if ( !helper_ ) helper_ = new MinMaxHelper<P>(*this); 
		return (*helper_).max_;
	}

protected:
	//! Method to print string about this class on to a stream of type ostream (virtual).
	virtual void print(ostream& out) const { out << "AbstractPoints"; }
	void operator()(const P&) const { points_++; }
	mutable MinMaxHelper<P>* helper_;
	mutable int points_;

private:
	//! Copy constructor - No copy allowed
	AbstractPoints(const AbstractPoints&);
	//! Overloaded << operator to copy - No copy allowed
	AbstractPoints& operator=(const AbstractPoints&);
    
// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const AbstractPoints<P>& p)
		{ p.print(s); return s; }
};


template <class P>
class PointsList : public magvector<P>, public AbstractPoints<P>
{
public:
	PointsList()  {}
	~PointsList() {}
	// Implements the AbstractPoints interface
	virtual void setToFirst() const	{ current_ = this->begin(); }
	virtual bool more() const	{ return current_ != this->end(); }
	virtual const P& current() const{ return *current_; }
	virtual void advance() const	{ current_++; }
	virtual int size() const	{ return magvector<P>::size(); }


protected :
	mutable typename magvector<P>::const_iterator current_;
};

} // namespace magics

#endif
