/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"

#include "syncml_internals.h"
#include "sml_transport_internals.h"

#include "transports/http_client.h"
#include "transports/http_server.h"
#include "transports/obex_client.h"
#include "transports/obex_server.h"

/**
 * @defgroup TransportPrivate Transport Private API
 * @ingroup PrivateAPI
 * @brief Private Interfaces to manage transports
 * 
 */
/*@{*/


void smlTransportSetEventCallback(SmlTransport *tsp, SmlTransportEventCb callback, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, callback, userdata);
	smlAssert(tsp);
	
	tsp->event_callback = callback;
	tsp->event_callback_userdata = userdata;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlTransportSend(SmlTransport *tsp, SmlLink *link, SmlTransportData *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, tsp, link, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	SmlTransportCommand *cmd = smlTryMalloc0(sizeof(SmlTransportCommand), error);
	if (!cmd)
		goto error;
	
	cmd->type = SML_TRANSPORT_CMD_SEND;
	cmd->data = data;
	if (link) {
		cmd->link = link;
		smlLinkRef(cmd->link);
	}
	
	smlTransportDataRef(cmd->data);
	
	//Call the fin function
	smlQueueSend(tsp->command_queue, cmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlTransportWorkerHandler(void *message, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, message, userdata);
	smlAssert(message);
	smlAssert(userdata);
	SmlTransportCommand *cmd = message;
	SmlTransport *tsp = userdata;
	
	switch (cmd->type) {
		case SML_TRANSPORT_CMD_SEND:
			tsp->functions.send(tsp->transport_data, cmd->link ? cmd->link->link_data : NULL, cmd->data, cmd->error);
			break;
		case SML_TRANSPORT_CMD_CONNECT:
			if (!tsp->functions.connect) {
				smlTransportReceiveEvent(tsp, NULL, SML_TRANSPORT_EVENT_CONNECT_DONE, NULL, NULL);
				smlTrace(TRACE_INTERNAL, "%s: No connect function", __func__);
				break;
			}
			tsp->functions.connect(tsp->transport_data);
			break;
		case SML_TRANSPORT_CMD_DISCONNECT:	
			if (!tsp->functions.disconnect) {
				smlTransportReceiveEvent(tsp, NULL, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
				smlTrace(TRACE_INTERNAL, "%s: No disconnect function", __func__);
				break;
			}
			tsp->functions.disconnect(tsp->transport_data, cmd->link ? cmd->link->link_data : NULL);
			break;
	}
	
	if (cmd->link)
		smlLinkDeref(cmd->link);
	
	if (cmd->data)
		smlTransportDataDeref(cmd->data);
	
	g_free(cmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
}

SmlBool smlTransportReceiveEvent(SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p, %p)", __func__, tsp, link, type, data, error);
	smlAssert(tsp);
	smlAssert(tsp->event_callback);
	
	SmlBool ret = tsp->event_callback(tsp, link, type, data, error, tsp->event_callback_userdata);
	
	smlTrace(TRACE_EXIT, "%s: %i", __func__, ret);
	return ret;
}

/** @brief Asynchronously runs the transport
 * 
 * This function will spawn a new thread in which the transport is
 * dispatched. The transport must be in the state "Uninitialized".
 * 
 * @param tsp The transport
 * @param error Return location if an error occured
 * @returns TRUE if the call succeded or FALSE in the case of an error
 * 
 */
SmlBool smlTransportRunAsync(SmlTransport *tsp, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, tsp, error);
	
	if (tsp->state != SML_TRANSPORT_UNINITIALIZED) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Transport was not in the state \"Uninitialized\"");
		goto error;
	}
	
	tsp->context = g_main_context_new();
	
	tsp->thread = smlThreadNew(tsp->context, error);
	if (!tsp->thread)
		goto error_free_loop;
	
	smlThreadStart(tsp->thread);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error_free_loop:
	if (tsp->context) {
		g_main_context_unref(tsp->context);
		tsp->context = NULL;
	}
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief Stops a asynchronously running transport
 * 
 * @param tsp The transport
 * 
 */
void smlTransportStop(SmlTransport *tsp)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, tsp);
	
	smlAssert(tsp->thread);

	smlThreadStop(tsp->thread);
	
	smlThreadFree(tsp->thread);
	tsp->thread = NULL;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlTransportConnect(SmlTransport *tsp, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, tsp, error);
	smlAssert(tsp);
	
	SmlTransportCommand *cmd = smlTryMalloc0(sizeof(SmlTransportCommand), error);
	if (!cmd)
		goto error;
	
	cmd->type = SML_TRANSPORT_CMD_CONNECT;
	
	//Call the connect function
	smlQueueSend(tsp->command_queue, cmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlTransportDisconnect(SmlTransport *tsp, SmlLink *link, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, link, error);
	smlAssert(tsp);
	
	SmlTransportCommand *cmd = smlTryMalloc0(sizeof(SmlTransportCommand), error);
	if (!cmd)
		goto error;
	
	cmd->type = SML_TRANSPORT_CMD_DISCONNECT;
	if (link) {
		cmd->link = link;
		smlLinkRef(cmd->link);
	}
	
	//Call the disconnect function
	smlQueueSend(tsp->command_queue, cmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlTransportType smlTransportGetType(SmlTransport *tsp)
{
	smlAssert(tsp);
	return tsp->type;
}

SmlTransportData *smlTransportDataNew(char *data, unsigned long size, SmlMimeType mimetype, SmlBool ownsData, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %d, %i, %i, %p)", __func__, data, size, mimetype, ownsData, error);
	
	//Append the data to the outgoing queue
	SmlTransportData *cmd = smlTryMalloc0(sizeof(SmlTransportData), error);
	if (!cmd)
		goto error;
	
	cmd->type = mimetype;
	cmd->data = data;
	cmd->size = size;
	cmd->ownsData = ownsData;
	cmd->refCount = 1;
	cmd->needsAnswer = TRUE;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlTransportData *smlTransportDataRef(SmlTransportData *data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	smlAssert(data);
	
	g_atomic_int_inc(&data->refCount);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return data;
}

void smlTransportDataDeref(SmlTransportData *data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	smlAssert(data);
	
	if (!g_atomic_int_dec_and_test(&data->refCount)) {
		smlTrace(TRACE_EXIT, "%s: refCount > 0", __func__);
		return;
	}
	
	if (data->ownsData)
		g_free(data->data);
	
	g_free(data);
		
	smlTrace(TRACE_EXIT, "%s: Freed", __func__);
}

void smlTransportSetError(SmlTransport *tsp, SmlLink *link, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p(%p))", __func__, tsp, link, error, error ? *error : NULL);
	smlAssert(tsp);
	
	SmlTransportCommand *cmd = smlTryMalloc0(sizeof(SmlTransportCommand), error);
	if (!cmd)
		return;
	
	cmd->type = SML_TRANSPORT_CMD_SEND;
	cmd->data = NULL;
	if (link) {
		cmd->link = link;
		smlLinkRef(cmd->link);
	}
	
	if (error && *error) {
		cmd->error = *error;
		smlErrorRef(error);
	}
	
	//Call the fin function
	smlQueueSend(tsp->command_queue, cmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlLink *smlLinkNew(SmlTransport *tsp, void *link_data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, link_data, error);
	
	SmlLink *link = smlTryMalloc0(sizeof(SmlLink), error);
	if (!link)
		goto error;
	link->tsp = tsp;
	link->link_data = link_data;
	link->refCount = 1;
	
	link->event_queue = smlQueueNew(error);
	if (!link->event_queue)
		goto error_free_link;
	
	tsp->links = g_list_append(tsp->links, link);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, link);
	return link;

error_free_link:
	smlLinkDeref(link);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlLink *smlLinkFind(SmlTransport *tsp, void *link_data)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, tsp, link_data);
	
	GList *l = NULL;
	for (l = tsp->links; l; l = l->next) {
		SmlLink *link = l->data;
		if (link->link_data == link_data) {
			smlTrace(TRACE_EXIT, "%s: %p", __func__, link);
			return link;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s: Not Found", __func__);
	return NULL;
}

SmlLink *smlLinkRef(SmlLink *link)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, link);
	smlAssert(link);
	
	g_atomic_int_inc(&link->refCount);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return link;
}

void smlLinkDeref(SmlLink *link)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, link);
	smlAssert(link);
	
	if (!g_atomic_int_dec_and_test(&link->refCount)) {
		smlTrace(TRACE_EXIT, "%s: refCount > 0", __func__);
		return;
	}
	
	if (link->event_queue)
		smlQueueFree(link->event_queue);
	
	g_free(link);
		
	smlTrace(TRACE_EXIT, "%s: Freed", __func__);
}

/*@}*/

/**
 * @defgroup TransportPublic Transport API
 * @ingroup PublicAPI
 * @brief Transports can be used to connect to other syncml capable devices and servers
 * 
 */
/*@{*/

/**
 * @name Transport Management
 * These functions allow to create, delete, initialize and finalize transports
 */
/*@{*/

/** @brief Creates a new transport
 * 
 * A transport is a abstraction of a transport type like http or obex
 * 
 * @param type The type of the transport
 * @param error Return location if an error occured
 * @returns The new transport or NULL in the case of an error
 * 
 */
SmlTransport *smlTransportNew(SmlTransportType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %p)", __func__, type, error);
	
	SmlTransport *tsp = smlTryMalloc0(sizeof(SmlTransport), error);
	if (!tsp)
		goto error;
	tsp->type = type;
	
	switch (type) {
#ifdef ENABLE_HTTP
		case SML_TRANSPORT_HTTP_SERVER:
			if (!smlTransportHttpServerNew(tsp, error))
				goto error_free_tsp;
			break;
		case SML_TRANSPORT_HTTP_CLIENT:
			if (!smlTransportHttpClientNew(tsp, error))
				goto error_free_tsp;
			break;
#else
		case SML_TRANSPORT_HTTP_SERVER:
		case SML_TRANSPORT_HTTP_CLIENT:
			smlErrorSet(error, SML_ERROR_GENERIC, "HTTP Transport not enabled in this build");
			goto error_free_tsp;
#endif
#ifdef ENABLE_OBEX
		case SML_TRANSPORT_OBEX_CLIENT:
			if (!smlTransportObexClientNew(tsp, error))
				goto error_free_tsp;
			break;
		case SML_TRANSPORT_OBEX_SERVER:
			if (!smlTransportObexServerNew(tsp, error))
				goto error_free_tsp;
			break;
#else
		case SML_TRANSPORT_OBEX_SERVER:
		case SML_TRANSPORT_OBEX_CLIENT:
			smlErrorSet(error, SML_ERROR_GENERIC, "OBEX Transport not enabled in this build");
			goto error_free_tsp;
#endif
	}
	
	tsp->command_queue = smlQueueNew(error);
	if (!tsp->command_queue)
		goto error_free_tsp;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, tsp);
	return tsp;

error_free_tsp:
	smlTransportFree(tsp);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

/** @brief Frees the given transport
 * 
 * @param tsp The transport to free
 * 
 */
void smlTransportFree(SmlTransport *tsp)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, tsp);
	smlAssert(tsp);
	
	if (tsp->command_queue)
		smlQueueFree(tsp->command_queue);
		
	if (tsp->context)
		g_main_context_unref(tsp->context);
	
	g_free(tsp);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/** @brief Initializes the transport with the given config
 * 
 * This function will init the transport with the options that you specify
 * in the config. The options that are available depend on the transport used.
 * The transport must be in the state "Uninitialized" to use this functions. The state will then
 * switch to "Initialized".
 * 
 * @param tsp The transport
 * @param config The configuration for this transport
 * @param error Return location if an error occured
 * @returns TRUE if the call succeded or FALSE in the case of an error
 * 
 */
SmlBool smlTransportInitialize(SmlTransport *tsp, const void *config, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, config, error);
	smlAssert(tsp);
	smlAssert(tsp->functions.initialize);
	
	if (tsp->state != SML_TRANSPORT_UNINITIALIZED) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Transport was not in the state \"Uninitialized\"");
		goto error;
	}
	
	//Start the queue
	smlQueueSetHandler(tsp->command_queue, (SmlQueueHandler)smlTransportWorkerHandler, tsp);
	smlQueueAttach(tsp->command_queue, tsp->context);
	
	if (!(tsp->transport_data = tsp->functions.initialize(tsp, config, error)))
		goto error_detach;
	
	tsp->state = SML_TRANSPORT_INITIALIZED;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error_detach:
	smlQueueDetach(tsp->command_queue);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief Finalizes the transport 
 * 
 * This function will finalize the transport . The transport must be in the state "Initialized" 
 * to use this functions. The state will then switch to "Uninitialized".
 * 
 * @param tsp The transport
 * @param error Return location if an error occured
 * @returns TRUE if the call succeded or FALSE in the case of an error
 * 
 */
SmlBool smlTransportFinalize(SmlTransport *tsp, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, tsp, error);
	smlAssert(tsp);
	smlAssert(tsp->functions.finalize);
	
	if (tsp->state != SML_TRANSPORT_INITIALIZED) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Transport was not in the state \"Initialized\"");
		goto error;
	}
	
	smlQueueDetach(tsp->command_queue);
	
	if (!tsp->functions.finalize(tsp->transport_data, error))
		goto error;
	
	tsp->state = SML_TRANSPORT_UNINITIALIZED;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/

/*@}*/
