/*
 * mont_ipear.c - example of how use the IP-EAR with the Dual-Core Itanium 2 PMU
 *
 * Copyright (c) 2005-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <inttypes.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_default_smpl.h>
#include <perfmon/pfmlib_montecito.h>

#define NUM_PMCS PFMLIB_MAX_PMCS
#define NUM_PMDS PFMLIB_MAX_PMDS

#define MAX_EVT_NAME_LEN	128
#define MAX_PMU_NAME_LEN	32

#define SMPL_PERIOD	(500*16)

#define EVENT_NAME	"misaligned_loads_retired"

#define M_PMD(x)		(((uint64_t)1)<<(x))
#define IPEAR_SMPL_MASK		(M_PMD(48)|M_PMD(49)|M_PMD(50)|M_PMD(51)|M_PMD(52)|M_PMD(53)|M_PMD(54)|M_PMD(55)|M_PMD(56)|\
				 M_PMD(57)|M_PMD(58)|M_PMD(59)|M_PMD(60)|M_PMD(61)|M_PMD(62)|M_PMD(63)|M_PMD(38)|M_PMD(39))
#define IPEAR_RESET_MASK	(M_PMD(38)|M_PMD(39)|IPEAR_SMPL_MASK)

typedef pfm_default_smpl_hdr_t		ipear_hdr_t;
typedef pfm_default_smpl_entry_t	ipear_entry_t;
typedef pfm_default_smpl_ctx_arg_t	ipear_ctx_arg_t;
#define IPEAR_FMT_UUID	        	PFM_DEFAULT_SMPL_UUID

static pfm_uuid_t buf_fmt_id = IPEAR_FMT_UUID;


static void *smpl_vaddr;
static unsigned long entry_size;
static int id;
static unsigned long instr_addr;
static int delay;



#if defined(__ECC) && defined(__INTEL_COMPILER)

/* if you do not have this file, your compiler is too old */
#include <ia64intrin.h>

#define hweight64(x)	_m64_popcnt(x)
#define clear_psr_ac()  __rum(1UL<<3)

#elif defined(__GNUC__)

static __inline__ int
hweight64 (unsigned long x)
{
	unsigned long result;
	__asm__ ("popcnt %0=%1" : "=r" (result) : "r" (x));
	return (int)result;
}

static inline void
clear_psr_ac(void)
{
	        __asm__ __volatile__("rum psr.ac;;" ::: "memory" );
}
#else
#error "you need to provide inline assembly from your compiler"
#endif

typedef union {
	unsigned long   l_tab[2];
	unsigned int    i_tab[4];
	unsigned short  s_tab[8];
	unsigned char   c_tab[16];
} test_data_t;

static test_data_t test_data;

static int
do_test(test_data_t *data)
{
	unsigned int *l, v;

	l = (unsigned int *)(data->c_tab+1);

	if (((unsigned long)l & 0x1) == 0) {
		printf("Data is not unaligned, can't run test\n");
		return  -1;
	}
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	v = *l;
	__asm__ __volatile__(";;nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	__asm__ __volatile__("nop.m 0x0; nop.f 0x0; nop.i 0x0;;" ::: "memory");
	v++;
	*l = v;

	return 0;
}

static void fatal_error(char *fmt,...) __attribute__((noreturn));

static void
fatal_error(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	exit(1);
}

/*
 * print content of sampling buffer
 *
 * XXX: using stdio to print from a signal handler is not safe with multi-threaded
 * applications
 */
#define safe_printf	printf

typedef struct {
	unsigned long	addr;
	uint16_t	cycles;
} ipear_t;

static ipear_t ipear_tab[16];

static void
show_ipear_reg(unsigned int idx, unsigned long etb_ext, pfm_mont_pmd_reg_t *pmd)
{
	unsigned long ef, f;
	unsigned long cycles;

	ef       = (etb_ext >> 3) & 0x1;
	f        = (etb_ext >> 2) & 0x1;

	if (ef) {
		cycles = ((etb_ext & 0x3)<< 4) | pmd->pmd48_63_ipear_ef_mont_reg.ipear_cycles;
		safe_printf("%5u: 0x%016lx f=%lu ef=%lu cycles=%lu delay=%u\n",
			idx,
			(unsigned long)pmd->pmd48_63_ipear_ef_mont_reg.ipear_addr << 12,
			f, ef, cycles,
			(unsigned int)pmd->pmd48_63_ipear_ef_mont_reg.ipear_delay);
	} else {
		cycles = ((etb_ext & 0x3)<< 4) | pmd->pmd48_63_ipear_mont_reg.ipear_cycles;
		safe_printf("%5u: 0x%016lx f=%lu ef=%lu cycles=%lu\n",
			idx,
			(unsigned long)pmd->pmd48_63_ipear_mont_reg.ipear_addr << 4,
			f, ef, cycles);
	}
}

static int
build_ipear_reg(ipear_t *ent, unsigned long etb_ext, pfm_mont_pmd_reg_t *pmd)
{
	unsigned long ef, f;

	ef       = (etb_ext >> 3) & 0x1;
	f        = (etb_ext >> 2) & 0x1;

	if (f || ef) return -1;

	if (ef) {
		ent->addr   = pmd->pmd48_63_ipear_ef_mont_reg.ipear_addr << 12;
		ent->cycles = ((etb_ext & 0x3)<< 4) | pmd->pmd48_63_ipear_ef_mont_reg.ipear_cycles;
	} else {
		ent->addr   = pmd->pmd48_63_ipear_mont_reg.ipear_addr << 4;
		ent->cycles = ((etb_ext & 0x3)<< 4) | pmd->pmd48_63_ipear_mont_reg.ipear_cycles;
	}
	return 0;
}

static inline int
exist_addr(unsigned long addr, ipear_t *tab, int last)
{
	int e;

	for(e=0; e < last; e++) {
		if (tab[e].addr == addr) return e;
	}
	return -1;
}

static void
process_smpl_buffer(void)
{
	static unsigned long last_ovfl = ~0UL;
	ipear_hdr_t *hdr;
	ipear_entry_t *ent;
	unsigned long pos, etb_ext;
	unsigned long smpl_entry = 0;
	pfm_mont_pmd_reg_t *reg, pmd39, pmd38;
	int ret;
	int i, j, e, p, last, d, int_instr, ret_instr;

	hdr = (ipear_hdr_t *)smpl_vaddr;

	/*
	 * check that we are not diplaying the previous set of samples again.
	 * Required to take care of the last batch of samples.
	 */
	if (hdr->hdr_overflows <= last_ovfl && last_ovfl != ~0UL) {
		printf("skipping identical set of samples %lu <= %lu\n", hdr->hdr_overflows, last_ovfl);
		return;
	}

	pos = (unsigned long)(hdr+1);

	/*
	 * walk through all the entries recored in the buffer
	 */
	for(i=0; i < hdr->hdr_count; i++) {

		ret = 0;

		ent = (ipear_entry_t *)pos;
		/*
		 * print entry header
		 */
		safe_printf("------------------------------------------------\n");
		safe_printf("Entry %ld CPU:%d IIP:0x%016lx\n",
				smpl_entry++,
				ent->cpu,
				ent->ip);

		/*
		 * point to first recorded register (always contiguous with entry header)
		 */
		reg = (pfm_mont_pmd_reg_t*)(ent+1);
		pmd38.pmd_val = reg->pmd_val;
		reg++;
		pmd39.pmd_val = reg->pmd_val;
		reg++;

		if (instr_addr == 0) {
			safe_printf("PMD38: 0x%016lx full=%u ebi=%u\n",
				pmd38.pmd_val,
				pmd38.pmd38_mont_reg.etbi_full,
				pmd38.pmd38_mont_reg.etbi_ebi);
			safe_printf("PMD39: 0x%016lx\n", reg->pmd_val);
		}

		e    = pmd38.pmd38_mont_reg.etbi_full ? pmd38.pmd38_mont_reg.etbi_ebi : 0;
		last =  pmd38.pmd38_mont_reg.etbi_ebi;
		j    = 0;

		if (instr_addr == 0) {
			do {
				if (e < 8)
					etb_ext = (pmd39.pmd_val>>(8*e)) & 0xf;
				else
					etb_ext = (pmd39.pmd_val>>(4+8*(e-1))) & 0xf;

				show_ipear_reg(j, etb_ext, reg+e);
				e = (e+1) % 16;
				j++;
			} while (e != last);
		} else {
			do {
				if (e < 8)
					etb_ext = (pmd39.pmd_val>>(8*e)) & 0xf;
				else
					etb_ext = (pmd39.pmd_val>>(4+8*(e-1))) & 0xf;
				ret = build_ipear_reg(ipear_tab+j, etb_ext, reg+e);
				e = (e+1) % 16;
				j++;
			} while (ret == 0 && e != last);
			last = j;

			if (ret == -1) {
				printf("<early freeze/flush>>\n");
				goto skip;
			}

#if 0
			for(e=0; e < last; e++) {
				printf("%2d 0x%016lx %u\n", e, ipear_tab[e].addr, ipear_tab[e].cycles);
			}
#endif
			safe_printf("------------------------------------------------\n");

			int_instr = exist_addr(instr_addr, ipear_tab, last);
			if (int_instr == -1) {
				safe_printf("<address 0x%lx not found>\n", instr_addr);
				goto skip;
			}
			for(p = 0; p < 10; p++) {

				for(e=last-1, d=0; e > -1; e--) {

					if (ipear_tab[e].cycles)
						d += ipear_tab[e].cycles;
					else
						d++;
					//safe_printf("%2d 0x%016lx %3u %3u\n", e, ipear_tab[e].addr, ipear_tab[e].cycles, d); 
					if (d >= delay) break;
				}
				if (d != delay) {
					safe_printf("<<mismatch delay=%u pstage=%u>>\n", delay, p);
					continue;
				}
				ret_instr = e;

				for(e=last-1, d=0; e >-1; e--) {
					d += ipear_tab[e].cycles ? ipear_tab[e].cycles : 1;
					safe_printf("%2d 0x%016lx %3u %3u", e, ipear_tab[e].addr, ipear_tab[e].cycles, d);
					if (e == int_instr) safe_printf(" <== instr of interest");
					if (e == ret_instr) safe_printf(" <== retired at freeze (%d)", delay);
					safe_printf("\n");
				}
				if (ret_instr > int_instr) {
					safe_printf("<<mismatch stage>>\n");
					goto skip;
				}
				safe_printf("<<GOOD>>\n");
			}

		}
skip:
		/*
		 * move to next entry
		 */
		pos += entry_size;
	}
}

static void
overflow_handler(int n, struct siginfo *info, struct sigcontext *sc)
{
	process_smpl_buffer();
	/*
	 * And resume monitoring
	 */
	if (perfmonctl(id, PFM_RESTART, NULL, 0))
		fatal_error("pfm_restart errno %d\n", errno);
}

int
main(int argc, char **argv)
{
	pfarg_reg_t pd[NUM_PMDS];
	pfarg_reg_t pc[NUM_PMCS];
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfmlib_mont_input_param_t mont_inp;
	ipear_ctx_arg_t ctx;
	ipear_ctx_arg_t buf_arg;
	pfarg_load_t load_args;
	pfmlib_options_t pfmlib_options;
	struct sigaction act;
	unsigned int i, ev, j;
	int ret, type = 0;

	/*
	 * Initialize pfm library (required before we can use it)
	 */
	if (pfm_initialize() != PFMLIB_SUCCESS) {
		fatal_error("Can't initialize library\n");
	}

	delay      = argc > 1 ? atoi(argv[1]) : 5;
	instr_addr = argc > 2 ? strtoul(argv[2], NULL, 0)  : 0;
	printf("delay=%d instr_addr=0x%lx\n", delay, instr_addr);

	/*
	 * Let's make sure we run this on the right CPU
	 */
	pfm_get_pmu_type(&type);
	if (type != PFMLIB_MONTECITO_PMU) {
		char model[MAX_PMU_NAME_LEN];
		pfm_get_pmu_name(model, MAX_PMU_NAME_LEN);
		fatal_error("this program does not work with %s PMU\n", model);
	}

	/*
	 * Install the overflow handler (SIGIO)
	 */
	memset(&act, 0, sizeof(act));
	act.sa_handler = (sig_t)overflow_handler;
	sigaction (SIGIO, &act, 0);

	/*
	 * pass options to library (optional)
	 */
	memset(&pfmlib_options, 0, sizeof(pfmlib_options));
	pfmlib_options.pfm_debug = 0; /* set to 1 for debug */
	pfmlib_options.pfm_verbose = 1; /* set to 1 for debug */
	pfm_set_options(&pfmlib_options);


	memset(pd, 0, sizeof(pd));
	memset(pc, 0, sizeof(pc));
	memset(&ctx, 0, sizeof(ctx));
	memset(&buf_arg, 0, sizeof(buf_arg));
	memset(&load_args, 0, sizeof(load_args));

	/*
	 * prepare parameters to library. we don't use any Itanium
	 * specific features here. so the pfp_model is NULL.
	 */
	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));

	/*
	 * To count the number of occurence of this instruction, we must
	 * program a counting monitor with the IA64_TAGGED_INST_RETIRED_PMC8
	 * event.
	 */
	if (pfm_find_event_byname(EVENT_NAME, &ev) != PFMLIB_SUCCESS) {
		fatal_error("cannot find event %s\n", EVENT_NAME);
	}

	/*
	 * set the (global) privilege mode:
	 * 	PFM_PLM0 : kernel level only
	 */
	inp.pfp_dfl_plm   = PFM_PLM3;

	/*
	 * how many counters we use
	 */
	inp.pfp_event_count = 1;

	/*
	 * propagate the event descriptor
	 */
	inp.pfp_events[0].event = ev;
	
	memset(&mont_inp,0, sizeof(mont_inp));
	mont_inp.pfp_mont_ipear.ipear_used  = 1;
	mont_inp.pfp_mont_ipear.ipear_plm   = inp.pfp_dfl_plm;
	mont_inp.pfp_mont_ipear.ipear_mode  = PFMLIB_MONT_IPEAR_MODE_IPEAR;
	mont_inp.pfp_mont_ipear.ipear_delay = delay;

	/*
	 * let the library figure out the values for the PMCS
	 *
	 * We use all global settings for this EAR.
	 */
	if ((ret=pfm_dispatch_events(&inp, &mont_inp, &outp, NULL)) != PFMLIB_SUCCESS) {
		fatal_error("cannot configure events: %s\n", pfm_strerror(ret));
	}

	/*
	 * prepare context structure.
	 *
	 * format specific parameters MUST be concatenated to the regular
	 * pfarg_context_t structure. For convenience, the default sampling
	 * format provides a data structure that already combines the pfarg_context_t
	 * with what is needed fot this format.
	 */

	 /*
	  * We initialize the format specific information.
	  * The format is identified by its UUID which must be copied
	  * into the ctx_buf_fmt_id field.
	  */
	memcpy(ctx.ctx_arg.ctx_smpl_buf_id, buf_fmt_id, sizeof(pfm_uuid_t));

	/*
	 * the size of the buffer is indicated in bytes (not entries).
	 *
	 * The kernel will record into the buffer up to a certain point.
	 * No partial samples are ever recorded.
	 */
	ctx.buf_arg.buf_size = getpagesize();


	/*
	 * now create the context for self monitoring/per-task
	 */
	if (perfmonctl(0, PFM_CREATE_CONTEXT, &ctx, 1) == -1 ) {
		if (errno == ENOSYS) {
			fatal_error("Your kernel does not have performance monitoring support!\n");
		}
		fatal_error("Can't create PFM context %s\n", strerror(errno));
	}
	/*
	 * extract the file descriptor we will use to
	 * identify this newly created context
	 */
	id = ctx.ctx_arg.ctx_fd;

	/*
	 * retrieve the virtual address at which the sampling
	 * buffer has been mapped
	 */
	smpl_vaddr = ctx.ctx_arg.ctx_smpl_vaddr;
	if (smpl_vaddr == MAP_FAILED) {
		fatal_error("cannot mmap sampling buffer errno %d\n", errno);
	}

	printf("Sampling buffer mapped at %p\n", smpl_vaddr);

	/*
	 * Now prepare the argument to initialize the PMDs and PMCS.
	 * We must pfp_pmc_count to determine the number of PMC to intialize.
	 * We must use pfp_event_count to determine the number of PMD to initialize.
	 * Some events causes extra PMCs to be used, so  pfp_pmc_count may be >= pfp_event_count.
	 *
	 * This step is new compared to libpfm-2.x. It is necessary because the library no
	 * longer knows about the kernel data structures.
	 */

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	/*
	 * figure out pmd mapping from output pmc
	 */
	for (i=0, j=0; i < inp.pfp_event_count; i++) {
		pd[i].reg_num   = outp.pfp_pmcs[j].reg_pmd_num;
		for(; j < outp.pfp_pmc_count; j++)  if (outp.pfp_pmcs[j].reg_evt_idx != i) break;
	}

	printf("programming %u PMCs and %u PMDs\n", outp.pfp_pmc_count, inp.pfp_event_count);

	/*
	 * indicate we want notification when buffer is full
	 */
	pc[0].reg_flags |= PFM_REGFL_OVFL_NOTIFY | PFM_REGFL_RANDOM;

	/*
	 * indicate which PMD to include in the sample
	 */
	pc[0].reg_smpl_pmds[0]  = IPEAR_SMPL_MASK;
	pc[0].reg_reset_pmds[0] = IPEAR_RESET_MASK;
	pd[0].reg_random_seed   = 5;
	pd[0].reg_random_mask   = 0xff;

	/*
	 * compute size of each sample: fixed-size header + all our DEAR regs
	 */
	entry_size = sizeof(ipear_entry_t)+(hweight64(IPEAR_SMPL_MASK)<<3);

	printf("smpl_pmds=0x%lx reset_pmds=0x%lx entry_size=%zu\n",
		pd[0].reg_smpl_pmds[0],
		pd[0].reg_reset_pmds[0], entry_size);

	/*
	 * initialize the PMD and the sampling period
	 */
	pd[0].reg_value       = - SMPL_PERIOD;
	pd[0].reg_long_reset  = - SMPL_PERIOD;
	pd[0].reg_short_reset = - SMPL_PERIOD;

	/*
	 * Now program the registers
	 *
	 * We don't use the save variable to indicate the number of elements passed to
	 * the kernel because, as we said earlier, pc may contain more elements than
	 * the number of events we specified, i.e., contains more thann coutning monitors.
	 */
	if (perfmonctl(id, PFM_WRITE_PMCS, pc, outp.pfp_pmc_count))
		fatal_error("pfm_write_pmcs error errno %d\n",errno);

	if (perfmonctl(id, PFM_WRITE_PMDS, pd, inp.pfp_event_count))
		fatal_error("pfm_write_pmds error errno %d\n",errno);

	/*
	 * attach context to stopped task
	 */
	load_args.load_pid = getpid();
	if (perfmonctl(id, PFM_LOAD_CONTEXT, &load_args, 1))
		fatal_error("pfm_load_context error errno %d\n",errno);

	/*
	 * setup asynchronous notification on the file descriptor
	 */
	ret = fcntl(id, F_SETFL, fcntl(id, F_GETFL, 0) | O_ASYNC);
	if (ret == -1)
		fatal_error("cannot set ASYNC: %s\n", strerror(errno));

	/*
	 * get ownership of the descriptor
	 */
	ret = fcntl(id, F_SETOWN, getpid());
	if (ret == -1)
		fatal_error("cannot setown: %s\n", strerror(errno));

	clear_psr_ac();
	/*
	 * Let's roll now.
	 */
	pfm_self_start(id);

	for(i=0; i < 1000000; i++) {
		do_test(&test_data);
	}

	pfm_self_stop(id);

	/*
	 * We must call the processing routine to cover the last entries recorded
	 * in the sampling buffer, i.e. which may not be full
	 */
	process_smpl_buffer();

	/*
	 * let's stop this now
	 */
	close(id);
	return 0;
}
