/* 
 * pam_rsa PAM-module for local authentication with RSA keypairs
 * copyright (c) 2006 Vesa-Matti Kari <hyperllama@laamanaama.helsinki.fi>
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include "util.h"
#include "pam_rsa.h"
#include "conf.h"

static void opt_pamrsaconf_init(struct opt_pamrsaconf *o);
static int privkey_name_hash_set(hashtype *ht, const char *s);
static int pamrsaconf_parse(struct pamrsaconf *cf, FILE *fp, unsigned int *errs);
static int already_defined(const char *opt, int *o, unsigned int *errs, int line);
static int exactly_two_tokens(const char *s);


static int privkey_name_hash_set(hashtype *ht, const char *s)
{
	if (!strcasecmp(s, "sha1")) {
		*ht = HASH_SHA1;
		return 0;
	} else if (!strcasecmp(s, "none")) {
		*ht = HASH_NONE;
		return 0;
	} 

	return -1;
}


static int has_non_ws(const char *s)
{
	if (s == NULL) {
		pamrsa_log(LOG_ALERT, "impossible internal error: has_non_ws(NULL)");
		return 0;
	}

	if (*s == '\0') {
		return 0;	/* Empty string has no non-whitespace characters */
	}

	while (*s != '\0') {
		if (*s != ' ' && *s != '\t') {
			return 1;
		}
		++s;
	}

	return 0;
}


static int exactly_two_tokens(const char *s)
{
	if (s == NULL) {
		pamrsa_log(LOG_ALERT, "impossible internal error: exactly_two_tokens(NULL)");
		return 0;
	}

	/* skip initial whitespace if any */
	while (*s != '\0' && (*s == ' ' || *s == '\t')) {
			++s;	
	}
	if (*s == '\0') {
		/* no tokens so fail */
		return 0;
	}

	/* at the start of first token, scan until whitespace is seen */
	while (*s != '\0' && *s != ' ' && *s != '\t') {
			++s;	
	}
	if (*s == '\0') {	
		/* only one token so fail */
		return 0;
	}

	/* now the skip whitespace */
	while (*s != '\0' && (*s == ' ' || *s == '\t')) {
			++s;	
	}
	if (*s == '\0') {	
		/* still only one token so fail */
		return 0;
	}

	/* at the start of second token, scan until whitespace is seen */
	while (*s != '\0' && *s != ' ' && *s != '\t') {
			++s;	
	}
	if (*s == '\0') {
		/* got exactly two tokens so accept */
		return 1;
	}

	/* now allow whitespace but reject everything else */
	while (*s != '\0' && (*s == ' ' || *s == '\t')) {
			++s;	
	}
	if (*s == '\0') {
		/* still got exactly two tokens so accept */
		return 1;
	}

	/* got two tokens and trailing garbage so reject */
	return 0;
}


static int pamrsaconf_parse(struct pamrsaconf *cf, FILE *fp, unsigned int *errs)
{
	struct opt_pamrsaconf o;
	char buf[LINE_LEN];
	char opt[LINE_LEN];
	char val[LINE_LEN];
	char *p;
	unsigned int li;
	int r;
	char *ret;

	opt_pamrsaconf_init(&o); /* for catching option redefinitions */
	li = 0;

	while ((ret = fgets(buf, sizeof(buf), fp)) != NULL) {

		if (++li >= CONF_MAXLINES) {
			++*errs;
			pamrsa_log(LOG_ALERT, "too many lines, must be less than %d", CONF_MAXLINES);
			return CONF_PARSEFAIL;
		}

		if ((p = strchr(buf, '\n')) != NULL) {
			*p = '\0';
		} else {
			++*errs;
			pamrsa_log(LOG_ALERT, "line %d: too long, starts with '%.32s'...", li, buf);
			continue;
		}

		/* reject strings that contain suspicious characters */
		if (!is_safestr(buf)) {
			++*errs;
			pamrsa_log(LOG_ALERT, "line %d: contains suspicious characters", li, buf);
			continue;
		}

		/* We ignore lines that have:
           1) only whitespace 
           2) only whitespace followed by a comment
		 */

		if ((p = strchr(buf, '#')) != NULL) {
			*p = '\0';
		}

		if (has_non_ws(buf) == 0) {
			/* ignore blank lines */
			continue;
		}

		if (exactly_two_tokens(buf) == 0) {
			/* line contains garbage */	
			++*errs;
			pamrsa_log(LOG_ALERT, "line %d: malformed '%s'", li, buf);
			continue;
		}

		memset(opt, '\0', sizeof(opt));
		memset(val, '\0', sizeof(val));

		if (sscanf(buf, "%s %s", opt, val) != 2) {
			++*errs;
			pamrsa_log(LOG_ALERT, "line %d: malformed '%s'", li, buf);
		} else if (!strcmp(opt, "pubkey_dir")) {
			if (already_defined(opt, &o.pubkey_dir, errs, li)) {
				continue;
			}
			if (val[0] != '/') {
				++*errs;
				pamrsa_log(LOG_ALERT, "line %d: pubkey_dir '%s' path not absolute", li, val);
			} else {
				if ((cf->pubkey_dir = strdup(val)) == NULL) {
					pamrsa_log(LOG_CRIT, "memory allocation failure");
					return CONF_SYSFAIL;
				}
			}

		} else if (!strcmp(opt, "privkey_dir")) {
			if (already_defined(opt, &o.privkey_dir, errs, li)) {
				continue;
			}
			if (val[0] != '/') {
				++*errs;
				pamrsa_log(LOG_ALERT, "line %d: privkey_dir '%s' path not absolute", li, val);
			} else {
				if ((cf->privkey_dir = strdup(val)) == NULL) {
					pamrsa_log(LOG_CRIT, "memory allocation failure");
					return CONF_SYSFAIL;
				}
			}

		} else if (!strcmp(opt, "pam_prompt")) {
			if (already_defined(opt, &o.pam_prompt, errs, li)) {
				continue;
			}

			if (strlen(val) >= CONF_PAM_PROMPT_MAXLEN) {
				pamrsa_log(LOG_ERR, "%s value must be less than %d characters long",
					opt, CONF_PAM_PROMPT_MAXLEN);
					++*errs;
			} else {
				if ((cf->pam_prompt = strdup(val)) == NULL) {
					pamrsa_log(LOG_CRIT, "memory allocation failure");
					return CONF_SYSFAIL;
				}
			}

		} else if (!strcmp(opt, "privkey_name_hash")) {
			if (already_defined(opt, &o.privkey_name_hash, errs, li)) {
				continue;
			}
			if ((r = privkey_name_hash_set(&(cf->privkey_name_hash), val)) != 0) {
				++*errs;
				pamrsa_log(LOG_ALERT, "%s: line %d: privkey_name_hash value '%s' unknown", li, val);

			}
		} else if (!strcmp(opt, "log_auth_result")) {
			int *b;
			if (already_defined(opt, &o.log_auth_result, errs, li)) {
				continue;
			}
			b = &(cf->log_auth_result);

			if (set_bool(b, val) == 0) {
				++*errs;
				pamrsa_log(LOG_ALERT, "line %d: log_auth_result '%s' not boolean", li, val);
			}

		} else {
				++*errs;
				pamrsa_log(LOG_ALERT, "line %d: option '%s' unknown", li, opt);
		}

	}


	if (ret == NULL && ferror(fp)) {
		++*errs;
		return CONF_SYSFAIL;
	} 

	if (*errs > 0) {
		return CONF_PARSEFAIL;
	}

	return CONF_SUCCESS;
}



int pamrsaconf_read(struct pamrsaconf *cf, const char *fn)
{
	FILE *fp;
	int r;
	char ebuf[512];
	unsigned int errs;

	if ((fp = fopen(fn, "r")) == NULL) {
		pamrsa_log(LOG_ALERT, "could not open configuration file %s: %s",
			fn, STRERROR(errno));
		return -1;	
	}

	errs = 0;
	r = pamrsaconf_parse(cf, fp, &errs);

	if (r == CONF_SYSFAIL) {	
		pamrsa_log(LOG_ALERT, "%s: %s", fn, STRERROR(errno));
	}

	if (r == CONF_SYSFAIL || r == CONF_PARSEFAIL) {
		pamrsa_log(LOG_ALERT, "%d error%s while processing configuration file %s",
				errs, (errs == 1)?"":"s", fn);
	}
	
	if (fclose(fp) == EOF) {
		pamrsa_log(LOG_ALERT, "could not close configuration file %s: %s", 
				fn, STRERROR(errno));
		return -2;
	}

	return (r == CONF_SUCCESS)?0:-1;
}


/* Configuration options that take boolean values 
   are given their default arguments in this first 
   initialization in pamrsaconf_init(). All other
   defaults are set after pamrsaconf_read()
   in pamrsaconf_set_defaults(). */

struct pamrsaconf *pamrsaconf_alloc(void)
{
	struct pamrsaconf *cf;
	if ((cf = malloc(sizeof(*cf))) == NULL) {
		return NULL;
	}
	cf->log_auth_result = 1;
	cf->privkey_name_hash = HASH_INITIALIZED;
	cf->pubkey_dir = NULL;
	cf->privkey_dir = NULL;
	cf->pam_prompt = NULL;

	return cf;
}

int pamrsaconf_set_defaults(struct pamrsaconf *cf)
{
	if (cf->pubkey_dir == NULL) {
		if ((cf->pubkey_dir = strdup(PUBKEYDIR_DEFAULT)) == NULL) {
			pamrsa_log(LOG_CRIT, "memory allocation failure");
			return 0;
		}
	}

	if (cf->privkey_dir == NULL) {
		if ((cf->privkey_dir = strdup(PRIVKEYDIR_DEFAULT)) == NULL) {
			pamrsa_log(LOG_CRIT, "memory allocation failure");
			return 0;
		}
	}

	if (cf->pam_prompt == NULL) {
		if ((cf->pam_prompt = strdup(PROMPT_DEFAULT)) == NULL) {
			pamrsa_log(LOG_CRIT, "memory allocation failure");
			return 0;
		}
	}

	if (cf->privkey_name_hash == HASH_INITIALIZED) {
		cf->privkey_name_hash = HASH_SHA1;
	}

	return 1;
}


void pamrsaconf_free(struct pamrsaconf *cf)
{
	if (cf != NULL) {
		free(cf->pubkey_dir);
		free(cf->privkey_dir);
		free(cf->pam_prompt);
	}
	free(cf);
}


struct pamrsaarg *pamrsaarg_alloc(void)
{
	struct pamrsaarg *ar;
	if ((ar = malloc(sizeof(*ar))) == NULL) {
		return NULL;
	}

	ar->debug= 0;
	ar->ask_pass = 0;
	ar->ask_passphrase = 0;

	return ar;
}


void pamrsaarg_free(struct pamrsaarg *ar)
{
	free(ar);
}


static void opt_pamrsaconf_init(struct opt_pamrsaconf *o)
{
	memset(o, 0, sizeof(*o));

	o->pubkey_dir = 0;
	o->privkey_dir = 0;
	o->pam_prompt = 0;
	o->privkey_name_hash = 0;
	o->log_auth_result = 0;
}

/* Catch option redefinitions in the module's configuration file */
static int already_defined(const char *opt, int *o, unsigned int *errs, int line)
{
	if (*o == 1) {
		++*errs;
		pamrsa_log(LOG_ALERT, "line %d: %s redefinition not allowed", line, opt);
		return 1;
	} else {
		*o = 1;
		return 0;
	}
}


int pamrsaarg_read(struct pamrsaarg *ar, int argc, const char **argv)
{
	unsigned int errs;

	errs = 0;
	for (; argc-- > 0; ++argv) {
		if (!is_safestr(*argv)) {
			pamrsa_log(LOG_ERR, "module argument contains suspicious characters");
			++errs;	
		}

		else if (!strcmp("debug", *argv)) {
			if (ar->debug) {
				pamrsa_log(LOG_WARNING, "module argument '%s' given more than once", *argv);
			} else {
				ar->debug = 1;
			}
		}

		else if (!strcmp("ask_pass", *argv)) {
			if (ar->ask_pass) {
				pamrsa_log(LOG_WARNING, "module argument '%s' given more than once", *argv);
			} else {
				if (ar->ask_passphrase) {
					pamrsa_log(LOG_ALERT, "module argument %s conflicts with ask_passphrase", *argv);
					++errs;
				}
				ar->ask_pass = 1;
			}
		}

		else if (!strcmp("ask_passphrase", *argv)) {
			if (ar->ask_passphrase) {
				pamrsa_log(LOG_WARNING, "module argument '%s' given more than once", *argv);
			} else {
				if (ar->ask_pass) {
					pamrsa_log(LOG_ALERT, "module argument %s conflicts with ask_pass", *argv);
					++errs;
				}
				ar->ask_passphrase = 1;
			}
		}

		else {
			pamrsa_log(LOG_ERR, "module argument '%s' unknown", *argv);
			++errs;	
		}
	}

	if (errs > 0) {
		pamrsa_log(LOG_ERR, "encountered %d error%s while reading arguments",
			errs, (errs == 1)?"":"s");
		return -1;
	}

	return 0;
}
