// $Id: legacy_xdr_io.h 2560 2007-12-03 17:52:20Z benkirk $

// The libMesh Finite Element Library.
// Copyright (C) 2002-2007  Benjamin S. Kirk, John W. Peterson
  
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
  
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
  
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



#ifndef __xdr_io_h__
#define __xdr_io_h__


// C++ inludes
#include <string>
#include <vector>

// Local includes
#include "libmesh.h"
#include "mesh_input.h"
#include "mesh_output.h"

// Forward declarations
class MeshBase;
class MeshData;
class Xdr;
class Elem;


/**
 *
 * @author Benjamin Kirk, John Peterson, 2004.
 */

// ------------------------------------------------------------
// XdrIO class definition
class XdrIO : public MeshInput<MeshBase>,
	      public MeshOutput<MeshBase>
{

 public:

  /**
   * Constructor.  Takes a writeable reference to a mesh object.
   * This is the constructor required to read a mesh.
   * The optional parameter \p binary can be used to switch
   * between ASCII (\p false, the default) or binary (\p true)
   * files.
   */
  XdrIO (MeshBase&,       const bool=false);

  /**
   * Constructor.  Takes a reference to a constant mesh object.
   * This constructor will only allow us to write the mesh.
   * The optional parameter \p binary can be used to switch
   * between ASCII (\p false, the default) or binary (\p true)
   * files.
   */
  XdrIO (const MeshBase&, const bool=false);
  
  /**
   * Destructor.
   */
  virtual ~XdrIO ();
  
  /**
   * This method implements reading a mesh from a specified file.
   */
  virtual void read (const std::string&);
    
  /**
   * This method implements writing a mesh to a specified file.
   */
  virtual void write (const std::string&);

  /**
   * Get/Set the flag indicating if we should read/write binary.
   */
  bool   binary() const { return _binary; };
  bool & binary()       { return _binary; };
  
  /**
   * Get/Set the flag indicating if we should read/write legacy.
   */
  bool   legacy() const { return _legacy; };
  bool & legacy()       { return _legacy; };
  
  /**
   * Get/Set the version string.  Vailid version strings:
     \verbatim

     "libMesh-0.7.0+" 
     "libMesh-0.7.0+ parallel"

     \endverbatim
     If "libMesh" is not detected in the version string the
     \p LegacyXdrIO class will be used to read older 
     (pre version 0.7.0) mesh files.
   */
  const std::string & version () const { return _version; };
  std::string &       version ()       { return _version; };

  /**
   * Get/Set the boundary condition file name.
   */
  const std::string & boundary_condition_file_name() const { return _bc_file_name; };
  std::string &       boundary_condition_file_name()       { return _bc_file_name; };

  /**
   * Get/Set the partitioning file name.
   */
  const std::string & partition_map_file_name() const { return _partition_map_file; };
  std::string &       partition_map_file_name()       { return _partition_map_file; };

  /**
   * Get/Set the subdomain file name.
   */
  const std::string & subdomain_map_file_name() const { return _subdomain_map_file; };
  std::string &       subdomain_map_file_name()       { return _subdomain_map_file; };

  /**
   * Get/Set the polynomial degree file name.
   */
  const std::string & polynomial_level_file_name() const { return _p_level_file; };
  std::string &       polynomial_level_file_name()       { return _p_level_file; };


 private:


  //---------------------------------------------------------------------------
  // Write Implementation  
  /**
   * Write the connectivity for a parallel, distributed mesh
   */
  void write_serialized_connectivity (Xdr &io, const unsigned int n_elem) const;

  /**
   * Write the nodal locations for a parallel, distributed mesh
   */
  void write_serialized_nodes (Xdr &io, const unsigned int n_nodes) const;

  /**
   * Write the boundary conditions for a parallel, distributed mesh
   */
  void write_serialized_bcs (Xdr &io, const unsigned int n_bcs) const;


  
  //---------------------------------------------------------------------------
  // Read Implementation
  /**
   * Read the connectivity for a parallel, distributed mesh
   */
  void read_serialized_connectivity (Xdr &io, const unsigned int n_elem);

  /**
   * Read the nodal locations for a parallel, distributed mesh
   */
  void read_serialized_nodes (Xdr &io, const unsigned int n_nodes);

  /**
   * Read the boundary conditions for a parallel, distributed mesh
   */
  void read_serialized_bcs (Xdr &io);

  //-------------------------------------------------------------------------
  /**
   * Pack an element into a transfer buffer for parallel communication.
   */
  void pack_element (std::vector<unsigned int> &conn, 
		     const Elem *elem, 
		     const unsigned int parent_id  = libMesh::invalid_uint,
		     const unsigned int parent_pid = libMesh::invalid_uint) const;

  bool _binary;
  bool _legacy;
  std::string _version;
  std::string _bc_file_name;
  std::string _partition_map_file;
  std::string _subdomain_map_file;
  std::string _p_level_file;

  /**
   * Define the block size to use for chunked IO.
   */
  static const unsigned int io_blksize;
};






#endif // #define __xdr_io_h__
