/*
 * libjingle
 * Copyright 2004--2005, Google Inc.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "talk/base/httpcommon-inl.h"

#include "talk/base/asyncsocket.h"
#include "talk/base/common.h"
#include "talk/base/httpclient.h"
#include "talk/base/logging.h"
#include "talk/base/socketstream.h"
#include "talk/base/stringutils.h"

namespace cricket {

//////////////////////////////////////////////////////////////////////
// HttpClient
//////////////////////////////////////////////////////////////////////

HttpClient::HttpClient(const char* agent, StreamPool* pool)
    : agent_(agent), pool_(pool), fail_redirect_(false),
      transparent_proxy_(false) {
  base_.notify(this);
}

HttpClient::~HttpClient() {
  base_.notify(NULL);
  base_.abort(HE_SHUTDOWN);
  release();
}

void
HttpClient::reset() {
  request_.clear();
  response_.clear();
  base_.abort(HE_OPERATION_CANCELLED);
}

void
HttpClient::call(const char* server, uint16 port) {
  if (base_.mode() != HM_NONE) {
    ASSERT(false);
    return;
  }

  // Exact size must be known on the client.  Instead of using chunked
  // encoding, wrap data with auto-caching file or memory stream.
  std::string temp;
  if (request_.hasHeader("Transfer-Encoding", temp)) {
    ASSERT(false);
    return;
  }
  // If no content has been specified, using length of 0.
  request_.setHeader("Content-Length", "0", false);
  
  request_.setHeader("User-Agent", agent_, false);
  request_.setHeader("Connection", "Keep-Alive", false);
  if (_strnicmp(request_.path.c_str(), "http", 4) == 0) {
    request_.setHeader("Proxy-Connection", "Keep-Alive", false);
  }

  char buffer[1024];
  if (port == HTTP_DEFAULT_PORT) {
    sprintfn(buffer, sizeof(buffer), "%s", server);
  } else {
    sprintfn(buffer, sizeof(buffer), "%s:%d", server, port);
  }
  request_.setHeader("Host", buffer);

  // When using a transparent proxy, we need to use canonical url form
  if (transparent_proxy_
      && (_strnicmp(request().path.c_str(), "http://", 7) != 0)) {
    request().version = HVER_1_0;
    std::string canonical_path("http://");
    canonical_path.append(buffer);
    canonical_path.append(request().path);
    request().path = canonical_path;
  }

  int err;
  SocketAddress remote(server, port, false);
  StreamInterface* stream = pool_->RequestConnectedStream(remote, &err);
  if (stream == NULL) {
    if (err) LOG(LS_ERROR) << "RequestConnectedStream returned: " << err;
    onHttpComplete(HM_CONNECT, (err == 0) ? HE_NONE : HE_SOCKET);
  } else {
    base_.attach(stream);
    if (stream->GetState() == SS_OPEN) {
      base_.send(&request_);
    }
  }
}

void
HttpClient::get(const char* url) {
  reset();
  Url<char> purl(url);
  request().verb = HV_GET;
  request().path = purl.path();
  call(purl.server(), purl.port());
}

void
HttpClient::post(const char* url, const char* content_type,
                 StreamInterface* request_doc) {
  reset();
  Url<char> purl(url);
  request().verb = HV_POST;
  request().path = purl.path();
  request().setContent(content_type, request_doc);
  call(purl.server(), purl.port());
}

void
HttpClient::release() {
  if (StreamInterface* stream = base_.detach()) {
    pool_->ReturnConnectedStream(stream);
  }
}

//
// HttpBase Implementation
//

HttpError
HttpClient::onHttpHeaderComplete(size_t& data_size) {
  uint32 scode = response_.scode;
  uint32 group = scode / 100;
  if ((request_.verb == HV_HEAD) || (group == 1)
      || (scode == HC_NO_CONTENT)
      || (scode == HC_NOT_MODIFIED)) {
    data_size = 0;
  } else if (group == 3) {
    base_.ignore_data();
  }
  return HE_NONE;
}

void
HttpClient::onHttpComplete(HttpMode mode, HttpError err) {
  if (err != HE_NONE) {
    // fall through
  } else if (mode == HM_CONNECT) {
    base_.send(&request_);
    return;
  } else if (mode == HM_SEND) {
    base_.recv(&response_);
    return;
  } else {
    // TODO(bpm): Move keep-alive logic to function, and implement
    // on server as well.
    std::string value;
    bool keep_alive;
    if ((response_.hasHeader("Proxy-Connection", value)
        || response_.hasHeader("Connection", value))) {
      keep_alive = (_stricmp(value.c_str(), "keep-alive") == 0);
    } else {
      keep_alive = (response_.version >= HVER_1_1);
    }
    if (!keep_alive) {
      LOG(INFO) << "HttpClient: closing socket";
      base_.stream()->Close();
    }
    if (is_redirect(response_.scode)) {
      if (!response_.hasHeader("Location", value)) {
        err = HE_PROTOCOL;
      } else if (!fail_redirect_) {
        Url<char> purl(value.c_str());
        request_.path = purl.path();
        if (response_.scode == HC_SEE_OTHER) {
          request_.verb = HV_GET;
          // TODO: remove content-length/content-type headers?
          request_.document.reset();
        } else if (request_.document.get() && !request_.document->Rewind()) {
          err = HE_STREAM;
        }
        if (err == HE_NONE) {
          release();
          call(purl.server(), purl.port());
          return;
        }
      }
    }
  }
  release();
  SignalHttpClientComplete(this, err);
}

void
HttpClient::onHttpClosed(HttpError err) {
  SignalHttpClientClosed(this, err);
}

} // namespace cricket
