/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <sstream>
#include <sys/stat.h>//for file size checking
#include <sys/time.h>//for time querying
#include <math.h> //for stringToFloat

#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdutils.h>

#undef LOG_HEADER
#define LOG_HEADER "Error inside Utils method: \n"
#include <libinstrudeo/isdloggermacros.h>

ISDUtils* ISDUtils::instance = NULL;//init instance

//-----PUBLIC METHODS-----
ISDUtils* ISDUtils::getInstance()
{
    //create an instance if necessary
    if (instance==NULL){
	instance = new ISDUtils();
    }

    return instance;
}
ISDObject::ISDErrorCode ISDUtils::createTempFile(string& name)
{
    char templateString[] = TEMP_FILE_TEMPLATE;
    if (mkstemp(templateString)==-1){
	RETURN_ERROR(ISD_FILE_ERROR);
    }
    else{
	name = templateString;
	RETURN_SUCCESS;
    }
}
ISDObject::ISDErrorCode ISDUtils::createTempDir(string& name)
{
    char* dir;
    char templateString[] = TEMP_DIR_TEMPLATE;
    if ((dir=mkdtemp(templateString))==NULL){
	RETURN_ERROR(ISD_FILE_ERROR);
    }
    else{
	name = string(dir);
	name = name + DIR_DELIM;
	RETURN_SUCCESS;
    }
}
string ISDUtils::intToString(int i)
{
    stringstream s;
    s << i;
    return s.str();
}
int ISDUtils::stringToInt(string& numStr)
{
    return atoi(numStr.c_str());
}
string ISDUtils::floatToString(float f)
{
    double intPart, floatPart;
    floatPart = modf(f, &intPart);
    //floatPart contains the floating point section,
    // but with the 0. included, so remove that
    return intToString((int)intPart) + "." + intToString((int)(floatPart*10));
    
    /*
     * Don't use this anymore, since we changed the stringToFloat method and need to be sure
     * we are compatible with that function.
     *
     * TODO: This is an ugly solution, needs fixing with locales.
     *
     * stringstream s;
     * s << f;
     * return s.str();
     */
}
float ISDUtils::stringToFloat(string& numStr)
{
    /*
     * Note: the function strtod(numStr.c_str(), (char**)NULL) returned
     *       0 instead of 0.5 (leaving .0 in the pointer).
     *       I have no reason for this and I couldn't find a solution,
     *       so I implemented my own (inferior) method.
     * SOLUTION: cause of problems was a different locale on the machine (using , as a decimal separator)
     */
    int dotPos = numStr.find_last_of(".");
    int commaPos = numStr.find_last_of(",");

    //if no dot or comma is present, it is an integer
    if (dotPos == -1 && commaPos == -1) {
	return stringToInt(numStr);
    }

    /*
     * Here, we know there is a dot or a comma present.
     * We can't assume both for abvious reasons.
     */
    int sepPos = dotPos==-1?commaPos:dotPos;
    
    string temp = numStr.substr(0, sepPos);
    int intPart = stringToInt(temp);
    temp = numStr.substr(sepPos+1);
    float floatPart = (float)stringToInt(temp);
    int decBase = (int)pow((double)10, (double)temp.size());

    return intPart + floatPart/decBase;
}
string ISDUtils::getFileDirectory(string fileName)
{
    return fileName.substr(0, fileName.find_last_of(DIR_DELIM)) + DIR_DELIM;
}

string ISDUtils::stripFileDirectories(string fileName)
{
    return fileName.substr(fileName.find_last_of(DIR_DELIM)+1);
}

list<Glib::ustring> ISDUtils::explodeString(Glib::ustring str, Glib::ustring substr)
{
    list<Glib::ustring> retVal;
    int iPos = str.find(substr, 0);
    int iPit = substr.length();
    
    while(iPos>-1) {
	if(iPos!=0) {
	    retVal.push_back(str.substr(0, iPos));
	}
	/*
	 * If a single substring is encountered, add a dummy entry to the list
	 * This way, newlines (\n, etc.) will be interpreted correctly
	 */
	if (str.substr(0, iPos+iPit)==substr) {
	    retVal.push_back("");
	}
	str.erase(0, iPos+iPit);
	iPos = str.find(substr, 0);
    }
    
    if(str!="")
	retVal.push_back(str);
    
    return retVal;
}
off64_t ISDUtils::getFileSizeInBytes(string& fileName)
{
    if (!VALID_FILE(fileName)) {
	LOG_WARNING("Requesting the file size of an unexisting file.");
	return -1;
    }
    
    struct stat fileStat;
    stat(fileName.c_str(), &fileStat);
    return fileStat.st_size;
}
string ISDUtils::getExtension(string fileName)
{
    return fileName.substr(fileName.find_last_of('.')+1);
}
string ISDUtils::getRandomString()
{
    /*
     * WARNING: this method is used in IPD and to this date,
     *          no clash-checking is performed. So if you intent
     *          to change this method, keep in mind to re-examine
     *          the main routine of IPD!!!
     */

    struct timeval now;
    gettimeofday(&now, NULL);
    stringstream s;
    s << now.tv_sec << now.tv_usec;
    return s.str();
}
bool ISDUtils::sendMail(string& recipient, string& subject, string& body)
{
    string cmd = string(MAIL_PROGRAM) + " -s '" + subject + "' " + recipient;
    FILE *mail = popen(cmd.c_str(), "w");
    if (!mail) {
	return false;
    }
    
    fprintf(mail, body.c_str());
    
    if (pclose(mail)) {
	return false;
    }

    return true;
}

//-----(PRIVATE) CONSTRUCTORS-----
ISDUtils::ISDUtils()
    : ISDObject()
{
    
}
