/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDGLOBJFILE_H
#define ISDGLOBJFILE_H

/**
 * Abstract superclass that wraps around an OpenGL (Wavefront) .obj file (and around the glm.h file).
 * To create a new obj file, eg. in Blender (<a href="http://www.blender.org">www.blender.org</a>),
 * create two mesh-objects: the first is the real 3D object that needs to be drawn,
 * the second is a 2D-plane in 3D space that defines where the text can be drawn.
 * Specify both object-names in the appropriate variables of the constrtuctor.
 * Note: depending on your mesh, if you export both objects in one pass, it is possible
 * that the edges and faces are corrupted. To workaround this, export both objects to 
 * seperate files and merge them to one file (assuming you know a thing or two about
 * Wavefronts .obj file format, if not, look into it, it isn't difficult).
 *
 * IMPORTANT NOTE: The ordering of the vertices in the texbox plane is counterclockwise,
 *                 starting with the top left node, indicating the plane is facing forward.
 *                 Make sure this is correct in the exported file.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <libinstrudeo/isdobject.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include <libinstrudeo/glm.h>

using namespace std;

/*
 * Defines the value to where the "unit" of the unitize function will be scaled
 * to have a more comfortable number to work with.
 */
#define MODEL_UNIT_SCALE 100
#define DEFAULT_SMOOTHING_ANGLE 90.0

/*
 * Plane faces forward; nodes are ordered counterclockwise
 * note: *3 for three dimensions
 */
#define TOP_LEFT 1*3
#define BOTTOM_LEFT 2*3
#define BOTTOM_RIGHT 3*3
#define TOP_RIGHT 4*3

class ISDGLObjFile : public ISDObject
{
 public:
    /**
     * Loads in a .obj file, initialises variables and does some preprocessing
     *
     * @param fileName The filename of the .obj file.
     * @param meshObjName The name of the main mesh-object in the .obj file.
     * @param tectboxObjName The name of the 2D-plane that specifies the bounds
     *        of the textbox in the mesh.
     */
    ISDGLObjFile(string fileName, string meshObjName, string textboxObjName);

    virtual ~ISDGLObjFile();

    /**
     * Returns the main model that contains the mesh and the textplane.
     * This method is usually not necessary for general-use, but is used
     * inside the isdcommentbox-subclasses to adjust or tweak different
     * values of the model (eg. the colors or materials).
     *
     * @return The model or NULL if an error occurred.
     */
    GLMmodel* getMainModel();

    /**
     * Returns the main-mesh-object of this file or NULL if something went wrong.
     *
     * @return The object or NULL if an error occurred.
     */
    GLMobject* getMeshObject();

    /**
     * Returns the main-mesh-object GL display-list of this file or 0 if something went wrong.
     *
     * @return The display-list for the object or 0 if something went wrong.
     */
    GLuint getMeshObjectList();

    /**
     * Returns the text-plane-object (that defines the text-area) of this file
     * or NULL if something went wrong.
     *
     * @return A reference to the object or NULL if an error occurred.
     */
    GLMobject* getTextPlaneObject();

    /**
     * Returns the upper left corner of the textplane object in GL coordinates.
     *
     * @param orig The array that will be used to return the x, y and z position.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getTextPlaneOrigin(float orig[3]);

    /**
     * Returns the width and height of the textplane object in GL coordinates.
     *
     * @param width Returns the width of the textplane.
     * @param height Returns the height of the textplane.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getTextPlaneDimensions(float* width, float* height);

    /**
     * Returns the three dimensions of the model.
     *
     * @param width The width is returned.
     * @param height The height is returned.
     * @param depth The depth is returned.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getDimensions(float* width, float* height, float* depth);

    /**
     * This method is an alternative for the GL lists, returned by the getMeshObjectList()
     * method. If you want to change (eg.) the material or color between displays, lists
     * are not very useful, since they fixate the color in the list.
     * This method isn't as efficient, but provides more flexibility, so think it over
     * before using it in favor of GL lists.
     *
     * @param mode The mode to draw;
     *             a bitwise OR of values describing what is to be rendered.
     *             GLM_NONE    -  render with only vertices
     *             GLM_FLAT    -  render with facet normals
     *             GLM_SMOOTH  -  render with vertex normals
     *             GLM_TEXTURE -  render with texture coords
     *             GLM_FLAT and GLM_SMOOTH should not both be specified.
     *             See glm.h (method draw()) for more details.
     */
    void drawModel(GLuint mode);

 protected:
    //-----CONSTANTS-----
    enum { X, Y, Z, W };
    enum {
	MATERIAL_NONE,
	MATERIAL_COLOR,
	MATERIAL_OBJECT
    };

    //-----METHODS-----
    /**
     * Reads the .obj file, creates and calculates the model structure.
     *
     * @param fileName The filename of the .obj file.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode initModel(string fileName);

    /**
     * Creates the GL display-list for the model, depending on the material or color.
     * Note: the underlying glm-file routines return 0 if something went wrong, 
     * so this value can be used on the modelList variable to test for errors.
     *
     * @param model The (initialised) model.
     * @param materialMode Specifies if the model has a material, color, ...
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode createLists(GLMmodel* model, GLuint materialMode = MATERIAL_NONE);

    /**
     * Initialises different variables regarding the 2D textbox plane in the model.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode initTextbox();
    
    //-----VARIABLES-----
    GLMmodel* model;
    GLfloat smoothingAngle;
    GLuint modelList;
    /*
     * Draw the model with facet normal?
     */
    GLboolean facetNormal;
    GLMobject* meshObject;
    GLMobject* textPlaneObject;
    GLfloat textboxOrigin[3];
    GLfloat textboxWidth;
    GLfloat textboxHeight;
};

#endif
