/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdprogresscallback.h>
#include <libinstrudeo/isdvideocanvas.h>
#include <libinstrudeo/isdvideoproperties.h>
#include <libinstrudeo/isdrecording.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdffmpegexporter.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while exporting recording using the FFmpeg library: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTRUCTORS-----
ISDFFmpegExporter::ISDFFmpegExporter(string outFileName, ISDRecording* rec,
				     Glib::ustring* lang, ISDProgressCallback* callbackClass,
				     int quality, float framerate)
    : ISDExporter(outFileName, rec, lang, callbackClass, quality, framerate),
      codec(NULL),
      codecContext(NULL),
      inputFrame(NULL),
      inputBuf(NULL),
      outputBuf(NULL),
      outputContext(NULL),
      outputFormat(NULL),
      videoStream(NULL)
{
    if (initializeEncoder() != ISD_SUCCESS) {
	LOG_WARNING("Error while initialising the FFmpeg encoder.");
	lastError = ISD_EXPORT_INIT_ERROR;
	return;
    }
    initOK = true;
}

//-----DESTRUCTOR-----
ISDFFmpegExporter::~ISDFFmpegExporter()
{
    cleanup();
}

//-----PUBLIC OVERLOADED METHODS-----
ISDObject::ISDErrorCode ISDFFmpegExporter::doExport(char* pixelBuffer)
{
    char* grabBuffer;
    
    if (!initOK) {
	LOG_WARNING("Export routine called without initializing the encoder.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }

    ISDVideoProperties* videoProperties = recording->getVideoProperties();

    /*
     * If we're not using the (offscreen) pixelbuffer that's controlled by the calling program,
     * we need to create a buffer to grab the pixels to.
     */
    if (pixelBuffer==NULL) {
	int bufSize = videoProperties->getHeight()*videoProperties->getWidth()*videoProperties->getBytesPerPixel();
	grabBuffer = (char*)malloc(bufSize);
    }

    /*
     * Note: video time is in milliseconds, so if we have a framerate
     * of 25, we must increment the loop with 1000/25 milliseconds
     */
    bool noVideoChange, noCommentChange;
    int outSize, ret;
    for (int pos=0; pos<recording->getVideoProperties()->getLength();pos+=(int)(1000.0/framerate)) {
	//change the position
	recording->update(pos, lang, noVideoChange, noCommentChange);

	//force a redisplay (essential for mesa)
	// Note: the getCurrentFrame-method already does this
	//recording->getVideoCanvas()->display();

	/*
	 * Get raw pixel data source:
	 * If we use offscreen rendering mesa, we don't need (and can't)
	 * grab the framebuffer. Instead, we must use our own provided buffer.
	 */
	if (pixelBuffer==NULL) {
	    recording->getVideoCanvas()->getCurrentFrame(grabBuffer);
	    //insert the buffer in the image structure
	    avpicture_fill(&inputPicture, (uint8_t*)grabBuffer, inputPixFmt, codecContext->width, codecContext->height);
	}
	else {
	    //insert the buffer in the image structure
	    avpicture_fill(&inputPicture, (uint8_t*)pixelBuffer, inputPixFmt, codecContext->width, codecContext->height);
	}
	    
	//convert input pic to yuv420p
	if (img_convert(&yuvInputPicture, FFMPEG_STREAM_PIX_FMT, &inputPicture, 
			inputPixFmt, codecContext->width, codecContext->height) < 0)
	    {
		LOG_WARNING("Pixel format conversion not handled while exporting.");
		RETURN_ERROR(ISD_EXPORT_FORMAT_ERROR);
	    }
	    
	outSize = avcodec_encode_video(codecContext, outputBuf, outputBufSize, inputFrame);
	//if zero size, it means the image was buffered
	if (outSize > 0) {
	    AVPacket pkt;
	    av_init_packet(&pkt);
		
	    //rescale a 64bit integer by 2 rational numbers
	    pkt.pts= av_rescale_q(codecContext->coded_frame->pts, codecContext->time_base, videoStream->time_base);
	    if(codecContext->coded_frame->key_frame)
		pkt.flags |= PKT_FLAG_KEY;
	    pkt.stream_index = videoStream->index;
	    pkt.data = outputBuf;
	    pkt.size = outSize;
		
	    //write the compressed frame to the media file
	    ret = av_write_frame(outputContext, &pkt);
		
	    if (ret != 0){
		LOG_WARNING("Error while writing video frame.");
		RETURN_ERROR(ISD_EXPORT_FILE_ERROR);
	    }
	}
	    
	//notify and check the callback
	if (callbackClass!=NULL) {
	    if (callbackClass->wasCancelled()) {
		abortCleanup();
		break;
	    }
	    else {
		callbackClass->procentDone((float)pos/(float)videoProperties->getLength()*100.0);
	    }
	}
    }

    //free the pixelbuffer if necessary
    if (pixelBuffer==NULL) {
	free(grabBuffer);
    }

    /*
     * Set the position to 0, this is not always right (if the pos was
     * other than 0), but it's a good second-best action, since we
     * don't keep track of the position.
     */
    recording->update(0, lang, noVideoChange, noCommentChange);

    RETURN_SUCCESS;
}

//-----PROTECTED OVERLOADED METHODS-----
void ISDFFmpegExporter::cleanup()
{
    if (videoStream != NULL){
	avcodec_close(videoStream->codec);
	videoStream = NULL;
    }
    
    if (outputContext != NULL){
	//write trailer
	av_write_trailer(outputContext);
	//free stream(s) ... probably always only one
        for (int i=0;i<outputContext->nb_streams;i++) {
	    av_freep(&outputContext->streams[i]->codec);
            av_freep(&outputContext->streams[i]);
	}
	//close file
        url_fclose (&outputContext->pb);
	
	//free format context
        av_free (outputContext);
        outputContext = NULL;
    }
    
    //free buffers
    if (inputBuf){
	av_free(inputBuf);
	inputBuf = NULL;
    }
    if (outputBuf){
	av_free(outputBuf); //avcodec seems to do that job?
	outputBuf = NULL;
    }
    if (inputFrame){
	free(inputFrame);
	inputFrame = NULL;
    }
    
    codecContext = NULL;
    codec = NULL;
}

//-----PROTECTED METHODS-----
void ISDFFmpegExporter::abortCleanup()
{
    //delete the file
    REMOVE_FILE(fileName);
}
ISDObject::ISDErrorCode ISDFFmpegExporter::initializeEncoder()
{
    ///initialize libavcodec, and register all codecs and formats
    av_register_all();

    outputFormat = guess_format(NULL, fileName.c_str(), NULL);
    if (!outputFormat) {
	LOG_INFO("Could not determine output format from file extension: using MPEG.");
	outputFormat = guess_format(DEFAULT_FFMPEG_EXPORT_FORMAT, NULL, NULL);
    }
    
    if (!outputFormat){
	LOG_WARNING("Could not find suitable output format.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }

    /*
     * For some extensions, we don't use the default guessed output codec.
     */
    string ext = ISDUtils::getInstance()->getExtension(fileName);
    if (ext=="avi") {
	outputFormat = guess_format("avi", NULL, NULL);
	/* Manually override the codec id. */
	outputFormat->video_codec = CODEC_ID_MSMPEG4V2;
    }

    //allocate the output media context
    outputContext = av_alloc_format_context();
    if (!outputContext){
	LOG_WARNING("Output context memory allocation error.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }
    outputContext->oformat = outputFormat;
    //filename is limited to 1024 chars, trim if more
    snprintf(outputContext->filename, sizeof(outputContext->filename), "%s", fileName.c_str());

    //create the video stream
    if (outputFormat->video_codec != CODEC_ID_NONE) {
        if (addVideoStream(outputFormat->video_codec) != ISD_SUCCESS) {
	    LOG_WARNING("Error while adding the video stream.");
	    RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
	}
    }
    
    //set the output parameters (must be done even if no parameters).
    if (av_set_parameters(outputContext, NULL) < 0) {
	LOG_WARNING("Invalid output format parameters.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }

    outputContext->max_delay = 700;
    
    //for debugging purposes; dump stream-information to stdout
    //dump_format(outputContext, 0, fileName.c_str(), 1);

    //now that all the parameters are set, we can open the video codecs
    //and allocate the necessary encode and input buffers
    if (allocInputPicture() != ISD_SUCCESS) {
	LOG_WARNING("Error while allocating input picture.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }
    if (videoStream) {
        if (openVideo() != ISD_SUCCESS) {
	    LOG_WARNING("Error while opening new video stream.");
	    RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
	}
    }
    
    //open the file
    if (url_fopen (&outputContext->pb, fileName.c_str(), URL_WRONLY) < 0){
	LOG_WARNING("Could not open output file.");
	RETURN_ERROR(ISD_EXPORT_FILE_ERROR);
    }
    
    //write the stream header, if any
    if (av_write_header(outputContext) < 0){
	LOG_WARNING("Could not write header for output file (incorrect codec paramters ?).");
	RETURN_ERROR(ISD_EXPORT_FILE_ERROR);
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDFFmpegExporter::addVideoStream(CodecID codecId)
{
    videoStream = av_new_stream(outputContext, 0);
    if (!videoStream) {
	LOG_WARNING("Error while allocating/creating a new video stream.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }
    
    codecContext = videoStream->codec;
    codecContext->codec_id = codecId;
    codecContext->codec_type = CODEC_TYPE_VIDEO;

    //Set video resolution; must be a multiple of two.
    codecContext->width = recording->getVideoProperties()->getWidth();
    codecContext->height = recording->getVideoProperties()->getHeight();
    
    //if the video resolution isn't a multiple of two, clip it
    if (codecContext->width%2) {
	LOG_INFO("Video width wasn't a multiple of two, clipping.");
	codecContext->width--;
    }
    if (codecContext->height%2) {
	LOG_INFO("Video height wasn't a multiple of two, clipping.");
	codecContext->height--;
    }
	
    //Calculate bitrate.
    codecContext->bit_rate = (codecContext->width * codecContext->height * 
			       (((((codecContext->height + codecContext->width)/ 100)- 5)>> 1)+ 10) * quality ) / 100;
    if (codecContext->bit_rate < 300000) codecContext->bit_rate = 300000;

    /*
     * time base: this is the fundamental unit of time (in seconds) in terms
     * of which frame timestamps are represented. for fixed-fps content,
     * timebase should be 1/framerate and timestamp increments should be
     * identically 1.
     */
    codecContext->time_base.den = (int)(framerate*FFMPEG_FRAMERATE_PRECISION);
    codecContext->time_base.num = FFMPEG_FRAMERATE_PRECISION;
    codecContext->gop_size = FFMPEG_GOP_VALUE; //emit one intra frame every GOP_VALUE frames at most
    codecContext->pix_fmt = FFMPEG_STREAM_PIX_FMT;
    if (codecContext->codec_id == CODEC_ID_MPEG2VIDEO) {
        // just for testing, we also add B frames
        codecContext->max_b_frames = 2;
    }

    if (codecContext->codec_id == CODEC_ID_MPEG1VIDEO) {
        /*
	 * needed to avoid using macroblocks in which some coeffs overflow 
	 * this doesnt happen with normal video, it just happens here as the 
	 * motion of the chroma plane doesnt match the luma plane
	 */
        codecContext->mb_decision = 2;
    }

    // some formats want stream headers to be seperate
    if(!strcmp(outputContext->oformat->name, "mp4") ||
       !strcmp(outputContext->oformat->name, "mov") ||
       !strcmp(outputContext->oformat->name, "3gp"))
	{
	    codecContext->flags |= CODEC_FLAG_GLOBAL_HEADER;
	}

    return ISD_SUCCESS;
}
ISDObject::ISDErrorCode ISDFFmpegExporter::allocInputPicture()
{    
    ISDVideoProperties* videoProperties = recording->getVideoProperties();
    //determine input picture format
    //endianness is treated by avcodec
    switch (videoProperties->getBytesPerPixel()*8) {
    case 8:
	inputPixFmt = PIX_FMT_PAL8;
	break;
    case 16:
	if (videoProperties->getRedMask() == 0xF800 &&
	    videoProperties->getGreenMask() == 0x07E0 &&
	    videoProperties->getBlueMask() == 0x1F)
	    {
		inputPixFmt = PIX_FMT_RGB565;
	    }
	else if (videoProperties->getRedMask() == 0x7C00 &&
		   videoProperties->getGreenMask() == 0x03E0 &&
		   videoProperties->getBlueMask() == 0x1F)
	    {
		inputPixFmt = PIX_FMT_RGB555;
	    }
	else {
	    LOG_WARNING("RGB ordering at specified image depth, 16, not supported.");
	    RETURN_ERROR(ISD_EXPORT_FORMAT_ERROR);
	}
	break;
    case 24:
	if (videoProperties->getRedMask() == 0xFF0000 &&
	    videoProperties->getGreenMask() == 0xFF00 &&
	    videoProperties->getBlueMask() == 0xFF)
	    {
		inputPixFmt = PIX_FMT_BGR24;
	    }
	else if (videoProperties->getRedMask() == 0xFF &&
		 videoProperties->getGreenMask() == 0xFF00 &&
		 videoProperties->getBlueMask() == 0xFF0000)
	    {
		inputPixFmt = PIX_FMT_RGB24;
	    }
	else {
	    LOG_WARNING("RGB ordering at specified image depth, 24, not supported.");
	    RETURN_ERROR(ISD_EXPORT_FORMAT_ERROR);
	}
	break;
    case 32:
	inputPixFmt = PIX_FMT_RGBA32;
	break;
    default:
	LOG_WARNING("RGB ordering at specified image depth not supported.");
	RETURN_ERROR(ISD_EXPORT_FORMAT_ERROR);
    }
    
    return ISD_SUCCESS;
}
ISDObject::ISDErrorCode ISDFFmpegExporter::openVideo()
{
    codecContext = videoStream->codec;

    //find the video encoder
    codec = avcodec_find_encoder(codecContext->codec_id);
    if (!codec){
	LOG_WARNING("Codec not found.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }

    //open the codec
    if (avcodec_open(codecContext, codec) < 0) {
	LOG_WARNING("Could not open codec.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }
    
    //calculate the size of the resulting YUV image
    outputBufSize = avpicture_get_size(FFMPEG_STREAM_PIX_FMT, codecContext->width, codecContext->height);
    
    //allocate the buffer for the YUV-converted input image
    inputBuf = (uint8_t*)malloc(outputBufSize);
    if (!inputBuf){
	LOG_WARNING("Could not allocate buffer for YUV pic.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }
    avpicture_fill(&yuvInputPicture, inputBuf, FFMPEG_STREAM_PIX_FMT, codecContext->width, codecContext->height);
    
    //allocate input-frame for the encoder
    inputFrame = avcodec_alloc_frame();
    inputFrame->type = FF_BUFFER_TYPE_SHARED;
    inputFrame->data[0] = inputBuf;
    inputFrame->data[1] = inputFrame->data[0] + ( codecContext->width * codecContext->height);
    inputFrame->data[2] = inputFrame->data[1] + ( codecContext->width * codecContext->height) / 4;
    inputFrame->data[3] = NULL;
    inputFrame->linesize[0] = codecContext->width;
    inputFrame->linesize[1] = codecContext->width / 2;
    inputFrame->linesize[2] = codecContext->width / 2;
    inputFrame->linesize[3] = 0;
	
    //allocate output buffer for encoded frames
    outputBuf = (uint8_t*)malloc(outputBufSize);
    if (!outputBuf){
	LOG_WARNING("Could not allocate buffer for encoded frame.");
	RETURN_ERROR(ISD_EXPORT_INIT_ERROR);
    }

    return ISD_SUCCESS;
}
