/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDDATAFILE_H
#define ISDDATAFILE_H

/**
 * This class represents the data file in the instrudeo file format.
 * It abstacts the access to the file by a specified api.
 * 
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>
#include <fstream>
#include <map>

#include <libinstrudeo/isdobject.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing instrudeo data file: \n"
#include <libinstrudeo/isdloggermacros.h>

using namespace std;

class ISDRectangle;
class ISDVideoProperties;
class ISDVideoCanvas;

class ISDDataFile  : public ISDObject
{
 public:
    //-----CONSTANTS-----
    #define ISD_MAGIC_TEMPLATE "instrudeo*.*"
    #define ISD_MAGIC "instrudeo0.1"
    #define MAX_MAJOR_SUPPORTED_VERSION 0
    #define MAX_MINOR_SUPPORTED_VERSION 1

    #define ISD_FORMAT_BITS_PER_PIXEL 32
    #define ISD_FORMAT_COLOUR_DEPTH 24
    #define ISD_FORMAT_BIG_ENDIAN 0
    #define ISD_FORMAT_TRUE_COLOUR 1
    #define ISD_FORMAT_RED_MAX 255
    #define ISD_FORMAT_GREEN_MAX 255
    #define ISD_FORMAT_BLUE_MAX 255
    #define ISD_FORMAT_RED_SHIFT 16
    #define ISD_FORMAT_GREEN_SHIFT 8
    #define ISD_FORMAT_BLUE_SHIFT 0

    //ARGB (as formatted above), on a little endian machine (as formatted above)
    //   results in BGRA
    #define ISD_GL_PIXEL_FORMAT GL_BGRA
    #define ISD_GL_PIXEL_TYPE GL_UNSIGNED_BYTE


    //-----TYPEDEFS-----
    typedef unsigned int U32;
    typedef unsigned short U16;
    typedef unsigned char U8;

    typedef struct {
	U8 bitsPerPixel;        /* 8,16,32 only */
	U8 depth;               /* 8 to 32 */
	U8 bigEndian;		/* True if multi-byte pixels are interpreted
				   as big endian, or if single-bit-per-pixel
				   has most significant bit of the byte
				   corresponding to first (leftmost) pixel. Of
				   course this is meaningless for 8 bits/pix */
	U8 trueColour;		/* If false then we need a "colour map" to
				   convert pixels to RGB.  If true, xxxMax and
				   xxxShift specify bits used for red, green
				   and blue */
	/* the following fields are only meaningful if trueColour is true */
	U16 redMax;	      	/* maximum red value (= 2^n - 1 where n is the
				   number of bits used for red). Note this
				   value is always in big endian order. */
	U16 greenMax;		/* similar for green */
	U16 blueMax;		/* and blue */
	U8 redShift;		/* number of shifts needed to get the red
				   value in a pixel to the least significant
				   bit. To find the red value from a given
				   pixel, do the following:
				   1) Swap pixel value according to bigEndian
				   (e.g. if bigEndian is false and host byte
				   order is big endian, then swap).
				   2) Shift right by redShift.
				   3) AND with redMax (in host byte order).
				   4) You now have the red value between 0 and
				   redMax. */
	U8 greenShift;		/* similar for green */
	U8 blueShift;		/* and blue */
	
	U8 pad1;
	U16 pad2;
    } ISDPixelFormat;

#define ISD_PIXEL_FORMAT_SIZE 16
    
    typedef struct {
	U16 framebufferWidth;
	U16 framebufferHeight;
	ISDPixelFormat format;
    } ISDFileHeader;

#define ISD_FILE_HEADER_SIZE 20

    typedef struct {
	U32 frameTime;
	U16 x;
	U16 y;
	U16 w;
	U16 h;
	U32 datalen;
	U8 encType; /* 
		      * 0 = raw pixel data
		      * 1 = rle encoding
		      */
	U32 seekBackPos;
	// followed by the data
    } ISDRectangleHeader;

#define ISD_RECT_HEADER_SIZE 21

    enum rectangleEncodings {
	//raw pixel encoding
	ISD_RECT_ENCODING_RAW,
	//run-length encoding
	ISD_RECT_ENCODING_RLE
    };

    //-----CONSTRUCTORS-----
    /**
     * Load in a new binary file, allocate and init the videoProperties.
     * The videoProperties-object will be created and returned through the pointer.
     * The seekBacks are calculated by default.
     * This improves the random-access to the videostream a lot,
     * but takes longer to load. If you will only process the data in
     * a forward sequence, it can speed up things to disable seekBack calculation.
     * However, for most purposes, it is advisable, despite the load penalty.
     *
     * @param fileName The filename of the binary file.
     * @param videoProperties Will return the video properties or NULL if something went wrong.
     * @param calcSeekBacks Specifies if the seekBacks of the rects must be calculated or not.
     */
    ISDDataFile(string dataFileName, ISDVideoProperties*& videoProperties, bool calcSeekBacks = true);

    ~ISDDataFile();

    //-----METHODS-----
    /**
     * Returns the filename (on disk) of the data file that is used.
     *
     * @return The fileName
     */
    string getFileName();

    /**
     * Returns the magic string of this version of the instrudeo file type.
     *
     * @return The magic string
     */
    static string getMagicString();

    /**
     * Positions the current video at position pos and updates the provided videoCanvas
     * at that position, if necessary. The position is the time in milliseconds and must be
     * between the videos time-bounds.
     *
     * @param videoCanvas The canvas that needs to be updated with new data from the file.
     * @param pos The position in the videostream.
     * @param noChange Is set to true if the image didn't change since last update.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode updateCanvas(ISDVideoCanvas* videoCanvas, int pos, bool& noChange);

    /**
     * This method provides a hook into the stream of rectangles.
     * ISDRectangle::getNext() and ISDRectangle::getPrevious() can be used
     * to navigate through the stream.
     *
     * @return The first rectangle-pointer, or NULL on error.
     */
    ISDRectangle* getFirstRectangle();

    /**
     * Returns the compressed pixel data, belonging to the provided rectangle.
     * Note: the supplied buffer must be allocated.
     *
     * @param rect The rect header of the data.
     * @param data Pointer to the buffer where the data will be read into.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getPixelData(ISDRectangle* rect, char* data);

 protected:
    //-----METHODS-----
    /**
     * Reads in general rectangle info and sets up the inter-linking between the rects
     * The videoProperties-object will be created during this method.
     * 
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode preprocessData();

    /**
     * Reads the magic code string from the input file and stores it in the supplied string.
     * The string must contain a template of the magic code, so it can be used to determine how much data must be read
     * eg. instrudeo*.* (* will be filled with major and minor version numbers)
     * 
     * @param magic Is used to return the magic string.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readMagic(string& magic);

    /**
     * Reads in and processes the file header and stores it in the fileHeader variable.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readFileHeader();

    /**
     * Reads the next header in the supplied datastructure.
     *
     * @param header The header struct that must be filled with data.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readRectHeader(ISDRectangleHeader& header);

    /**
     * Reads data from the datafile into the databuffer.
     * The data buffer must be allocated (at least size rect->dataLen)
     *
     * @param rect The rect header of the data.
     * @param data Pointer to the buffer where the data will be read into.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readData(ISDRectangle* rect, char* data);

    //-----VARIABLES-----
    string fileName;
    ifstream inputStream;
    ISDFileHeader fileHeader;
    ISDRectangle* rectListHead;
    ISDRectangle* rectListTail;
    ISDRectangle* currentRect;
    ISDVideoProperties* videoProperties;
    /*
     * This is a map that maps filepointers (pointing to the start of each rect header in the file)
     * to ISDRectangle* entries in the above rectList, so the seekback can occur in logarithmic time, by
     * accessing this map.
     */
    map<int, ISDRectangle*> rectMap;
};

#endif
