/*******************************************************************************************************************************************
 cwidget.c
*******************************************************************************************************************************************/

// gtkol widget handles the popup menu calls, so it has to know the popup menu api
#include "cmenu.h"

// gtkol widget handles part of layout analyse too
#include "clayout.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidgetListener::CWidgetListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidgetListener::~CWidgetListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnQueryDestroy
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnQueryDestroy (GtkWidget *, GdkEvent *, gpointer inData)
{
	// defaults to permission
	Bool doDestroy = true;

	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send request to the listener if any
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnQueryDestroy (theWidget, doDestroy);

	// ok
	return !doDestroy;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDestroy
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnDestroy (GtkWidget *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// be paranoïd
	if (theWidget != NULL)
	{
		// send notification
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnDestroy (theWidget);

		// find if there are children that are not handled by the gtk gui container aspect i.e if the OnDestroy is not indirectly
		// called, so we have to check for their presence in the gtkol hierarchy and to manually call the destroy request if so in
		// order to avoid segmentation faults when the gtk main event loop dispatcher exits while there are nore gtk widgets
		// instances that have not properly been deleted before !
		CComponents inChildren (theWidget -> GetChildren (__metaclass(CWidget)));
		for (size_t i=inChildren.GetLength(); i>0; i--) 
			if (!static_cast <CWidget *> (*inChildren[i-1]) -> m_PerformContainerProcess) 
				CWidget::DestroyWidget (static_cast <CWidget *> (*inChildren[i-1]));

		// ok, here gtk widget is destroyed by gtk himself, do not keep a pointer to a freed memory !
		theWidget -> m_GtkWidget = NULL;
	}

	// the first gtkol application widget should be a form - the main one - so stop the main events loop if the absolute widget is
	// destroyed and keep on the binary main entry point execution terminaison
	if (theWidget == *CComponent::GetComponents (__metaclass(CWidget))[0]) ::gtk_main_quit ();
}  

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnShow
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnShow (GtkWidget *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnShow (theWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnHide
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnHide (GtkWidget *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnHide (theWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnKeyPress
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnKeyPress (GtkWidget *, GdkEventKey *ioGdkEventKey, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check pointer
	if (theWidget != NULL)
	{
		// key value reference
		UInt16 &ioKey = reinterpret_cast <UInt16 &> (ioGdkEventKey -> keyval);
		
		// get shift states
		int inShiftStates = 0;
		if (ioGdkEventKey -> state & GDK_SHIFT_MASK  ) inShiftStates |= SHIFTSTATE_SHIFT;
		if (ioGdkEventKey -> state & GDK_LOCK_MASK   ) inShiftStates |= SHIFTSTATE_LOCK;
		if (ioGdkEventKey -> state & GDK_CONTROL_MASK) inShiftStates |= SHIFTSTATE_CONTROL;
		if (ioGdkEventKey -> state & GDK_MOD1_MASK   ) inShiftStates |= SHIFTSTATE_ALT;

		// send notification
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnKeyPress (theWidget, inShiftStates, ioKey);
	}

	// propagate the event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnKeyRelease
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnKeyRelease (GtkWidget *, GdkEventKey *inGdkEventKey, gpointer inData)
{	
	// retreive gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check pointer
	if (theWidget != NULL)
	{
		// get shift states
		int inShiftStates = 0;
		if (inGdkEventKey -> state & GDK_SHIFT_MASK  ) inShiftStates |= SHIFTSTATE_SHIFT;
		if (inGdkEventKey -> state & GDK_LOCK_MASK   ) inShiftStates |= SHIFTSTATE_LOCK;
		if (inGdkEventKey -> state & GDK_CONTROL_MASK) inShiftStates |= SHIFTSTATE_CONTROL;
		if (inGdkEventKey -> state & GDK_MOD1_MASK   ) inShiftStates |= SHIFTSTATE_ALT;

		// send notification
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> 
				OnKeyRelease (theWidget, inShiftStates, inGdkEventKey -> keyval);
	}

	// propagate the event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnSetFocus
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnSetFocus (GtkWidget *, GdkEventFocus *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);
	
	// send notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnSetFocus (theWidget);

	// propagate further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnKillFocus
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnKillFocus (GtkWidget *, GdkEventFocus *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);
	
	// send notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnKillFocus (theWidget);

	// propagate further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnMouseDown
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnMouseDown (GtkWidget *, GdkEventButton *inGdkEventButton, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check the pointer and propagate event further if not defined (that should never happen)
	if (theWidget == NULL) return false;

	// shift states check in
	int inShiftStates = 0;
	if (inGdkEventButton -> state & GDK_SHIFT_MASK	 ) inShiftStates |= SHIFTSTATE_SHIFT;
	if (inGdkEventButton -> state & GDK_LOCK_MASK	 ) inShiftStates |= SHIFTSTATE_LOCK;
	if (inGdkEventButton -> state & GDK_CONTROL_MASK ) inShiftStates |= SHIFTSTATE_CONTROL;
	if (inGdkEventButton -> state & GDK_MOD1_MASK	 ) inShiftStates |= SHIFTSTATE_ALT;
	if (inGdkEventButton -> button == 1		 ) inShiftStates |= SHIFTSTATE_LBUTTON;
	if (inGdkEventButton -> button == 2		 ) inShiftStates |= SHIFTSTATE_MBUTTON;
	if (inGdkEventButton -> button == 3		 ) inShiftStates |= SHIFTSTATE_RBUTTON;
	if (inGdkEventButton -> type == GDK_2BUTTON_PRESS) inShiftStates |= SHIFTSTATE_DBUTTON;	
	if (inGdkEventButton -> type == GDK_3BUTTON_PRESS) inShiftStates |= SHIFTSTATE_TBUTTON;

	// potential drag & drop beginning, so keep original relative point without threshold i.e. the real beginning d&d point
	if (inShiftStates & SHIFTSTATE_LBUTTON)
	{
		CWidget::m_DraggedPoint.x = static_cast <SInt32> (inGdkEventButton->x);
		CWidget::m_DraggedPoint.y = static_cast <SInt32> (inGdkEventButton->y);
	}

	// send the notification to the listener if any
	if (theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) ->  
			OnMouseDown (theWidget, inShiftStates, CWidget::m_DraggedPoint);

	// popup context analyse request ? (do not popup on double / triple click and do not popup on poup !)
	if ((inShiftStates&SHIFTSTATE_RBUTTON) && !(inShiftStates&SHIFTSTATE_DBUTTON) && !(inShiftStates&SHIFTSTATE_TBUTTON) &&
	    !theWidget -> ClassIs (__metaclass(CMenuPopup)))
	{
		// let's search for menu popup instance in the ascendant components hierarchy
		CMenuPopup *inMenuPopup = NULL; CComponent *Owner = theWidget;

		// go through the ascendant hierarchy
		while (inMenuPopup == NULL && Owner != NULL)
		{
			CComponents inChildren (Owner -> GetChildren (__metaclass(CMenuPopup)));
			if (inChildren.GetLength() > 0) inMenuPopup = static_cast <CMenuPopup *> (*inChildren[0]);
			Owner = Owner -> GetOwner ();
		}

		// if we found a popup menu in the ascendant hierarchy
		if (inMenuPopup != NULL)
		{
			// menu popup attributes affectation (for this purpose to succed, we are a friend of the CMenuPopup definition
			// that allows the gtkol widget to set those attributes)
			inMenuPopup -> m_RelativePoint.x = static_cast <SInt32> (inGdkEventButton->x);
			inMenuPopup -> m_RelativePoint.y = static_cast <SInt32> (inGdkEventButton->y);	
			inMenuPopup -> m_Sender		 = theWidget -> GetControlAtPoint (inMenuPopup -> m_RelativePoint);

			// defaults to do popup
			bool ioDoPopup = true;

			// send notification to listener if any
			if (inMenuPopup -> GetListener() != NULL)
				static_cast <CMenuPopupListener *> (inMenuPopup -> GetListener()) -> 
						OnQueryPopup (inMenuPopup, ioDoPopup);

			// do we have to execute the popup menu ?
			if (ioDoPopup)
			{
				// send notification to listener if any
				if (inMenuPopup -> GetListener() != NULL)
					static_cast <CMenuPopupListener *> (inMenuPopup -> GetListener()) -> OnPopup (inMenuPopup);

				// popup execution
				inMenuPopup -> Popup ();

				// do not propagate the event further
				return true;
			}
		}
	}

	// propagate the event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnMouseMove
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnMouseMove (GtkWidget *inGtkWidget, GdkEventMotion *inGdkEventButton, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// this call is very important : it requests the next motion event; the reason is that we specified GDK_POINTER_MOTION_HINT_MASK 
	// to gtk_widget_set_events() (see CWidget::CreateWidget); if we hadn't specified that, we could just use event->x and event->y
	// as the pointer location but we'd also get deluged in events; by requesting the next event as we handle the current one, we 
	// avoid getting a huge number of events faster than specific listeners can potentially cope...
	int x, y; GdkModifierType state; ::gdk_window_get_pointer (inGdkEventButton -> window, &x, &y, &state);

	// if not drag & dropping yet and if mouse left button has initiated a d&d potential beginning...
	if (CWidget::m_DraggedWidget == NULL && !(CWidget::m_DraggedPoint.x == 0x80000000 || CWidget::m_DraggedPoint.y == 0x80000000))
	{
		// shall we begin drag & drop ? (check threshold system value)
		if (::gtk_drag_check_threshold (inGtkWidget, x, y, CWidget::m_DraggedPoint.x, CWidget::m_DraggedPoint.y))
		{
			// get real control to drag (that may differ from the widget itself if the widget handles no windowed controls)
			CControl *theControl = theWidget -> GetControlAtPoint (CWidget::m_DraggedPoint);		

			// draggable control ?
			if (theControl != NULL && theControl -> GetDraggable())
			{
				// get a static copy of dragged widget and the potential real dragged control
				CWidget::m_DraggedWidget  = theWidget;
				CWidget::m_DraggedControl = theControl;

				// initiate gtk drag and drop
				GtkTargetEntry  TargetEntry[]    = { "", GTK_TARGET_SAME_APP, theWidget -> GetId() };
				GtkTargetList  *inGtkTargetList  = ::gtk_target_list_new (TargetEntry, 1);
				GdkDragContext *inGdkDragContext = ::gtk_drag_begin (inGtkWidget, inGtkTargetList, GDK_ACTION_ASK, 1, 
										     reinterpret_cast <GdkEvent *> (inGdkEventButton));
				// free unused values
				::gtk_target_list_unref (inGtkTargetList);

				// do not propagate event to owners
				return true;
			}
			// not a draggable control or NULL pointer
			else
			{
				// do not check again until next left mouse button pressed
				CWidget::m_DraggedPoint = TPoint (0x80000000, 0x80000000);
			}
		}
	}

	// check pointer
	if (theWidget != NULL)
	{
		// shift states check in
		int inShiftStates = 0;
		if (inGdkEventButton -> state & GDK_SHIFT_MASK	 ) inShiftStates |= SHIFTSTATE_SHIFT;
		if (inGdkEventButton -> state & GDK_LOCK_MASK	 ) inShiftStates |= SHIFTSTATE_LOCK;
		if (inGdkEventButton -> state & GDK_CONTROL_MASK ) inShiftStates |= SHIFTSTATE_CONTROL;
		if (inGdkEventButton -> state & GDK_MOD1_MASK	 ) inShiftStates |= SHIFTSTATE_ALT;

		// send notification
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> 
				OnMouseMove (theWidget, inShiftStates, TPoint (x, y));
	}

	// ok
	return false;
	// do not propagate the event to owners
//	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnMouseUp
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnMouseUp (GtkWidget *, GdkEventButton *inGdkEventButton, gpointer inData)
{
	// initializes drag values (avoid dnd beginning analyse on OnMouseMove execution)
	CWidget::m_DraggedPoint = TPoint (0x80000000, 0x80000000);

	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// if we are not dragging (inGdkEventButton is not available in this case) and if the pointer is correct !
	if (CWidget::m_DraggedWidget == NULL && theWidget != NULL)
	{
		// shift states check in
		int inShiftStates = 0;
		if (inGdkEventButton -> state & GDK_SHIFT_MASK	 ) inShiftStates |= SHIFTSTATE_SHIFT;
		if (inGdkEventButton -> state & GDK_LOCK_MASK	 ) inShiftStates |= SHIFTSTATE_LOCK;
		if (inGdkEventButton -> state & GDK_CONTROL_MASK ) inShiftStates |= SHIFTSTATE_CONTROL;
		if (inGdkEventButton -> state & GDK_MOD1_MASK	 ) inShiftStates |= SHIFTSTATE_ALT;
		if (inGdkEventButton -> button == 1		 ) inShiftStates |= SHIFTSTATE_LBUTTON;
		if (inGdkEventButton -> button == 2		 ) inShiftStates |= SHIFTSTATE_MBUTTON;
		if (inGdkEventButton -> button == 3		 ) inShiftStates |= SHIFTSTATE_RBUTTON;
		if (inGdkEventButton -> type == GDK_2BUTTON_PRESS) inShiftStates |= SHIFTSTATE_DBUTTON;	
		if (inGdkEventButton -> type == GDK_3BUTTON_PRESS) inShiftStates |= SHIFTSTATE_TBUTTON;

		// send notification
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> 
				OnMouseUp (theWidget, inShiftStates, TPoint ((SInt32)inGdkEventButton->x, (SInt32)inGdkEventButton->y));
	}

	// propagate the event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnMouseEnter
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnMouseEnter (GtkWidget *, GdkEventCrossing *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send notification if available to
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnMouseEnter (theWidget);

	// propagate event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnMouseExit
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnMouseExit (GtkWidget *, GdkEventCrossing *, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send notification if available to
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnMouseExit (theWidget);

	// propagate event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragDataGet
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnDragDataGet (GtkWidget *, GdkDragContext *, GtkSelectionData *inGtkSelectionData, guint, guint, gpointer)
{
	// fill in some specifications... gtk/x is waiting for inter process communication, so give it something (normalization TODO)
	::gtk_selection_data_set (inGtkSelectionData, GDK_SELECTION_TYPE_STRING, 8, (const guchar*)"", 0);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragDataReceived
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnDragDataReceived (GtkWidget *, GdkDragContext *inGdkDragContext, gint, gint, GtkSelectionData *, guint, guint inTime,
				  gpointer)
{
	// move action requested...
	if (inGdkDragContext -> action == GDK_ACTION_ASK) inGdkDragContext -> action = GDK_ACTION_MOVE;

	// ...and finish d&d
	::gtk_drag_finish (inGdkDragContext, true, true, inTime);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragStart
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnDragStart (GtkWidget *, GdkDragContext *inGdkDragContext, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// get the pixbuf representation of the dragged control
	CPixbuf *ioPixbuf = (CWidget::m_DraggedControl != NULL) ? CWidget::m_DraggedControl -> GetControlPixbuf() : NULL;

	// change the hotspot if required (that is if the dragged control is not a widget itself)
	if (CWidget::m_DraggedControl != NULL)
	{
		// if the dragged control is not a widget, it is, at least, a control !
		if (!CWidget::m_DraggedControl -> ClassIs (__metaclass(CWidget)))
		{
			// so get the control bounds in its owner widget
			TBounds inBounds (CWidget::m_DraggedControl -> GetBounds());

			// and change the hot spot (relative point) to friendly place the associated pixbuf if the control is able to
			// give the required bounds information...
			if (inBounds.x != 0L || inBounds.y != 0L || inBounds.w != 0L || inBounds.h != 0L)
			{
				CWidget::m_DraggedPoint.x -= inBounds.x;
				CWidget::m_DraggedPoint.y -= inBounds.y;
			}
			else
			{
				CWidget::m_DraggedPoint.x = 0L;
				CWidget::m_DraggedPoint.y = 0L;
			}
		}
	}

	// send notification to the listener if any
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> 
			OnDragStart (reinterpret_cast <CObject *&> (CWidget::m_DraggedControl), CWidget::m_DraggedPoint, ioPixbuf);

	// set the drag representation if any after listener section execution
	if (ioPixbuf != NULL)
	{
		// get the pixmap and the mask of the pixbuf representation
		GdkBitmap *inGdkMask; GdkPixmap *inGdkPixmap; 
		::gdk_pixbuf_render_pixmap_and_mask (ioPixbuf->GetPixbuf(), &inGdkPixmap, &inGdkMask, 0x01);

		// set the drag interface with the pixmap, the mask input and the hotspot
		::gtk_drag_set_icon_pixmap (inGdkDragContext, ::gdk_colormap_get_system(), inGdkPixmap, inGdkMask, 
					    CWidget::m_DraggedPoint.x, CWidget::m_DraggedPoint.y);

		// release the instances
		delete ioPixbuf; ::g_object_unref (inGdkPixmap); ::g_object_unref (inGdkMask);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragOver
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnDragOver (GtkWidget *, GdkDragContext *, gint inX, gint inY, guint, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// defaults to no drop acceptation unless checked
	bool outAccept = false;

	// check the pointer, stop if null
	if (theWidget == NULL) return false;

	// get the gui real target control i.e. the control under the relative pointer's coordinates, set the default gtkol target
	CControl *theControl = theWidget -> GetControlAtPoint (TPoint(inX,inY)); 
	CControl *theTarget  = theControl;

	// check the dnd source and target pointers
	if (CWidget::m_DraggedControl == NULL || theControl == NULL) return false;

	// if user pointed target control is not a drop site one
	if (!(outAccept = theControl -> GetDropSite()))
	{
		// get the gui target control's owner control
		theTarget = static_cast <CControl *> (theControl -> GetOwner (__metaclass(CControl)));

		// see if it is a layout control that could be a drop site i.e. that could manage the next potential drop request
		if (theTarget != NULL && theTarget -> ClassIs (__metaclass(CLayout)) && theControl != CWidget::m_DraggedControl) 
			outAccept = theTarget -> GetDropSite();
	
		// the gui target's owner is not a layout or not defined, reset the gtkol dnd target pointer to same as the gui one
		else theTarget = theControl;
	}

	// just check for gtkol expected hierarchy types concordance
	outAccept = outAccept && CWidget::m_DraggedControl -> CheckSetOwner (theTarget);

	// send notification to listener if any
	if (CWidget::m_DraggedWidget != NULL && CWidget::m_DraggedWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (CWidget::m_DraggedWidget -> GetListener()) ->
			OnDragOver (CWidget::m_DraggedControl, theControl, theTarget, TPoint(inX,inY), outAccept);

	// ok ?
	return outAccept;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragDrop
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnDragDrop (GtkWidget *, GdkDragContext *, gint inX, gint inY, guint, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// defaults to automatic drop process handling if enabled to do so...
	Bool doDrop = (theWidget != NULL);

	// check
	if (!doDrop) return false;

	// get the gui real target control i.e. the control under the relative pointer's coordinates, set the default gtkol target
	CControl *theControl = theWidget -> GetControlAtPoint (TPoint(inX,inY));
	CControl *theTarget  = theControl;

	// defaults to an append request on the specified target
	SInt16 theIndex = -1;

	// check the dnd source and target pointers
	if (CWidget::m_DraggedControl == NULL || theControl == NULL) return false;

	// if target control is not a drop site one
	if (!(doDrop = theControl -> GetDropSite()))
	{
		// get the gui target control's owner control
		theTarget = static_cast <CControl *> (theControl -> GetOwner (__metaclass(CControl)));

		// see if it is a layout and get a special handling if so...
		if (theTarget != NULL && theTarget -> ClassIs (__metaclass(CLayout)) && theControl != CWidget::m_DraggedControl)
		{
			// see if the target may be a drop site for the dragged control
			if (doDrop = (theTarget -> GetDropSite() && CWidget::m_DraggedControl -> CheckSetOwner (theTarget)))

				// get the supposed index insertion of the dragged control into the layout from the given relative point
				theIndex = static_cast <CLayout *> (theTarget) -> 
						GetDropIndexAtPoint (TPoint(inX,inY), CWidget::m_DraggedControl);
		}
		// the gui target's owner is not a layout or is not defined, reset the gtkol dnd target pointer to same as the gui one
		else theTarget = theControl;
	}
	// the target is a drop site, just check for expected gkol hierarchy concordance
	else doDrop = CWidget::m_DraggedControl -> CheckSetOwner (theTarget);

	// send notification, get potential target modification and automatic drop process negociation
	if (CWidget::m_DraggedWidget != NULL && CWidget::m_DraggedWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (CWidget::m_DraggedWidget -> GetListener()) ->
			OnDragDrop (CWidget::m_DraggedControl, theControl, (CObject*&)theTarget, theIndex, TPoint(inX,inY), doDrop);

	// so, shall we handle the drop process (and is it possible to do so) ?
	if (doDrop && theTarget != NULL && theTarget -> GetDropSite() && theControl != CWidget::m_DraggedControl)
	{
		// owner affectation, automatic generic check of expected hierarchical types before
		doDrop = CWidget::m_DraggedControl -> SetOwner (theTarget, theIndex);

		// ok ?
		return doDrop;
	}

	// stop
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDragStop
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnDragStop (GtkWidget *, GdkDragContext *, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check pointers
	if (theWidget != NULL && CWidget::m_DraggedControl != NULL)

		// send notification if allowed to
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnDragStop (CWidget::m_DraggedControl);

	// reset static d&d variables
	CWidget::m_DraggedWidget  = NULL;
	CWidget::m_DraggedControl = NULL;
	CWidget::m_DraggedPoint	  = TPoint (0x80000000, 0x80000000);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnConfigure
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnConfigure (GtkWidget *inGtkWidget, GdkEventConfigure *inGdkEvent, gpointer)
{
	// GtkAllocation declaration
	GtkAllocation theGtkAllocation;

	// allocation fill in (client bounds)
	theGtkAllocation.x	= inGdkEvent -> x;
	theGtkAllocation.y	= inGdkEvent -> y;
	theGtkAllocation.width	= inGdkEvent -> width;
	theGtkAllocation.height = inGdkEvent -> height;

	// allocation validation, indirect call of CWidget::OnResize
	::gtk_widget_size_allocate (inGtkWidget, &theGtkAllocation);

	// propagate the event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnQueryResize
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnQueryResize (GtkWidget *, GtkRequisition *ioGtkRequisition, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// TSize declaration
	TSize ioSize (ioGtkRequisition -> width, ioGtkRequisition -> height);

	// check pointer and send notification to the associated listener if any
	if (theWidget != NULL)
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnQueryResize (theWidget, ioSize);

	// attributes affectation
	ioGtkRequisition -> width  = ioSize.w;
	ioGtkRequisition -> height = ioSize.h;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnResize 
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnResize (GtkWidget *, GtkAllocation *inGtkAllocation, gpointer inData)
{
	// get the gtkol instane
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check the pointer
	if (theWidget != NULL)
	{
		// default attributes affectation
		theWidget -> m_Bounds = TBounds (inGtkAllocation -> x, inGtkAllocation -> y, inGtkAllocation -> width, 
						 inGtkAllocation -> height);

		// get the widget's owner
		CControl *theControl = static_cast <CControl *> (theWidget -> GetOwner (__metaclass(CControl))); 

		// if the widget is owned in a layout, set its relative coordinates i.e. remap the considered owner widget window
		if (theControl != NULL && theControl -> ClassIs (__metaclass(CLayout)))
		{
			theWidget -> m_Bounds.x -= static_cast <CWidget *> (theControl) -> m_Bounds.x;
			theWidget -> m_Bounds.y -= static_cast <CWidget *> (theControl) -> m_Bounds.y;
		}

		// send notification if any listener is available
		if (theWidget -> GetListener() != NULL)
			static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnResize (theWidget);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnExpose
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CWidget::OnPaint (GtkWidget *inGtkWidget, GdkEventExpose *inGdkEventExpose, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// read in invalidated bounds
	TBounds inInvalidated (inGdkEventExpose->area.x, inGdkEventExpose->area.y, 
			       inGdkEventExpose->area.width, inGdkEventExpose->area.height);

	// declare a graphics tool
	CGraphics theGraphics (inGtkWidget, inInvalidated);

	// send the notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnPaint (theWidget, theGraphics);

	// propagate event further only if no painted occurred in the listener section...
	return theGraphics.GetDrawn ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnEventAfter
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::OnEventAfter (GtkWidget *inGtkWidget, GdkEvent *inGdkEvent, gpointer inData)
{
	// get the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CWidgetListener *> (theWidget -> GetListener()) -> OnEventAfter (theWidget, inGdkEvent);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CWidget);

//-----------------------------------------------------------------------------------------------------------------------------------------
// static variable resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidget  * CWidget::m_DraggedWidget  = NULL;
CControl * CWidget::m_DraggedControl = NULL;
TPoint	   CWidget::m_DraggedPoint    (0x80000000, 0x80000000);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidget::CWidget   		   (CComponent *inOwner, const CWidgetListener *inListener)
	:CControl  		   (inOwner, inListener),
	 m_GtkWidget 		   (NULL),
	 m_PerformContainerProcess (true),
	 m_PerformSignalConnection (true),
	 m_SerializeShown	   (true),
	 m_SerializeEnabled	   (true)
{
	// we can't execute CWidget::CreateWidget here because at this point, the derived constructor has not been executed yet and the 
	// available memory stack does not include specific virtual definitions (the vtable is under construction) so there is no 
	// particular resolution done here; it is the specific gtkol widget definition responsability to call the CWidget::CreateWidget
	// process handler
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidget::~CWidget ()
{
	// this function should not be handled at this point, if so, it is DANGEROUS; derived classes should call CWidget::DestroyWidget
	// in their specific destructors to keep entire memory stacks available while deletion !
	CWidget::DestroyWidget (this); 
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// private widget instanciation service
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::CreateWidget (CWidget *inThis)
{
	// check this pointer
	if (inThis == NULL) return false;

	// check associated gtk widget pointer
	if (inThis -> m_GtkWidget != NULL) return false;

	// specific instanciation process requested
	if ((inThis -> m_GtkWidget = inThis -> PerformWidgetInstanciate ()) == NULL) return false;

	// if the definition has to associate the default signals
	if (inThis -> m_PerformSignalConnection) 
	{
		// all events assignation if requested 
		::gtk_widget_set_events (inThis -> m_GtkWidget, GDK_ALL_EVENTS_MASK);

		// generic events connection
		::g_signal_connect (inThis -> m_GtkWidget, "delete-event",		G_CALLBACK(CWidget::OnQueryDestroy),	 inThis);
		::g_signal_connect (inThis -> m_GtkWidget, "destroy",			G_CALLBACK(CWidget::OnDestroy),		 inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "show",                 	G_CALLBACK(CWidget::OnShow),             inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "hide",                 	G_CALLBACK(CWidget::OnHide),             inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "event-after",               G_CALLBACK(CWidget::OnEventAfter),       inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "focus-in-event",       	G_CALLBACK(CWidget::OnSetFocus),         inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "focus-out-event",      	G_CALLBACK(CWidget::OnKillFocus),        inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "key-press-event",      	G_CALLBACK(CWidget::OnKeyPress),         inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "key-release-event",    	G_CALLBACK(CWidget::OnKeyRelease),       inThis);

        	::g_signal_connect (inThis -> m_GtkWidget, "button-press-event",   	G_CALLBACK(CWidget::OnMouseDown),        inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "motion-notify-event",  	G_CALLBACK(CWidget::OnMouseMove),        inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "button-release-event", 	G_CALLBACK(CWidget::OnMouseUp),          inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "drag-begin",           	G_CALLBACK(CWidget::OnDragStart),        inThis);
       		::g_signal_connect (inThis -> m_GtkWidget, "drag-motion",          	G_CALLBACK(CWidget::OnDragOver),         inThis);
       		::g_signal_connect (inThis -> m_GtkWidget, "drag-drop",            	G_CALLBACK(CWidget::OnDragDrop),         inThis);
       		::g_signal_connect (inThis -> m_GtkWidget, "drag-end",             	G_CALLBACK(CWidget::OnDragStop),         inThis);
       		::g_signal_connect (inThis -> m_GtkWidget, "drag-data-get",        	G_CALLBACK(CWidget::OnDragDataGet),      inThis);
       		::g_signal_connect (inThis -> m_GtkWidget, "drag-data-received",   	G_CALLBACK(CWidget::OnDragDataReceived), inThis);

        	::g_signal_connect (inThis -> m_GtkWidget, "enter-notify-event",   	G_CALLBACK(CWidget::OnMouseEnter),       inThis);
        	::g_signal_connect (inThis -> m_GtkWidget, "leave-notify-event",   	G_CALLBACK(CWidget::OnMouseExit),        inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "configure-event",		G_CALLBACK(CWidget::OnConfigure),	 inThis);
		::g_signal_connect (inThis -> m_GtkWidget, "size-request",		G_CALLBACK(CWidget::OnQueryResize),	 inThis);
		::g_signal_connect (inThis -> m_GtkWidget, "size-allocate",		G_CALLBACK(CWidget::OnResize),		 inThis);

		::g_signal_connect (inThis -> m_GtkWidget, "expose-event",		G_CALLBACK(CWidget::OnPaint),          	 inThis);
	}

	// hide widget on deletion
	::gtk_widget_hide_on_delete (inThis -> m_GtkWidget);

	// perform container process if requested and if enabled to do so
	if (inThis->m_PerformContainerProcess && inThis->GetOwner() != NULL && inThis->GetOwner()->ClassIs (__metaclass(CContainer)))
		static_cast <CContainer *> (inThis -> GetOwner()) -> PerformContainerAdd (inThis);

	// perform specific widget initialization
	inThis -> PerformWidgetInitialize ();

	// is this instance a container one ?
	if (inThis -> ClassIs (__metaclass(CContainer)))
	{
		// at this point, the gtk children widgets may not be instanciated if the gtkol instance has not been attached before 
		// allocating sub components, so get the direct children and perform a recursive call on them
		CComponents inChildren (inThis -> GetChildren (__metaclass(CWidget)));

		// go through the children and instanciate them if not done yet
		for (size_t i=inChildren.GetLength(), j=0; i>0; i--, j++) 
		{
			// get the child we are interested in
			CWidget *inChild = static_cast <CWidget *> (*inChildren[j]);

			// perform the gtk container aspect
			if (inChild -> m_PerformContainerProcess && ::gtk_widget_get_parent (inChild -> GetGtkWidget()) == NULL)

				// launch the request
				static_cast <CContainer *> (inThis) -> PerformContainerAdd (inChild);
		}
	}

	// ok
	return true;

}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget deletion service
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::DestroyWidget (CWidget *inThis)
{
	// be paranoïd...
	if (inThis == NULL) return true;

	// gtk widget may already have been deleted
	if (inThis -> m_GtkWidget == NULL) return true;

	// defaults to destroy
	Bool doDestroy = true;

	// listener interrogation if any
	if (inThis -> GetListener() != NULL)
		static_cast <CWidgetListener *> (inThis -> GetListener()) -> OnQueryDestroy (inThis, doDestroy);

	// so, shall we destroy ?
	if (!doDestroy) return false;

	// gtkol children instances
	CComponents inChildren (inThis -> GetChildren (__metaclass(CWidget)));

	// reverse deletion
	for (size_t i=inChildren.GetLength(); i>0; i--)
	{
		// check result !
		if (!CWidget::DestroyWidget (static_cast <CWidget *> (*inChildren[i-1])))
		{
			// error or not enabled to do so
			return false;
		}
	}

	// gtk destruction, NULL pointer affected by CWidget::OnDestroy event reception
	::gtk_widget_destroy (inThis -> m_GtkWidget);

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtkol widget associated to specified gtk widget
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidget * CWidget::GetGtkolWidget (const GtkWidget *inGtkWidget)
{
	// get all of the widget instance components
	CComponents inComponents (CComponent::GetComponents (__metaclass(CWidget)));
	
	// go through the widgets list
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
	{
		// current gtkol widget
		CWidget *inWidget = static_cast <CWidget *> (*inComponents[j]);

		// pointer check
		if (inWidget -> m_GtkWidget == const_cast <GtkWidget *> (inGtkWidget))
		{
			return inWidget;
		}
	}

	// not found
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtkol widget associated to specified gdk window
//-----------------------------------------------------------------------------------------------------------------------------------------
CWidget * CWidget::GetGtkolWidget (const GdkWindow *inGdkWindow)
{
	// get all of the widget instance components
	CComponents inComponents (CComponent::GetComponents (__metaclass(CWidget)));
	
	// go through the widgets list
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
	{
		// current gtkol widget
		CWidget *inWidget = static_cast <CWidget *> (*inComponents[j]);

		// pointer check only if the current widget has an associated gdk window defined
		if (!(GTK_WIDGET_FLAGS(inWidget->m_GtkWidget)&GTK_NO_WINDOW) && inWidget->m_GtkWidget->window == inGdkWindow)
		{
			return inWidget;
		}
	}

	// not found
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget of the gtkol widget
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CWidget::GetGtkWidget () const
{
	return m_GtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CWidget::ListenerMustBe () const
{
	return __metaclass(CWidgetListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// check set owner
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::CheckSetOwner (const CComponent *inCandidate) const
{
	// pointer check, owner removal authorized
	if (inCandidate == NULL) return true;

	// generic hierarchy concordance check
	if (!CControl::CheckSetOwner (inCandidate)) return false;

	// is the owner candidate a container instance i.e. validate the children handling number left if it would not be a same owner 
	// move in reaffectation
	if (m_PerformContainerProcess && const_cast <CComponent *> (inCandidate) != GetOwner() && 
	    inCandidate -> ClassIs (__metaclass(CContainer)))
	{
		// get the container children handling number left
		SInt16 inChildrenLeft = static_cast <const CContainer *> (inCandidate) -> GetGtkChildrenNumberLeft();

		// check the restriction if set
		return inChildrenLeft != 0;
	}
	
	// defaults to acceptation
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// owner affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// old potential direct owner
	CComponent *oldOwner = GetOwner ();

	// generic affectation check, gtkol application hierarchy representation
	if (!CheckSetOwner (inOwner)) return false;

	// if the gtk widget exists
	if (m_GtkWidget != NULL)
	{
		// do we have to handle the gtk gui container aspect for this widget ?
		if (m_PerformContainerProcess)
		{
			// remove the widget from its old container if any
			if (oldOwner != NULL && oldOwner -> ClassIs (__metaclass(CContainer)))
			{
				// keep a reference to avoid automatic gtk deletion of the widget while moving it 
				// from a container to another one !
				::g_object_ref (G_OBJECT(m_GtkWidget));
			
				// perform the specific removal
				static_cast <CContainer *> (oldOwner) -> PerformContainerRemove (this);
			}

			// generic spine affectation, should always be true as we checked above !
			if (!CControl::SetOwner (inOwner, inIndex)) return false;

			// add the widget to its new container if any specified
			if (inOwner != NULL && inOwner -> ClassIs (__metaclass(CContainer)))
			{
				// perform the specific append request
				static_cast <CContainer *> (inOwner) -> PerformContainerAdd (this);

				// delete the reference the removal reference...
				::g_object_unref (G_OBJECT(m_GtkWidget));
			}
		}
	}
	// the gtk widget instance does not exist
	else
	{
		// generic spine affectation, should always be true as we checked above !
		if (!CControl::SetOwner (inOwner, inIndex)) return false;

		// so launch the gtk widget instanciation process management
		return CWidget::CreateWidget (this);
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default expected owner types
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CWidget::OwnerMustBe () const
{
	return __metaclasses(CWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default expected children types
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CWidget::ChildMustBe () const
{
	return __metaclasses(CWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// bounds affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::SetBounds (const TBounds &inBounds)
{
	if (m_GtkWidget != NULL) ::gtk_widget_set_size_request (m_GtkWidget, inBounds.w, inBounds.h);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// draggable affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::SetDraggable (const Bool inDraggable)
{
	// generic attribute affectation only, nothing more to handle here
	CControl::SetDraggable (inDraggable);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// drop site affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::SetDropSite (const Bool inDropSite)
{
	// check pointer
	if (m_GtkWidget != NULL)
        {
                if (inDropSite)
                {
                        GtkTargetEntry TargetEntry[] = { "", GTK_TARGET_SAME_APP, GetId() };
                        ::gtk_drag_dest_set (m_GtkWidget, GTK_DEST_DEFAULT_ALL, TargetEntry, 1, GDK_ACTION_ASK);
                }
                else
                {
                        ::gtk_drag_dest_unset (m_GtkWidget);
                }
        }

	// generic attribute affectation
	CControl::SetDropSite (inDropSite);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get the widget pixbuf respresentation
//-----------------------------------------------------------------------------------------------------------------------------------------
CPixbuf * CWidget::GetControlPixbuf () const
{
	// be paranoïd...
	if (m_GtkWidget == NULL || m_GtkWidget -> window == NULL) return NULL;

	// get the owner of this instance
	CControl *inOwner (static_cast <CControl *> (GetOwner (__metaclass(CControl))));

	// go through and remap the window position about this instance
	SInt32 xx=m_Bounds.x, yy=m_Bounds.y; while (inOwner != NULL && inOwner -> ClassIs (__metaclass(CLayout))) 
	{
		// map the position
		xx += static_cast <CWidget *> (inOwner) -> m_Bounds.x;
		yy += static_cast <CWidget *> (inOwner) -> m_Bounds.y;

		// next owner...
		inOwner = static_cast <CControl *> (inOwner -> GetOwner (__metaclass(CControl)));
	}

	// ok, allocate the pixbuf to give a control reprensation (alpha mode requested)
	CPixbuf *Pixbuf = new CPixbuf (m_GtkWidget -> window, PIXELFORMAT_32, TBounds (xx, yy, m_Bounds.w, m_Bounds.h));

	// be paranoïd...
	if (Pixbuf -> GetPixelFormat() != PIXELFORMAT_32)
	{
		// delete it !
		delete Pixbuf;

		// stop
		return NULL;
	}

	// keep dark pixels (that is, darker than gray), forget the others !!! NOT A VERY OPTIMIZED PART !!!
	UInt8 *Line = Pixbuf -> GetBaseAddress();
	for (size_t i=m_Bounds.h; i>0; i--)
	{
		UInt32 *Pixel = reinterpret_cast <UInt32 *> (Line);
		for (size_t j=m_Bounds.w; j>0; j--, Pixel++)
			if ((*Pixel&0x00FFFFFF) <= 0x007F7F7F) *Pixel = 0xFF7F7F7F; else *Pixel = 0L;
		Line += Pixbuf -> GetRowBytes();
	}

	// set the default border representation of the dragged control (opaque gray borders), direct draw on
	// the pixbuf's buffer (see static declaration of CPixbuf::DrawLine and CPixbuf::DrawRectangle function for details in cgraphics.h)
	CPixbuf::DrawRectangle (TBounds (0L, 0L, m_Bounds.w-1L, m_Bounds.h-1L), Pixbuf->GetBaseAddress(), PIXELFORMAT_32, 
				Pixbuf->GetRowBytes(), 0xFF7F7F7F, 1.0f);

	// ok
	return Pixbuf;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default control at specified position
//-----------------------------------------------------------------------------------------------------------------------------------------
CControl * CWidget::GetControlAtPoint (const TPoint &) const
{
	// defaults to this instance as we don't know at this level what could be the non windowed managed controls !
	return const_cast <CWidget *> (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// show 
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Show (const Bool inAll)
{
	// be sure...
	if (m_GtkWidget == NULL) return;

	// show all ?
	if (inAll) ::gtk_widget_show_all (m_GtkWidget); else ::gtk_widget_show (m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// hide
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Hide (const Bool inAll)
{
	// be sure...
	if (m_GtkWidget == NULL) return;

	// hide all ?
	if (inAll) ::gtk_widget_hide_all (m_GtkWidget); else ::gtk_widget_hide (m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is shown ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::IsShown () const
{
	return m_GtkWidget == NULL ? false : GTK_WIDGET_VISIBLE(m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// enable
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Enable ()
{
	if (m_GtkWidget != NULL) ::gtk_widget_set_sensitive (m_GtkWidget, true);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// disable
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Disable ()
{
	if (m_GtkWidget != NULL) ::gtk_widget_set_sensitive (m_GtkWidget, false);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is enabled ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::IsEnabled () const
{
	return m_GtkWidget == NULL ? false : GTK_WIDGET_IS_SENSITIVE(m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// can focus ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::CanFocus () const
{
	return m_GtkWidget != NULL ? GTK_WIDGET_CAN_FOCUS(m_GtkWidget) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set focus
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::SetFocus ()
{
	if (m_GtkWidget != NULL) ::gtk_widget_grab_focus (m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is focused ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CWidget::IsFocused () const
{
	return m_GtkWidget == NULL ? false : GTK_WIDGET_HAS_FOCUS(m_GtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// bounds invalidation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Invalidate (const TBounds &inBounds)
{
	// check pointer...
        if (m_GtkWidget == NULL || m_GtkWidget -> window == NULL) return;

	// invalidated rectangle
	GdkRectangle Rect = { inBounds.x, inBounds.y, inBounds.w, inBounds.h };
         
	// total invalidation ?
        if (inBounds.w == 0L || inBounds.h == 0L)
	{
		Rect.x	    = 0L;
		Rect.y	    = 0L;
		Rect.width  = m_Bounds.w;
		Rect.height = m_Bounds.h;
	}
	
	// invalidate
	::gdk_window_invalidate_rect (m_GtkWidget -> window, &Rect, false);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// invalidation -> validation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Validate ()
{
	// check pointer
        if (m_GtkWidget == NULL || m_GtkWidget -> window == NULL) return;
                                                                                                                                                
        // validate now
        ::gdk_window_process_updates (m_GtkWidget -> window, true);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CWidget::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization process call first
	CControl::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element under the current one
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_WIDGET_ELEMENT);

			// set the attributes the widget definition is interested in (shown state and enabled state)
			if (m_SerializeShown)
				newXMLElement -> AddAttribute (XML_WIDGET_ATTR_ISSHOWN,   CString(IsShown()   ? "true" : "false"));
			if (m_SerializeEnabled)
				newXMLElement -> AddAttribute (XML_WIDGET_ATTR_ISENABLED, CString(IsEnabled() ? "true" : "false"));

			// modify the input output xml node pointer so that overwritten serialization continues under the current xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_WIDGET_ELEMENT);

			// check we got an expected cwidget node
			if (inXMLNode == NULL)
				throw new CException (CString("CWidget::Serialize, specified xml node is not a \"") + 
							      XML_WIDGET_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// retrieve the attributes this instance is interested in
			Bool inIsShown   = ::xml_node_get_attribute (inXMLNode, XML_WIDGET_ATTR_ISSHOWN).  GetValue().ToBool();
			Bool inIsEnabled = ::xml_node_get_attribute (inXMLNode, XML_WIDGET_ATTR_ISENABLED).GetValue().ToBool();
	
			// modify the in/out element node pointer so that the serialization process will continue under the cwidget node
			ioXMLElementNode = inXMLNode;

			// modify the widget accordinaly
			if (m_SerializeShown)
			{
				if ( inIsShown   && !IsShown  ()) Show    ();
				if (!inIsShown   &&  IsShown  ()) Hide    ();
			}
			if (m_SerializeEnabled)
			{
				if ( inIsEnabled && !IsEnabled()) Enable  ();
				if (!inIsEnabled &&  IsEnabled()) Disable ();
			}
		}
		break;
	}
}







