/*******************************************************************************************************************************************
 cscale.c
*******************************************************************************************************************************************/

#include "cscale.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CScaleListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScaleListener::CScaleListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScaleListener::~CScaleListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CScale);

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnValueQueryFormat
//-----------------------------------------------------------------------------------------------------------------------------------------
gchar * CScale::OnValueQueryFormat (GtkScale *inGtkScale, gdouble inValue, gpointer inData)
{
	// retreive the gtkol instance
	CScale *inScale = reinterpret_cast <CScale *> (inData);

	// format the value text default lookup
	gchar *inFormat = ::g_strdup_printf ("%0.*f", ::gtk_scale_get_digits(inGtkScale), inValue);

	// get it as a string object
	CString outString (inFormat);

	// free the local pointer
	::g_free (inFormat);

	// send the notification if any listener attached
	if (inScale != NULL && inScale -> GetListener() != NULL)
		static_cast <CScaleListener *> (inScale -> GetListener()) -> OnValueQueryFormat (inScale, inValue, outString);

	// ok
	return ::g_strdup (outString.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScale::CScale (CContainer *inOwner, const CScaleListener *inListener)
       :CRange (inOwner, inListener)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScale::~CScale ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected listener type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClass * CScale::ListenerMustBe () const
{
	// ok, expecting a CScaleListener listener instance
	return const_cast <CMetaClass *> (__metaclass(CScaleListener));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// the precision to be displayed by the scale
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScale::SetDigits (const UInt8 inDigits)
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set the property
	if (inGtkWidget != NULL) ::gtk_scale_set_digits (GTK_SCALE(inGtkWidget), inDigits);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// the precision to be displayed by the scale
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt8 CScale::GetDigits () const
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	return (inGtkWidget != NULL) ? ::gtk_scale_get_digits (GTK_SCALE(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// whether the current value is displayed as a string next to the slider
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScale::SetDrawValue (const bool inDrawValue)
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set the property
	if (inGtkWidget != NULL) ::gtk_scale_set_draw_value (GTK_SCALE(inGtkWidget), inDrawValue);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// whether the current value is displayed as a string next to the slider
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CScale::GetDrawValue () const
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	return (inGtkWidget != NULL) ? ::gtk_scale_get_draw_value (GTK_SCALE(inGtkWidget)) : true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// the position in which the current value is displayed
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScale::SetValuePosition (const GtkPositionType inGtkPositionType)
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set the property
	if (inGtkWidget != NULL) ::gtk_scale_set_value_pos (GTK_SCALE(inGtkWidget), inGtkPositionType);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// the position in which the current value is displayed
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkPositionType CScale::GetValuePosition () const
{
	// retreive the gtk instance
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	return (inGtkWidget != NULL) ? ::gtk_scale_get_value_pos (GTK_SCALE(inGtkWidget)) : GTK_POS_LEFT;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScale::PerformWidgetInitialize ()
{
	// generic call
	CRange::PerformWidgetInitialize ();

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;
	
	// specific signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "format-value", G_CALLBACK(CScale::OnValueQueryFormat), this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// scale serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScale::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CRange::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// create a new xml element node for the serialization
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_SCALE_ELEMENT);
			
			// add the digits attribute
			newXMLElement -> AddAttribute (XML_SCALE_ATTR_DIGITS, CString((UInt32)GetDigits()));

			// add the draw value attribute
			newXMLElement -> AddAttribute (XML_SCALE_ATTR_DRAW_VALUE, CString(GetDrawValue()?"true":"false"));

			// add the position attribute
			CString outPosition; switch (GetValuePosition())
			{
				case GTK_POS_LEFT   : outPosition = CString("left");   break;
				case GTK_POS_RIGHT  : outPosition = CString("right");  break;
				case GTK_POS_TOP    : outPosition = CString("top");    break;
				case GTK_POS_BOTTOM : outPosition = CString("bottom"); break;
			}
			newXMLElement -> AddAttribute (XML_SCALE_ATTR_VALUE_POSITION, outPosition);

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_SCALE_ELEMENT);

			// check we got an expected cbutton node
			if (inXMLNode == NULL)
				throw new CException (CString("CScale::Serialize, specified xml node is not a \"") + 
							      XML_SCALE_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the current node
			ioXMLElementNode = inXMLNode;

			// retreive the position
			CString inPosition (::xml_node_get_attribute (inXMLNode, XML_SCALE_ATTR_VALUE_POSITION).GetValue());
			if (inPosition == CString("left")) 
				SetValuePosition (GTK_POS_LEFT);
			else if (inPosition == CString("right"))
				SetValuePosition (GTK_POS_RIGHT);
			else if (inPosition == CString("top"))
				SetValuePosition (GTK_POS_TOP);
			else
				SetValuePosition (GTK_POS_BOTTOM);

			// set the other properties
			SetDigits    (::xml_node_get_attribute (inXMLNode, XML_SCALE_ATTR_DIGITS).    GetValue().ToULong());
			SetDrawValue (::xml_node_get_attribute (inXMLNode, XML_SCALE_ATTR_DRAW_VALUE).GetValue().ToBool ());
		}
		break;
	}
}
