/*******************************************************************************************************************************************
 crange.c
*******************************************************************************************************************************************/

#include "crange.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CRangeListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CRangeListener::CRangeListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CRangeListener::~CRangeListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CRange);

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnValueQueryChange
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CRange::OnValueQueryChange (GtkRange *, GtkScrollType *ioGtkScrollType, gdouble inValue, gpointer inData)
{
	// retreive the gtkol instance
	CRange *inRange = reinterpret_cast <CRange *> (inData);

	// the given value, defaults to process the modification
	double ioValue = inValue;

	// listener call
	if (inRange != NULL && inRange -> GetListener() != NULL)
		static_cast <CRangeListener *> (inRange -> GetListener()) -> OnValueQueryChange (inRange, *ioGtkScrollType, ioValue);

	// process the modification if any
	if (ioValue != inValue && inRange != NULL) inRange -> SetValue (ioValue);
	
	// ok
	return ioValue != inValue;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnValueChanged
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::OnValueChanged (GtkRange *, gpointer inData)
{
	// retreive the gtkol instance
	CRange *inRange = reinterpret_cast <CRange *> (inData);

	// send the notification if any listener is attached
	if (inRange != NULL && inRange -> GetListener() != NULL)
		static_cast <CRangeListener *> (inRange -> GetListener()) -> OnValueChanged (inRange);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CRange::CRange  (CContainer *inOwner, const CRangeListener *inListener)
       :CWidget (inOwner, inListener),
	m_Min	(0),
	m_Max	(10),
	m_Step	(1),
	m_Page	(1)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CRange::~CRange ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected listener type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClass * CRange::ListenerMustBe () const
{
	// ok, expecting a CRangeListener listener instance
	return const_cast <CMetaClass *> (__metaclass(CRangeListener));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::PerformWidgetInitialize ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// specific signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "change-value",  G_CALLBACK(CRange::OnValueQueryChange), this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "value-changed", G_CALLBACK(CRange::OnValueChanged),     this);

	// show the job
	Show();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CRange::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// value affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::SetValue (const double inValue)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget != NULL) ::gtk_range_set_value (GTK_RANGE(inGtkWidget), inValue);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// value access
//-----------------------------------------------------------------------------------------------------------------------------------------
double CRange::GetValue () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	return (inGtkWidget != NULL) ? ::gtk_range_get_value (GTK_RANGE(inGtkWidget)) : 0;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// update policy affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::SetUpdatePolicy (const GtkUpdateType inGtkUpdateType)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget != NULL) ::gtk_range_set_update_policy (GTK_RANGE(inGtkWidget), inGtkUpdateType);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// update policy access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkUpdateType CRange::GetUpdatePolicy () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	return (inGtkWidget != NULL) ? ::gtk_range_get_update_policy (GTK_RANGE(inGtkWidget)) : GTK_UPDATE_CONTINUOUS;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// range affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::SetRange (const double inMin, const double inMax)
{
	// local copy
	m_Min = inMin; m_Max = inMax;

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget != NULL) ::gtk_range_set_range (GTK_RANGE(inGtkWidget), inMin, inMax);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// min range access
//-----------------------------------------------------------------------------------------------------------------------------------------
double CRange::GetMin () const
{
	// ok, return the value
	return m_Min;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// max range access
//-----------------------------------------------------------------------------------------------------------------------------------------
double CRange::GetMax () const
{
	// ok, return the value
	return m_Max;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// increments affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::SetIncrements (const double inStep, const double inPage)
{
	// local copy
	m_Step = inStep; m_Page = inPage;

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget != NULL) ::gtk_range_set_increments (GTK_RANGE(inGtkWidget), inStep, inPage);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// step access
//-----------------------------------------------------------------------------------------------------------------------------------------
double CRange::GetStep () const
{
	// ok, return the value
	return m_Step;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// page access
//-----------------------------------------------------------------------------------------------------------------------------------------
double CRange::GetPage () const
{
	// ok, return the value
	return m_Page;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// inverted state affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::SetInverted (const bool inInverted)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget != NULL) ::gtk_range_set_inverted (GTK_RANGE(inGtkWidget), inInverted);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// inverted state access
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CRange::GetInverted () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	return (inGtkWidget != NULL) ? ::gtk_range_get_inverted (GTK_RANGE(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialize definition
//-----------------------------------------------------------------------------------------------------------------------------------------
void CRange::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CWidget::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// create a new xml element node for the serialization
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_RANGE_ELEMENT);
			
			// add the attributes
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_VALUE, CString(GetValue()));
			
			// add the update policy attribute
			CString outPolicy; switch (GetUpdatePolicy())
			{
				case GTK_UPDATE_CONTINUOUS    : outPolicy = "continous";     break;
				case GTK_UPDATE_DISCONTINUOUS : outPolicy = "discontinuous"; break;
				case GTK_UPDATE_DELAYED       : outPolicy = "delayed"; 	     break;
			}

			newXMLElement -> AddAttribute (XML_RANGE_ATTR_UPDATE_POLICY, 	outPolicy);
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_INVERTED, 	CString(GetInverted()?"true":"false"));
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_MIN, 		CString(GetMin()));
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_MAX, 		CString(GetMax()));
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_STEP, 		CString(GetStep()));
			newXMLElement -> AddAttribute (XML_RANGE_ATTR_PAGE, 		CString(GetPage()));

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_RANGE_ELEMENT);

			// check we got an expected cbutton node
			if (inXMLNode == NULL)
				throw new CException (CString("CRange::Serialize, specified xml node is not a \"") + 
							      XML_RANGE_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the current node
			ioXMLElementNode = inXMLNode;

			// retreive and set the range attributes
			CString inUpdatePolicy (::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_UPDATE_POLICY).GetValue());
			if (inUpdatePolicy == CString("discontinuous")) 
				SetUpdatePolicy (GTK_UPDATE_DISCONTINUOUS);
			else if (inUpdatePolicy == CString("delayed"))
				SetUpdatePolicy (GTK_UPDATE_DELAYED);
			else
				SetUpdatePolicy (GTK_UPDATE_CONTINUOUS);
			SetRange        (::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_MIN).GetValue().ToDouble(), 
				         ::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_MAX).GetValue().ToDouble());
			SetIncrements   (::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_STEP).GetValue().ToDouble(),
				         ::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_PAGE).GetValue().ToDouble());
			SetInverted	(::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_INVERTED).GetValue().ToBool());
			SetValue        (::xml_node_get_attribute (inXMLNode, XML_RANGE_ATTR_VALUE).GetValue().ToDouble());
		}
		break;
	}
}
