//-----------------------------------------------------------------------------------------------------------------------------------------
// clabel.c
//-----------------------------------------------------------------------------------------------------------------------------------------

#include "clabel.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CLabel);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CLabel::CLabel    (CContainer *inOwner, const CString &inCaption, const CWidgetListener *inListener)
       :CWidget   (inOwner, inListener),
	m_Caption (inCaption==CString()?("id"+CString(GetId())):inCaption)
{
	// widget instanciation requested
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CLabel::~CLabel ()
{
	// widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CLabel::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_label_new (m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CLabel::PerformWidgetInitialize ()
{
	// show it
	Show();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CLabel::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CLabel::SetCaption (const CString &inCaption)
{
	// attribute local copy
	m_Caption = inCaption;

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_label_set_text (GTK_LABEL(inGtkWidget), m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CLabel::GetCaption () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check and return
	return inGtkWidget == NULL ? CString() : ::gtk_label_get_text (GTK_LABEL(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CLabel::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CWidget::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_LABEL_ELEMENT);
		
			// add the xml attributes
			newXMLElement -> AddAttribute (XML_LABEL_ATTR_CAPTION, GetCaption());
			
			// change the io xml element node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_LABEL_ELEMENT);

			// check we got an expected clabel node
			if (inXMLNode == NULL)
				throw new CException (CString("CLabel::Serialize, specified xml node is not a \"") + 
							      XML_LABEL_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cframe node
			ioXMLElementNode = inXMLNode;

			// set the label attributes
			SetCaption (::xml_node_get_attribute (inXMLNode, XML_LABEL_ATTR_CAPTION).GetValue());
		}
		break;
	}
}


