/*******************************************************************************************************************************************
 cimage.c
*******************************************************************************************************************************************/

#include "cimage.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CImage);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CImage::CImage        (CContainer *inOwner, const CString &inFileName, const CWidgetListener *inListener)
       :CWidget       (inOwner, inListener),
	m_FileName    (inFileName),
	m_GtkIconSize (GTK_ICON_SIZE_INVALID),
	m_StockId     ()
{
	// perform the gtk instanciation
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CImage::CImage        (CContainer *inOwner, const char *inStockId, const GtkIconSize inGtkIconSize, const CWidgetListener *inListener)
       :CWidget       (inOwner, inListener),
	m_FileName    (),
	m_GtkIconSize (inGtkIconSize),
	m_StockId     (inStockId)
{
	// perform the gtk instanciation
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CImage::~CImage ()
{
	// request a proper deletion
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CImage::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_image_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CImage::PerformWidgetInitialize ()
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// load from given file if any
	if (m_FileName != CString()) 
		::gtk_image_set_from_file (GTK_IMAGE(inGtkWidget), m_FileName.Get());
	
	// check the stock member parameter
	if (m_StockId != CString())
		::gtk_image_set_from_stock (GTK_IMAGE(inGtkWidget), m_StockId.Get(), m_GtkIconSize);

	// show it
	Show();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// image from file
//-----------------------------------------------------------------------------------------------------------------------------------------
void CImage::SetFromFile (const CString &inFileName)
{
	// attribute local copy
	m_FileName = inFileName;

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// load from given file if any
	::gtk_image_set_from_file (GTK_IMAGE(inGtkWidget), m_FileName.Get());

	// forget the local stock properties if any
	m_GtkIconSize = GTK_ICON_SIZE_INVALID;
	m_StockId     = CString();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// associated file name
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CImage::GetFileName () const
{
	return m_FileName;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set from stock
//-----------------------------------------------------------------------------------------------------------------------------------------
void CImage::SetFromStock (const char *inStockId, const GtkIconSize inGtkIconSize)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// local reference of the stock properties
	m_GtkIconSize = inGtkIconSize;
	m_StockId     = inStockId;

	// gtk affectation
	::gtk_image_set_from_stock (GTK_IMAGE(inGtkWidget), m_StockId.Get(), m_GtkIconSize);

	// forget the local file name if any
	m_FileName = CString();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get stock specifications
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CImage::GetFromStock (CString &outStockId, GtkIconSize &outGtkIconSize) const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return false;

	// get the image type, be paranoïd...
	GtkImageType inGtkImageType = ::gtk_image_get_storage_type (GTK_IMAGE(inGtkWidget));

	// check it
	if (inGtkImageType != GTK_IMAGE_EMPTY && inGtkImageType != GTK_IMAGE_STOCK) return false;

	// retrieve the attributes
	gchar *inStockId; ::gtk_image_get_stock (GTK_IMAGE(inGtkWidget), &inStockId, &outGtkIconSize); outStockId = inStockId;

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CImage::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization first
	CWidget::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_IMAGE_ELEMENT);

			// change the io node so that derived serialization continues under this definition
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// add the element attributes
			if (m_FileName != CString()) 
				newXMLElement -> AddAttribute (XML_IMAGE_ATTR_FILENAME, m_FileName);
			else if (m_StockId != CString()) 
			{
				newXMLElement -> AddAttribute (XML_IMAGE_ATTR_STOCK_ID, m_StockId);
				GtkIconSize inGtkIconSize; CString unUsed; if (GetFromStock (unUsed, inGtkIconSize)) switch (inGtkIconSize)
				{
					case GTK_ICON_SIZE_INVALID :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("invalid"));
						break;
  					case GTK_ICON_SIZE_MENU :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("menu"));
						break;
  					case GTK_ICON_SIZE_SMALL_TOOLBAR :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("small-toolbar"));
						break;
  					case GTK_ICON_SIZE_LARGE_TOOLBAR :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("large-toolbar"));
						break;
  					case GTK_ICON_SIZE_BUTTON :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("button"));
						break;
  					case GTK_ICON_SIZE_DND :
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("dnd"));
						break;
  					case GTK_ICON_SIZE_DIALOG:
						newXMLElement -> AddAttribute (XML_IMAGE_ATTR_ICON_SIZE, CString("dialog"));
						break;
				}
			}
		}
		break;
	
		// load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_IMAGE_ELEMENT);

			// check we got an expected cimage node
			if (inXMLNode == NULL)
				throw new CException (CString("CImage::Serialize, specified xml node is not a \"") + 
							      XML_IMAGE_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cimage node
			ioXMLElementNode = inXMLNode;

			// get the associated attributes if any
			m_FileName 	   = ::xml_node_get_attribute (inXMLNode, XML_IMAGE_ATTR_FILENAME). GetValue();
			m_StockId 	   = ::xml_node_get_attribute (inXMLNode, XML_IMAGE_ATTR_STOCK_ID). GetValue();
			CString inIconSize = ::xml_node_get_attribute (inXMLNode, XML_IMAGE_ATTR_ICON_SIZE).GetValue();
			
			// load the image from given file if any
			if (m_FileName != CString()) 
				SetFromFile (CString(m_FileName));
			else if (m_StockId != CString())
			{
				if (inIconSize == CString("menu"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_MENU);
				else if (inIconSize == CString("small-toolbar"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_SMALL_TOOLBAR);
				else if (inIconSize == CString("large-toolbar"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_LARGE_TOOLBAR);
				else if (inIconSize == CString("button"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_BUTTON);
				else if (inIconSize == CString("dnd"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_DND);
				else if (inIconSize == CString("dialog"))
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_DIALOG);
				else 
					SetFromStock (m_StockId.Get(), GTK_ICON_SIZE_INVALID);
			}
		}
		break;
	}
}

