/*******************************************************************************************************************************************
 cfilechooserdialog.c
*******************************************************************************************************************************************/

#include "cfilechooserdialog.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CFileChooserDialogListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileChooserDialogListener::CFileChooserDialogListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileChooserDialogListener::~CFileChooserDialogListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// current folder change notification
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::OnFolderChanged (GtkFileChooser *inGtkFileChooser, gpointer inData)
{
	// check pointer...
	if (inData == NULL) return;

	// retreive the gtkol instance
	CFileChooserDialog *inFileChooserDialog = reinterpret_cast <CFileChooserDialog *> (inData);

	// send the notification to the listener if any
	if (inFileChooserDialog -> GetListener() != NULL)
		static_cast <CFileChooserDialogListener *> (inFileChooserDialog -> GetListener()) -> 
			OnFolderChanged (inFileChooserDialog);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selection change notification
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::OnSelectionChanged (GtkFileChooser *inGtkFileChooser, gpointer inData)
{
	// check pointer...
	if (inData == NULL) return;

	// retreive the gtkol instance
	CFileChooserDialog *inFileChooserDialog = reinterpret_cast <CFileChooserDialog *> (inData);

	// send the notification to the listener if any
	if (inFileChooserDialog -> GetListener() != NULL)
		static_cast <CFileChooserDialogListener *> (inFileChooserDialog -> GetListener()) -> 
			OnSelectionChanged (inFileChooserDialog);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// preview change handling
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::OnUpdatePreview (GtkFileChooser *inGtkFileChooser, gpointer inData)
{
	// check pointer...
	if (inData == NULL) return;

	// retreive the gtkol instance
	CFileChooserDialog *inFileChooserDialog = reinterpret_cast <CFileChooserDialog *> (inData);

	// get the file name that will be used on the preview
	char *filename = ::gtk_file_chooser_get_preview_filename (inGtkFileChooser);

	// get it as a string
	CString inFileName (filename);

	// ... and free the given one
	::g_free (filename);

	// default to preview acceptation on regular files
        Bool ioDoPreview = CFile::Exists (inFileName) == FILE_REGULAR;

	// send the notification to the listener if any
	if (inFileChooserDialog -> GetListener() != NULL)
		static_cast <CFileChooserDialogListener *> (inFileChooserDialog -> GetListener()) -> 
			OnUpdatePreview (inFileChooserDialog, inFileName, ioDoPreview);

        // check we got a regular file the preview will be generated from...
	ioDoPreview = ioDoPreview && CFile::Exists (inFileName) == FILE_REGULAR;

	// if the preview is enabled
	if (ioDoPreview)
	{
		// set the active state of the preview
		::gtk_file_chooser_set_preview_widget_active (inGtkFileChooser, true);

		// get its thumbnail
		GdkPixbuf *inPixbuf = ::gdk_pixbuf_new_from_file_at_size (inFileName.Get(), 128, 128, NULL);

		// set it on the associated preview widget
                ::gtk_image_set_from_pixbuf (GTK_IMAGE(::gtk_file_chooser_get_preview_widget(inGtkFileChooser)), inPixbuf);

		// forget the reference
		if (inPixbuf != NULL) ::gdk_pixbuf_unref (inPixbuf);
	}
	// the preview is not enabled
	else
	{
		// set the inactive state of the preview
		::gtk_file_chooser_set_preview_widget_active (inGtkFileChooser, false);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CFileChooserDialog);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileChooserDialog::CFileChooserDialog  (CForm *inOwner, const CString &inCaption, const GtkFileChooserAction inFileChooserDialog,
				 	 const bool inModal, const CFileChooserDialogListener *inListener)
		   :CFileDialog	        (NULL, inCaption, inModal, inListener),
		    m_FileChooserDialog (inFileChooserDialog)
{
	// widget instanciation coherence requested
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileChooserDialog::~CFileChooserDialog ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CFileChooserDialog::PerformWidgetInstanciate ()
{
	// instanciate the gtk widget and return it
	return ::gtk_file_chooser_dialog_new (m_Caption.Get(), NULL, m_FileChooserDialog, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::PerformWidgetInitialize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// signals connexion
	::g_signal_connect (G_OBJECT(inGtkWidget), "current-folder-changed", G_CALLBACK(CFileChooserDialog::OnFolderChanged),	 this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "selection-changed",      G_CALLBACK(CFileChooserDialog::OnSelectionChanged), this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "update-preview", 	     G_CALLBACK(CFileChooserDialog::OnUpdatePreview),	 this);

	// defaults to do preview widget
	SetPreviewable (true);

	// generic initialization
	CFileDialog::PerformWidgetInitialize ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CFileChooserDialog::ListenerMustBe () const
{
	return __metaclass(CFileChooserDialogListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// multiple selection property affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::SetSelectionMultiple (const bool inMultiple)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER(inGtkWidget), inMultiple);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// multiple selection property reader
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CFileChooserDialog::GetSelectionMultiple () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// return the gtk property
	return inGtkWidget != NULL ? ::gtk_file_chooser_get_select_multiple (GTK_FILE_CHOOSER(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default selection
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::SetFileSelectionOn (const CString &inSelection)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();
	
	// pointer check
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_file_chooser_select_filename (GTK_FILE_CHOOSER(inGtkWidget), inSelection.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default selection reader
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CFileChooserDialog::GetFileSelectionOn () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// out selection
	CString outSelection;
	
	// pointer check
	if (inGtkWidget == NULL) return outSelection;

	// get the gtk property
	gchar *inSelection = ::gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(inGtkWidget));

	// keep a selection pointer copy
	outSelection = CString(inSelection);

	// free the pointer
	::g_free (inSelection);

	// ok
	return outSelection;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selected file names output
//-----------------------------------------------------------------------------------------------------------------------------------------
CStrings CFileChooserDialog::GetSelection () const
{
	// out file name selection
	CStrings outFileNames;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the pointer and set the property
	if (inGtkWidget == NULL) return outFileNames;

	// multiple selection mode
	if (GetSelectionMultiple ())
	{
		GSList *inFileNames  = ::gtk_file_chooser_get_filenames (GTK_FILE_CHOOSER(inGtkWidget));
		GSList *theFileNames = inFileNames;
		while (theFileNames != NULL)
		{
			outFileNames += CString(reinterpret_cast <char *> (theFileNames->data));
			::g_free (reinterpret_cast <char *> (theFileNames->data));
			theFileNames = theFileNames->next;
		}
		::g_slist_free (inFileNames);
	}
	// single selection mode
	else
	{
		gchar *inFileName = ::gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(inGtkWidget));
		outFileNames += CString(inFileName);
		if (inFileName != NULL) ::g_free (inFileName);
	}

	// ok
	return outFileNames;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// preview state affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::SetPreviewable (const bool inPreviewable)
{
	// our gtk widget
        GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
        if (inGtkWidget == NULL) return;

	// request analyse
        if (inPreviewable)
        {
                GtkWidget *outImagePreview = ::gtk_image_new ();
                ::gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER(inGtkWidget), outImagePreview);
        }
        else
        {
                ::gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER(inGtkWidget), NULL);
        }
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// previewable state reader
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CFileChooserDialog::GetPreviewable () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// get the associated property
	return inGtkWidget != NULL ? ::gtk_file_chooser_get_preview_widget (GTK_FILE_CHOOSER(inGtkWidget)) != NULL : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set the file chooser dialog type
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::SetFileChooserDialog (const GtkFileChooserAction inFileChooserDialog)
{
	// local copy
	m_FileChooserDialog = inFileChooserDialog;

	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// gtk property affectation
	::gtk_file_chooser_set_action (GTK_FILE_CHOOSER(inGtkWidget), m_FileChooserDialog);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get the file chooser dialog type
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkFileChooserAction CFileChooserDialog::GetFileChooserDialog () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok
	return inGtkWidget != NULL ? ::gtk_file_chooser_get_action (GTK_FILE_CHOOSER(inGtkWidget)) : m_FileChooserDialog;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileChooserDialog::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CFileDialog::Serialize (ioXMLElementNode, inMode);

	// xml serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_FILECHOOSERDIALOG_ELEMENT);

			// dump the previewable attribute
			newXMLElement -> AddAttribute (XML_FILECHOOSERDIALOG_ATTR_PREVIEWABLE, CString(GetPreviewable()?"true":"false"));
			GtkFileChooserAction inAction (GetFileChooserDialog());
			switch (inAction)
			{
				case GTK_FILE_CHOOSER_ACTION_OPEN :
					newXMLElement -> AddAttribute (XML_FILECHOOSERDIALOG_ATTR_TYPE, CString("open"));
				break;
				case GTK_FILE_CHOOSER_ACTION_SAVE :
					newXMLElement -> AddAttribute (XML_FILECHOOSERDIALOG_ATTR_TYPE, CString("save"));
				break;
				case GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER :
					newXMLElement -> AddAttribute (XML_FILECHOOSERDIALOG_ATTR_TYPE, CString("select-folder"));
				break;
				case GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER :
					newXMLElement -> AddAttribute (XML_FILECHOOSERDIALOG_ATTR_TYPE, CString("create-folder"));
				break;
			}
		
			// modify the io xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the file chooser dialog xml node
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_FILECHOOSERDIALOG_ELEMENT);

			// check we got it !
			if (inXMLNode == NULL)
				throw new CException (CString("CFileChooserDialog::Serialize, specified xml node is not a \"") + 
							      XML_FILECHOOSERDIALOG_ELEMENT + CString("\" element one."), 
						      __exception(XMLPARSE));

			// modify the io xml node pointer
			ioXMLElementNode = inXMLNode;

			// set the previewable property from given xml attribute
			SetPreviewable (::xml_node_get_attribute (inXMLNode, XML_FILECHOOSERDIALOG_ATTR_PREVIEWABLE).GetValue().ToBool());

			// get the file chooser dialog type
			CString inAction = ::xml_node_get_attribute (inXMLNode, XML_FILECHOOSERDIALOG_ATTR_TYPE).GetValue();
			if (inAction == CString("save"))
				SetFileChooserDialog (GTK_FILE_CHOOSER_ACTION_SAVE);
			else if (inAction == CString("select-folder"))
				SetFileChooserDialog (GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
			else if (inAction == CString("create-folder"))
				SetFileChooserDialog (GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER);
			else
				SetFileChooserDialog (GTK_FILE_CHOOSER_ACTION_OPEN);
		}
		break;
	}
}



